//! APIs for executing HTTP requests.
//!

use super::error::{Error, HttpError};
use super::FetchResponse;
use bytes::Bytes;
use http::header::{HeaderMap, HeaderName, HeaderValue};
use http::{Method, Request, Uri, Version};
use num_traits::FromPrimitive;

#[link(wasm_import_module = "http_fetch")]
extern "C" {
    fn host_fetch_send(
        version_addr: *const u8,
        version_len: u32,
        headers_addr: *const u8,
        headers_len: u32,
        body_addr: *const u8,
        body_len: u32,
        url_ptr: *const u8,
        url_len: u32,
        method_ptr: *const u8,
        method_len: u32,
        err_code_ptr: *mut i32,
    ) -> i32;
}

/// An HTTP fetch request, including body, headers, method, and URL.
///
/// # Creation, conversion, and sending
///
/// New requests with an empty body can be created programmatically using [`HttpFetch::new()`]. In
/// addition, there are convenience constructors such as [`HttpFetch::get()`], [`HttpFetch::post()`],
/// [`HttpFetch::put()`], [`HttpFetch::delete()`] which automatically select the appropriate method.
/// Send requests using [`HttpFetch::send()`].
///
/// For interoperability with other Rust libraries, [`HttpFetch`] can be initiated using
/// [`http`], which is crate's [`http::Request`] type along with the [`HttpFetch::send_using_standard_http_lib()`] function
/// passing [`Request<Option<Bytes>>`] as parameters.
///
/// # Builder-style Methods
///
/// [`Request`] can be used as a
/// [builder](https://doc.rust-lang.org/1.0.0/style/ownership/builders.html), allowing requests to
/// be constructed and used through method chaining. Methods with the `with_` and `set_` name prefix, such as
/// [`set_header()`][`Self::with_raw_body()`], return `Self` to allow chaining. The builder style is
/// typically most useful when constructing and using a request in a single expression. 
/// For example:
///
/// ```no_run
/// use edjx::{HttpFetch, Request, HttpError};
/// # fn f() -> Result<(), HttpError> {
/// HttpFetch::get("https://example.com")
///     .set_header("my-header", "hello!")
///     .with_raw_body("empty body")
///     .send()?;
/// # Ok(()) }
/// ```
#[derive(Debug)]
pub struct HttpFetch {
    parts: Parts,
    body: Option<Vec<u8>>,
}

/// Component parts of `HttpFetch`
///
/// The HTTP fetch request head consists of a method, URI, version, and a set of
/// header fields.
#[derive(Debug)]
#[doc(hidden)]
struct Parts {
    /// The version of the request.
    version: Version,
    /// The headers for the request.
    headers: HeaderMap,
    /// The URI of the request.
    uri: Uri,
    /// The method for the request.
    method: Method,
}

#[doc(hidden)]
impl Parts {
    /// Creates a default instance of `Parts` with the version as [`HTTP/1.1`].
    fn new(method_p: Method, uri_p: Uri) -> Parts {
        Parts {
            version: Version::HTTP_11,
            headers: HeaderMap::new(),
            uri: uri_p,
            method: method_p,
        }
    }
}

impl HttpFetch {
    /// Creates a request with the given method and URI, no headers, and an empty body.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use edjx::{HttpFetch, Request, HttpError, Method};
    ///
    /// let fetch_uri = Uri::from_str("https://httpbin.org/get").unwrap();
    /// let request = HttpFetch::new(fetch_uri, Method::GET);
    /// ```
    pub fn new(uri: Uri, m: Method) -> HttpFetch {
        HttpFetch {
            body: None,
            parts: Parts::new(m, uri),
        }
    }

    /// Creates a request with the GET method and URI, no headers, and an empty body.
    pub fn get(uri: Uri) -> HttpFetch {
        HttpFetch {
            body: None,
            parts: Parts::new(Method::GET, uri),
        }
    }

    /// Creates a request with the POST method and URI, no headers, and an empty body.
    pub fn post(uri: Uri) -> HttpFetch {
        HttpFetch {
            body: None,
            parts: Parts::new(Method::POST, uri),
        }
    }

    /// Creates a request with the PUT method and URI, no headers, and an empty body.
    pub fn put(uri: Uri) -> HttpFetch {
        HttpFetch {
            body: None,
            parts: Parts::new(Method::PUT, uri),
        }
    }

    /// Creates a request with the DELETE method and URI, no headers, and an empty body.
    pub fn delete(uri: Uri) -> HttpFetch {
        HttpFetch {
            body: None,
            parts: Parts::new(Method::DELETE, uri),
        }
    }

    /// Sets the HTTP version of the request.
    pub fn set_version(mut self, version: Version) -> Self {
        self.parts.version = version;
        return self;
    }

    /// Sets the HTTP version of the request.
    pub fn get_version(&self) -> Version {
        return self.parts.version;
    }

    /// Sets the given [`&str`] value as the body of the request.
    ///
    /// Any body that was previously set on the request is discarded.
    pub fn with_raw_body(mut self, text: &str) -> Self {
        self.body = Some(Vec::from(text));
        return self;
    }

    /// Sets the given [`Vec<u8>`] value as the body of the request.
    ///
    /// Any body that was previously set on the request is discarded.
    pub fn with_binary_body(mut self, bytes: Vec<u8>) -> Self {
        self.body = Some(bytes);
        return self;
    }

    /// Sets a request header to the given value, discarding any previous values for the given
    /// header name.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use edjx::HttpFetch;
    ///
    /// let fetch_uri = Uri::from_str("https://httpbin.org/get").unwrap();
    /// let mut resp = HttpFetch::get(fetch_uri).set_header("hello".parse().unwrap(), "world!".parse().unwrap());
    ///
    /// let header_map = resp.get_headers();
    /// assert!(header_map.contains_key("hello"));
    /// ```
    pub fn set_header(mut self, header_name: HeaderName, value: HeaderValue) -> Self {
        self.parts.headers.append(header_name, value);
        return self;
    }

    /// Returns the request's header map as [`http::header::HeaderMap`].
    pub fn get_headers(&self) -> &HeaderMap {
        return &self.parts.headers;
    }

    /// Returns the request's header map as a mutable reference to [`http::header::HeaderMap`].
    pub fn headers_mut(&mut self) -> &mut HeaderMap {
        return &mut self.parts.headers;
    }

    /// Sends the request to the server, and returns after the response is
    /// received or an error occurs.
    ///
    /// # Error Response
    ///
    /// This method returns an error response of kind [`HttpError`] if any of the configured limits
    /// are exceeded.
    ///
    /// # Example
    ///
    /// Sending the client request:
    ///
    /// ```no_run
    /// use edjx::HttpFetch;
    ///
    /// HttpFetch::get("https://example.com").send();
    /// ```
    pub fn send(&self) -> Result<FetchResponse, HttpError> {
        let headers_vec =
            serde_json::to_vec(&super::utils::convert_to_hashmap(&self.parts.headers)).unwrap();
        let headers_slice = headers_vec.as_slice();
        let version: String = super::utils::to_version_string(&self.parts.version);
        let body_len: u32;
        let body_slice;
        let empty_body_vec = Vec::new();
        let method: String = self.parts.method.as_str().to_string();
        let uri: String = self.parts.uri.to_string();
        if self.body.is_none() {
            body_len = 0;
            body_slice = empty_body_vec.as_slice();
        } else {
            body_slice = self.body.as_ref().unwrap().as_slice();
            body_len = body_slice.len() as u32;
        }

        let mut error_code: i32 = 0;

        let response_value = unsafe {
            host_fetch_send(
                version.as_ptr(),
                version.len() as u32,
                headers_slice.as_ptr(),
                headers_slice.len() as u32,
                body_slice.as_ptr(),
                body_len,
                uri.as_ptr(),
                uri.len() as u32,
                method.as_ptr(),
                method.len() as u32,
                &mut error_code,
            )
        };

        if response_value < 0 {
            return Err(HttpError::from(Error::from_i32(error_code).unwrap()));
        }

        Ok(FetchResponse::from_server()?)
    }

    /// Sends the request to the server, using standard HTTP request library [`http::Request`] and
    /// returns after a response is received or an error occurs.
    ///
    /// # Error Response
    ///
    /// This method returns an error response of kind [`HttpError`] if any of the
    /// configured limits are exceeded.
    ///
    /// # Example
    ///
    /// Sending a request using [`http::Request`]:
    ///
    /// ```no_run
    /// use edjx::HttpFetch;
    /// use http::Request;
    /// use bytes::Bytes;
    ///
    /// let req_builder = http::request::Builder::new()
    ///       .method(http::Method::POST)
    ///       .uri(&query_url)
    ///       .header("header1", "header1_value");
    ///  
    /// let bytes = Bytes::from("Body bytes");
    /// let req = req_builder.body(Some(bytes)).unwrap();
    /// //let req = req_builder.body(None).unwrap(); //for request without body.
    ///
    /// let fetch_response = HttpFetch::send_using_standard_http_lib(req);
    ///
    /// ```
    pub fn send_using_standard_http_lib(
        req: Request<Option<Bytes>>,
    ) -> Result<FetchResponse, HttpError> {
        let headers_vec =
            serde_json::to_vec(&super::utils::convert_to_hashmap(&req.headers())).unwrap();
        let headers_slice = headers_vec.as_slice();
        let version: String = super::utils::to_version_string(&req.version());
        let method: String = req.method().as_str().to_string();
        let uri: String = req.uri().to_string();
        let body = match req.body() {
            None => Default::default(),
            Some(body) => body.as_ref(),
        };

        let mut error_code: i32 = 0;
        let response_value = unsafe {
            host_fetch_send(
                version.as_ptr(),
                version.len() as u32,
                headers_slice.as_ptr(),
                headers_slice.len() as u32,
                body.as_ptr(),
                body.len() as u32,
                uri.as_ptr(),
                uri.len() as u32,
                method.as_ptr(),
                method.len() as u32,
                &mut error_code,
            )
        };

        if response_value < 0 {
            return Err(HttpError::from(Error::from_i32(error_code).unwrap()));
        }

        Ok(FetchResponse::from_server()?)
    }
}
