//! Logging for WebAsssembly application
//!
//! This module is mostly re-export of the [`log`](mod@log) crate. The most
//! important exception is that logging is expected to be controlled from host.
//! So logger and max-level changes are allowed via this API.
//!
#![cfg_attr(feature="host", allow(dead_code))]

pub use log::{debug, error, info, log, log_enabled, trace, warn};
pub use log::{Record, RecordBuilder, Metadata, MetadataBuilder};
pub use log::{Level, LevelFilter, STATIC_MAX_LEVEL};
pub use log::{logger, max_level};

#[allow(dead_code)]
#[allow(unused_parens)]
// wit_bindgen_rust::import!("../wit/edgedb_log_v1.wit");
mod edgedb_log_v1 {
  #[repr(u8)]
  #[derive(Clone, Copy, PartialEq, Eq)]
  pub enum Level{
    Error,
    Warn,
    Info,
    Debug,
    Trace,
  }
  impl std::fmt::Debug for Level {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        Level::Error => {
          f.debug_tuple("Level::Error").finish()
        }
        Level::Warn => {
          f.debug_tuple("Level::Warn").finish()
        }
        Level::Info => {
          f.debug_tuple("Level::Info").finish()
        }
        Level::Debug => {
          f.debug_tuple("Level::Debug").finish()
        }
        Level::Trace => {
          f.debug_tuple("Level::Trace").finish()
        }
      }
    }
  }
  #[derive(Clone)]
  pub struct LogRecord<'a,> {
    pub level: Level,
    pub target: &'a  str,
    pub module_path: Option<&'a  str>,
    pub file: Option<&'a  str>,
    pub line: Option<u32>,
    pub message: &'a  str,
  }
  impl<'a,> std::fmt::Debug for LogRecord<'a,> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      f.debug_struct("LogRecord").field("level", &self.level).field("target", &self.target).field("module-path", &self.module_path).field("file", &self.file).field("line", &self.line).field("message", &self.message).finish()}
  }
  pub fn log(item: LogRecord<'_,>,) -> (){
    unsafe {
      let LogRecord{ level:level0, target:target0, module_path:module_path0, file:file0, line:line0, message:message0, } = item;
      let vec1 = target0;
      let ptr1 = vec1.as_ptr() as i32;
      let len1 = vec1.len() as i32;
      let (result3_0,result3_1,result3_2,) = match module_path0{
        None => { (0i32, 0i32, 0i32)}
        Some(e) => { {
          let vec2 = e;
          let ptr2 = vec2.as_ptr() as i32;
          let len2 = vec2.len() as i32;
          
          (1i32, ptr2, len2)
        }}
      };
      let (result5_0,result5_1,result5_2,) = match file0{
        None => { (0i32, 0i32, 0i32)}
        Some(e) => { {
          let vec4 = e;
          let ptr4 = vec4.as_ptr() as i32;
          let len4 = vec4.len() as i32;
          
          (1i32, ptr4, len4)
        }}
      };
      let (result6_0,result6_1,) = match line0{
        None => { (0i32, 0i32)}
        Some(e) => { (1i32, crate::bindgen::as_i32(e))}
      };
      let vec7 = message0;
      let ptr7 = vec7.as_ptr() as i32;
      let len7 = vec7.len() as i32;
      #[link(wasm_import_module = "edgedb_log_v1")]
      extern "C" {
        #[cfg_attr(target_arch = "wasm32", link_name = "log")]
        #[cfg_attr(not(target_arch = "wasm32"), link_name = "edgedb_log_v1_log")]
        fn wit_import(_: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, _: i32, );
      }
      wit_import(level0 as i32, ptr1, len1, result3_0, result3_1, result3_2, result5_0, result5_1, result5_2, result6_0, result6_1, ptr7, len7);
      ()
    }
  }
  pub fn max_level() -> Option<Level>{
    unsafe {
      let ptr0 = RET_AREA.as_mut_ptr() as i32;
      #[link(wasm_import_module = "edgedb_log_v1")]
      extern "C" {
        #[cfg_attr(target_arch = "wasm32", link_name = "max-level")]
        #[cfg_attr(not(target_arch = "wasm32"), link_name = "edgedb_log_v1_max-level")]
        fn wit_import(_: i32, );
      }
      wit_import(ptr0);
      match *((ptr0 + 0) as *const i32) {
        0 => None,
        1 => Some(match *((ptr0 + 8) as *const i32) {
          0 => Level::Error,
          1 => Level::Warn,
          2 => Level::Info,
          3 => Level::Debug,
          4 => Level::Trace,
          _ => panic!("invalid enum discriminant"),
        }),
        _ => panic!("invalid enum discriminant"),
      }
    }
  }
  static mut RET_AREA: [i64; 2] = [0; 2];
}

use edgedb_log_v1 as v1;

static mut LOGGER: HostLogger = HostLogger {
    max_level: log::STATIC_MAX_LEVEL,
};

struct HostLogger {
    max_level: log::LevelFilter,
}

impl From<log::Level> for v1::Level {
    fn from(value: log::Level) -> v1::Level {
        use v1::Level as T;
        use log::Level as S;

        match value {
            S::Error => T::Error,
            S::Warn => T::Warn,
            S::Debug => T::Debug,
            S::Info => T::Info,
            S::Trace => T::Trace,
        }
    }
}

fn convert_filter(value: Option<v1::Level>) -> log::LevelFilter {
    use log::LevelFilter as T;
    use v1::Level as S;

    match value {
        None => T::Off,
        Some(S::Error) => T::Error,
        Some(S::Warn) => T::Warn,
        Some(S::Debug) => T::Debug,
        Some(S::Info) => T::Info,
        Some(S::Trace) => T::Trace,
    }
}

#[cfg(not(feature="host"))]
pub(crate) fn init() {
    let level = convert_filter(v1::max_level());
    unsafe {
        // not sure if safe all all platforms
        LOGGER.max_level = level;

        log::set_logger(&LOGGER).expect("init_logging");
    }
    log::set_max_level(level);
}

impl log::Log for HostLogger {
    fn enabled(&self, metadata: &Metadata) -> bool {
        metadata.level() <= self.max_level
    }

    fn log(&self, record: &Record) {
        if self.enabled(record.metadata()) {
            v1::log(v1::LogRecord {
                target: record.target(),
                level: record.level().into(),
                message: &record.args().to_string(),
                line: record.line(),
                file: record.file(),
                module_path: record.module_path(),
            });
        }
    }

    fn flush(&self) {}
}
