//! Utility for simple and declarative style command line argument parsing.
//!
//! easy-args is meant to be used to set up simple command-line argumenst for
//! your applications and give them back in an easy to process way.
//!
//! # Getting Started
//!
//! ```
//! // First you must define an [`ArgSpec`] which will determine what the
//! // command-line arguments are for your program and will be used by the parser to
//! // do some simple checks.
//!
//! // You make an [`ArgSpec`] with the builder pattern.
//!
//! use easy_args::{arg_spec, ArgSpec};
//!
//! let spec = ArgSpec::build()
//!     .boolean("fullscreen")
//!     .uinteger_array(2, "size")
//!     .done()
//!     .unwrap();
//!
//! // There is an `arg_spec!` macro which provides a nicer syntax.
//! let spec = arg_spec! {
//!     fullscreen: bool,
//!     size: [u64; 2],
//! };
//!
//! // Second you call [`ArgSpecs`]'s [`parse()`] method to retrieve the command-line
//! // arguments in a processed form.
//!
//! let args = spec.parse().unwrap();
//! if args.boolean("fullscreen") == Some(&true) {
//!     // Put application into windowed mode
//! }
//! ```
//!
//! And that's it! The arguments have been parsed and processed and can be
//! accessed via [`Args`]'s getter methods.
//!
//! [`ArgSpec`] also has a [`parse()`] method so you don't have to make a
//! throwaway variable.
//!
//! ```
//! use easy_args::ArgSpec;
//!
//! let args = ArgSpec::build()
//!     .boolean("windowed")
//!     .string("mode")
//!     .parse()
//!     .unwrap();
//! ```

use std::collections::HashMap;
use std::fmt::{Debug, Formatter};

/// Specifies the valid arguments of the program and is used to parse
/// the command-line arguments into an [`Arg`].
pub struct ArgSpec {
  args: HashMap<String, ArgType>,
}

type ArgIter = std::iter::Peekable<std::iter::Skip<std::env::Args>>;

impl ArgSpec {
  /// Creates an [`ArgSpecBuilder`] that can be used to build the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::ArgSpec;
  ///
  /// let spec = ArgSpec::build()
  ///     .boolean("arg1")
  ///     .done()
  ///     .unwrap();
  /// ```
  pub fn build() -> ArgSpecBuilder {
    ArgSpecBuilder {
      args: HashMap::new(),
      err: None,
    }
  }

  /// Determines if an argument of a given name and [`ArgType`] exists
  /// within the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::{arg_spec, ArgType};
  ///
  /// let spec = arg_spec!(username: String);
  /// if spec.has_arg("username", ArgType::String) {
  ///     let args = spec.parse().unwrap();
  ///     if let Some(username) = args.string("username") {
  ///         // do something with username
  ///     }
  /// }
  /// ```
  pub fn has_arg(&self, name: impl Into<String>, ty: ArgType) -> bool {
    if let Some(_t) = self.args.get(&name.into()) {
      matches!(ty, _t)
    } else {
      false
    }
  }

  /// Parses the command-line arguments and Returns [`Ok(Args)`] if there
  /// were no parse errors and [`Err(Error)`] if otherwise.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let spec = arg_spec!(vsync: bool);
  /// match spec.parse() {
  ///     Ok(args) => {
  ///         // do stuff with the arguments
  ///     }
  ///     Err(err) => eprintln!("{:?}", err),
  /// }
  /// ```
  pub fn parse(&self) -> Result<Args> {
    let mut bools: HashMap<String, bool> = HashMap::new();
    let mut ints: HashMap<String, i64> = HashMap::new();
    let mut uints: HashMap<String, u64> = HashMap::new();
    let mut floats: HashMap<String, f64> = HashMap::new();
    let mut strs: HashMap<String, String> = HashMap::new();
    let mut bool_arrays: HashMap<String, Box<[bool]>> = HashMap::new();
    let mut int_arrays: HashMap<String, Box<[i64]>> = HashMap::new();
    let mut uint_arrays: HashMap<String, Box<[u64]>> = HashMap::new();
    let mut float_arrays: HashMap<String, Box<[f64]>> = HashMap::new();
    let mut str_arrays: HashMap<String, Box<[String]>> = HashMap::new();
    let mut free_args: Vec<String> = Vec::new();
    let mut args = std::env::args().skip(1).peekable();

    while let Some(arg) = args.next() {
      let mut chars = arg.chars().peekable();
      if chars.peek() == Some(&'-') {
        chars.next();
        if chars.peek() == Some(&'-') {
          chars.next();
        }
        let arg_name: String = chars.collect();
        let arg_type = *self
          .args
          .get(&arg_name)
          .ok_or(Error::UnknownArgument(arg_name.clone()))?;
        match arg_type {
          ArgType::Boolean => parse_bool(arg_name, &mut args, &mut bools)?,
          ArgType::Integer => parse_arg(arg_name, arg_type, &mut args, &mut ints)?,
          ArgType::UInteger => parse_arg(arg_name, arg_type, &mut args, &mut uints)?,
          ArgType::Float => parse_arg(arg_name, arg_type, &mut args, &mut floats)?,
          ArgType::String => parse_arg(arg_name, arg_type, &mut args, &mut strs)?,
          ArgType::BooleanArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut bool_arrays)?
          }
          ArgType::IntegerArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut int_arrays)?
          }
          ArgType::UIntegerArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut uint_arrays)?
          }
          ArgType::FloatArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut float_arrays)?
          }
          ArgType::StringArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut str_arrays)?
          }
        }
      } else {
        free_args.push(chars.collect());
      }
    }

    Ok(Args::new(
      bools,
      ints,
      uints,
      floats,
      strs,
      bool_arrays,
      int_arrays,
      uint_arrays,
      float_arrays,
      str_arrays,
      free_args,
    ))
  }
}

trait HashMapExt<V> {
  fn insert_arg(&mut self, name: String, v: V) -> Result<()>;
}

impl<V> HashMapExt<V> for HashMap<String, V> {
  fn insert_arg(&mut self, name: String, v: V) -> Result<()> {
    if self.contains_key(&name) {
      Err(Error::RepeatedArgument(name))
    } else {
      self.insert(name, v);
      Ok(())
    }
  }
}

trait StringExt {
  fn is_valid_arg_name(&self) -> bool;
}

impl StringExt for String {
  fn is_valid_arg_name(&self) -> bool {
    !(self.starts_with('-')
      || self.contains(|c: char| c.is_whitespace())
      || self.contains(|c: char| c.is_control()))
  }
}

fn parse_arg<T: std::str::FromStr>(
  arg_name: String,
  arg_type: ArgType,
  args: &mut ArgIter,
  dict: &mut HashMap<String, T>,
) -> Result<()> {
  let arg_str = args
    .next()
    .ok_or(Error::MissingParameter(arg_name.clone(), arg_type))?;
  dict.insert_arg(
    arg_name.clone(),
    arg_str.parse::<T>().or(Err(Error::InvalidParameter(
      arg_name.clone(),
      arg_type,
      arg_str,
    )))?,
  )?;
  Ok(())
}

fn parse_bool(
  arg_name: String,
  args: &mut ArgIter,
  bools: &mut HashMap<String, bool>,
) -> Result<()> {
  let value = if args.peek() == Some(&"false".to_string()) {
    args.next();
    false
  } else if args.peek() == Some(&"true".to_string()) {
    args.next();
    true
  } else {
    true
  };
  bools.insert_arg(arg_name, value)
}

fn parse_array_arg<T: std::str::FromStr>(
  arg_name: String,
  arg_type: ArgType,
  array_size: usize,
  args: &mut ArgIter,
  dict: &mut HashMap<String, Box<[T]>>,
) -> Result<()> {
  let mut params: Vec<T> = Vec::with_capacity(array_size);
  for _ in 0..array_size {
    let arg_str = args
      .next()
      .ok_or(Error::MissingParameter(arg_name.clone(), arg_type))?;
    params.push(arg_str.parse::<T>().or(Err(Error::InvalidParameter(
      arg_name.clone(),
      arg_type,
      arg_str,
    )))?);
  }
  dict.insert_arg(arg_name, params.into_boxed_slice())
}

impl std::fmt::Debug for ArgSpec {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    f.debug_map().entries(self.args.iter()).finish()
  }
}

/// Enumerates all data types that are handled by [`ArgSpec::parse()`].
#[derive(Copy, Clone, Debug)]
pub enum ArgType {
  Boolean,
  BooleanArray(usize),
  Integer,
  IntegerArray(usize),
  UInteger,
  UIntegerArray(usize),
  Float,
  FloatArray(usize),
  String,
  StringArray(usize),
}

/// Builder type for [`ArgSpec`].
pub struct ArgSpecBuilder {
  args: HashMap<String, ArgType>,
  err: Option<Error>,
}

impl ArgSpecBuilder {
  /// Completes building the [`ArgSpec`]. Returns [`Ok(ArgSpec)`] if no build
  /// errors and [`Err(Error)`] if otherwise.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::ArgSpec;
  ///
  /// let spec = ArgSpec::build()
  ///     .uinteger("chickens")
  ///     .done().
  ///     unwrap();
  /// ```
  pub fn done(self) -> Result<ArgSpec> {
    if let Some(err) = self.err {
      Err(err)
    } else {
      Ok(ArgSpec { args: self.args })
    }
  }

  /// Little Wrapper function that will complete building the [`ArgSpec`]
  /// and immediately call [`parse()`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec! {
  ///     fullscreen: bool,
  ///     vsync: bool,
  ///     username: String
  /// }
  /// .parse()
  /// .unwrap();
  /// ```
  pub fn parse(self) -> Result<Args> {
    self.done()?.parse()
  }

  /// Adds an argument the the [`ArgSpec`] with a given name and type.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .arg("flag", ArgType::Boolean)
  ///     .done()
  ///     .unwrap();
  /// ```
  pub fn arg(mut self, name: impl Into<String>, ty: ArgType) -> ArgSpecBuilder {
    let name_str = name.into();
    if self.err.is_none() && !self.args.contains_key(&name_str) {
      if name_str.is_valid_arg_name() {
        self.args.insert(name_str, ty);
      } else {
        self.err = Some(Error::InvalidArgumentName(name_str));
      }
    } else {
      self.err = Some(Error::RedeclaredArgument(name_str));
    }
    self
  }

  /// Adds a boolean argument to the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  ///  let spec = ArgSpec::build()
  ///     .boolean("vsync")
  ///     .done()
  ///     .unwrap();
  ///  assert_eq!(spec.has_arg("vsync", ArgType::Boolean), true);
  /// ```
  pub fn boolean(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::Boolean)
  }

  /// Adds an i64 argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .integer("num-bananas")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("num-bananas", ArgType::Integer), true);
  /// ```
  pub fn integer(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::Integer)
  }

  /// Adds a u64 argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .uinteger("screen-width")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("screen-width", ArgType::UInteger), true);
  /// ```
  pub fn uinteger(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::UInteger)
  }

  /// Adds an f64 argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .float("gravity")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("gravity", ArgType::Float), true);
  /// ```
  pub fn float(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::Float)
  }

  /// Adds a String argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .string("MOTD")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("MOTD", ArgType::String), true);
  /// ```
  pub fn string(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::String)
  }

  /// Adds a boolean array argument to the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  ///  let spec = ArgSpec::build()
  ///     .boolean_array(2, "bools")
  ///     .done()
  ///     .unwrap();
  ///  assert_eq!(spec.has_arg("bools", ArgType::BooleanArray(2)), true);
  /// ```
  pub fn boolean_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::BooleanArray(size))
  }

  /// Adds an i64 array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .integer_array(3, "position")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("position", ArgType::IntegerArray(3)), true);
  /// ```
  pub fn integer_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::IntegerArray(size))
  }

  /// Adds a u64 array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .uinteger_array(2, "size")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("size", ArgType::UIntegerArray(2)), true);
  /// ```
  pub fn uinteger_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::UIntegerArray(size))
  }

  /// Adds a f64 array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .float_array(3, "position")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("position", ArgType::FloatArray(2)), true);
  /// ```
  pub fn float_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::FloatArray(size))
  }

  /// Adds a String array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .string_array(2, "name")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("name", ArgType::StringArray(2)), true);
  /// ```
  pub fn string_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::StringArray(size))
  }
}

/// Macro that lets uers write [`ArgSpec`]s in a nicer way.
///
/// # Example
///
/// ```
/// use easy_args::{arg_spec, ArgType};
///
/// let spec = arg_spec!(vsync: bool, size: [u64; 2]);
/// assert_eq!(spec.has_arg("vsync", ArgType::Boolean), true);
/// assert_eq!(spec.has_arg("size", ArgType::UIntegerArray(2)), true);
/// ```
#[macro_export]
macro_rules! arg_spec {
  ($($arg_name:ident : $arg_type:tt),*$(,)?) => {{
    let b = $crate::ArgSpec::build();
    $(let b = arg_spec!(b, $arg_name, $arg_type);)*
    b.done().unwrap()
  }};

  ($builder:expr, $arg_name:ident, bool) => {
    $builder.boolean(stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, i64) => {
    $builder.integer(stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, u64) => {
    $builder.uinteger(stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, f64) => {
    $builder.float(stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, String) => {
    $builder.string(stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, [bool; $array_size:literal]) => {
    $builder.boolean_array($array_size, stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, [i64; $array_size:literal]) => {
    $builder.integer_array($array_size, stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, [u64; $array_size:literal]) => {
    $builder.uinteger_array($array_size, stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, [f64; $array_size:literal]) => {
    $builder.float_array($array_size, stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, [String; $array_size:literal]) => {
    $builder.string_array($array_size, stringify!($arg_name))
  };
  ($builder:expr, $arg_name:ident, $arg_type:tt) => {
    compile_error!(concat!("`", stringify!($arg_name), "` cannot be of type `", stringify!($arg_type), "` because `ArgSpec` doesn't support it"));
  };
}


/// Holds all the command-line arguments given by the user.
///
/// Each argument is contained within a [`HashMap`] that can be index by the
/// argument's name.
#[derive(Debug)]
pub struct Args {
  bools: HashMap<String, bool>,
  ints: HashMap<String, i64>,
  uints: HashMap<String, u64>,
  floats: HashMap<String, f64>,
  strs: HashMap<String, String>,
  bool_arrays: HashMap<String, Box<[bool]>>,
  int_arrays: HashMap<String, Box<[i64]>>,
  uint_arrays: HashMap<String, Box<[u64]>>,
  float_arrays: HashMap<String, Box<[f64]>>,
  str_arrays: HashMap<String, Box<[String]>>,
  free_args: Vec<String>,
}

impl Args {
  pub(crate) fn new(
    bools: HashMap<String, bool>,
    ints: HashMap<String, i64>,
    uints: HashMap<String, u64>,
    floats: HashMap<String, f64>,
    strs: HashMap<String, String>,
    bool_arrays: HashMap<String, Box<[bool]>>,
    int_arrays: HashMap<String, Box<[i64]>>,
    uint_arrays: HashMap<String, Box<[u64]>>,
    float_arrays: HashMap<String, Box<[f64]>>,
    str_arrays: HashMap<String, Box<[String]>>,
    free_args: Vec<String>,
  ) -> Self {
    Args {
      bools,
      ints,
      uints,
      floats,
      strs,
      bool_arrays,
      int_arrays,
      uint_arrays,
      float_arrays,
      str_arrays,
      free_args,
    }
  }

  /// Determines if an argument of a given name was set by the user in the
  /// command-line.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(size: [u64; 2]).parse().unwrap();
  /// if args.is_set("size") {
  ///     // resize the window height
  /// }
  /// ```
  pub fn is_set(&self, name: impl Into<String>) -> bool {
    let n = name.into();
    self
      .bools
      .keys()
      .chain(self.ints.keys())
      .chain(self.uints.keys())
      .chain(self.strs.keys())
      .chain(self.bool_arrays.keys())
      .chain(self.int_arrays.keys())
      .chain(self.uint_arrays.keys())
      .chain(self.str_arrays.keys())
      .find(|&k| *k == n)
      .is_some()
  }

  /// Returns a reference to the boolean value that corresponds with the
  /// given argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(fullscreen: bool).parse().unwrap();
  /// if args.boolean("fullscreen") == Some(&true) {
  ///     // go fullscreen
  /// }
  /// ```
  pub fn boolean(&self, name: impl Into<String>) -> Option<&bool> {
    self.bools.get(&name.into())
  }

  /// Returns a reference to the i64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(leaves: i64).parse().unwrap();
  /// let num_leaves_in_pile = *args.integer("leaves").unwrap_or(&0);
  /// ```
  pub fn integer(&self, name: impl Into<String>) -> Option<&i64> {
    self.ints.get(&name.into())
  }

  /// Returns a reference to the u64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(size: u64).parse().unwrap();
  /// let size = *args.uinteger("size").unwrap_or(&0);
  /// ```
  pub fn uinteger(&self, name: impl Into<String>) -> Option<&u64> {
    self.uints.get(&name.into())
  }

  /// Returns a reference to the f64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(gravity: f64).parse().unwrap();
  /// let size = *args.float("gravity").unwrap_or(&9.81);
  /// ```
  pub fn float(&self, name: impl Into<String>) -> Option<&f64> {
    self.floats.get(&name.into())
  }

  /// Returns a reference to the String value that corresponds with the
  /// given argument name.
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(username: String).parse().unwrap();
  /// let username = args.string("username").unwrap_or(&"Guest".to_string());
  /// ```
  pub fn string(&self, name: impl Into<String>) -> Option<&String> {
    self.strs.get(&name.into())
  }

  /// Returns a reference to the boolean slice that corresponds with the
  /// given argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(flags: [bool; 5]).parse().unwrap();
  /// if let Some(flags) = args.boolean_array("flags") {
  ///     // do something with flags
  /// }
  /// ```
  pub fn boolean_array(&self, name: impl Into<String>) -> Option<&[bool]> {
    Some(self.bool_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the i64 slice that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(position: [i64; 3]).parse().unwrap();
  /// if let Some([x, y, z]) = args.integer_array("position") {
  ///     // do something with the position
  /// }
  /// ```
  pub fn integer_array(&self, name: impl Into<String>) -> Option<&[i64]> {
    Some(self.int_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the u64 slice that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(size: [u64; 2]).parse().unwrap();
  /// if let Some([width, height]) = args.uinteger_array("size") {
  ///     // do something with screen size
  /// }
  /// ```
  pub fn uinteger_array(&self, name: impl Into<String>) -> Option<&[u64]> {
    Some(self.uint_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the f64 slice that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(position: [f64; 3]).parse().unwrap();
  /// if let Some([x, y, z]) = args.float_array("position") {
  ///     // do something with position
  /// }
  /// ```
  pub fn float_array(&self, name: impl Into<String>) -> Option<&[f64]> {
    Some(self.float_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the String slice that corresponds with the
  /// given argument name.
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!(login_details: [String; 2]).parse().unwrap();
  /// if let Some([username, password]) = args.string_array("login_details") {
  ///     // do something with username and password
  /// }
  /// ```
  pub fn string_array(&self, name: impl Into<String>) -> Option<&[String]> {
    Some(self.str_arrays.get(&name.into())?.as_ref())
  }

  /// A Vector of all strings passed as command-line arguments that weren't
  /// arguments of the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::arg_spec;
  ///
  /// let args = arg_spec!().parse().unwrap();
  /// for arg in args.free_args() {
  ///     println!("What the heck is this? '{}'.", arg);
  /// }
  /// ```
  pub fn free_args(&self) -> &Vec<String> {
    &self.free_args
  }
}


/// The error types for argument parsing.
///
/// If user inputs arguments incorrectly, [`ArgSpec::parse()`] will return [`Err(Error)`].
pub enum Error {
  UnknownArgument(String),
  MissingParameter(String, ArgType),
  InvalidParameter(String, ArgType, String),
  RedeclaredArgument(String),
  RepeatedArgument(String),
  InvalidArgumentName(String),
}

impl Debug for Error {
  fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
    match self {
      Error::UnknownArgument(name) => write!(f, "Unknown argument {}.", name),
      Error::MissingParameter(name, ty) => write!(
        f,
        "Missing {:?} parameter value for argument '{}'.",
        ty, name
      ),
      Error::InvalidParameter(name, ty, given) => write!(
        f,
        "Expected {:?} value for argument '{}' but found '{}'.",
        ty, name, given
      ),
      Error::RedeclaredArgument(name) => {
        write!(f, "Argument '{}' has already been declared.", name)
      }
      Error::RepeatedArgument(name) => {
        write!(f, "Argument '{}' has already been assigned a value.", name)
      }
      Error::InvalidArgumentName(name) => write!(
        f, 
        "'{}' is not a valid argument name. Argument names must not begin with a dash or contain any whitespace or control characters.", 
        name),
    }
  }
}

/// Convient [`Result`] type where [`E`] is [`Error`].
pub type Result<T> = std::result::Result<T, Error>;


#[cfg(test)]
mod tests {
  use super::*;

  #[test]
  fn spec_builder() -> Result<()> {
    let spec = arg_spec! {
      b: bool,
      n: i64,
      u: u64,
      name: String,
    };

    assert!(spec.has_arg("b", ArgType::Boolean));
    assert!(spec.has_arg("n", ArgType::Integer));
    assert!(spec.has_arg("u", ArgType::UInteger));
    assert!(spec.has_arg("name", ArgType::String));
    assert!(!spec.has_arg("none", ArgType::Boolean));

    Ok(())
  }

  #[test]
  fn parse() -> Result<()> {
    let args = arg_spec! {
      b: bool,
      n: i64,
      u: u64,
      name: String,
    }
    .parse()?;
    assert_eq!(args.free_args().len(), 0);
    Ok(())
  }
}
