# easy-args

A Rust library for simple and lightweight command-line argument parsing, with:

-   Easy to use declarative usage via [`ArgSpec`](https://docs.rs/easy-args/0.3.0/easy_args/struct.ArgSpec.html),
-   Zero dependencies with other crates,

It's also worth pointing out what `easy-args` *is not*:

-   Extensive. Only supports boolean flags, signed and unsigned 64-bit integers
    and strings and array versions of those types.
-   Fast. By no means is it super slow but it hasn't been made with maximum
    performance in mind.

Documentation:

-   [API reference (docs.rs)](https://docs.rs/easy-args)

## Usage

Add this to your `Cargo.toml`:

```toml
[dependencies]
easy-args = "0.3.0"
```

To get started using easy-args.

First you must define an `ArgSpec` which will determine what the command-line
arguments are for your program and will be used by the parser to do some
simple checks.
You make an `ArgSpec` with the builder pattern.
```
let spec = ArgSpec::build()
    .boolean("fullscreen")
    .uinteger_array(2, "size")
    .done()?;
```
There is an `arg_spec!` macro which provides a nicer syntax.
```
let spec = arg_spec! {
  fullscreen: bool,
  size: [u64; 2],
};
```
Second you call `ArgSpecs`'s `parse()` method to retrieve the command-line
arguments in a processed form.
```
let args = spec.parse()?;
if args.boolean("fullscreen") == Some(&true) {
    // Put application into fullscreen
}
```
And that's it! The arguments have been parsed and processed and can be
accessed via `Args`'s getter methods.
`ArgSpecBuilder` also has a `parse()` method so you don't have to make a
throwaway variable.
```
let args = ArgSpec::build()
    .boolean("fullscreen")
    .uinteger_array(2, "size")
    .parse()
    .unwrap();
```

## Versions

easy-args is an immature crate and as such may make breaking changes in future
versions.

Current easy-args versions are:

-   Version 0.1.0 was released in June 2021, with the very first implementation.
-   Version 0.1.1 was released a few hours afterwards to add documentation.
-   Version 0.2.0 was released in June 2021, introduced array types, renamed
    `unsigned_integer` methods to `uinteger` and added error checking for 
    to building `ArgSpec`'s.
-   Version 0.3.0 was released in June 2021. Reorganised crate into single module to
    simplify imports. Added features include:
    -   `arg_spec!` macro to provide a nicer syntax for `ArgSpec` building. 
    -   `f64` and `[f64]` arguments now supported.

# License

easy-args is distributed under the terms of the MIT license.