use crate::args::*;
use crate::errors::{Result, *};
use std::collections::HashMap;

/// Specifies the valid arguments of the program and is used to parse
/// the command-line arguments into an [`Arg`].
pub struct ArgSpec {
  args: HashMap<String, ArgType>,
}

type ArgIter = std::iter::Peekable<std::iter::Skip<std::env::Args>>;

impl ArgSpec {
  /// Creates an [`ArgSpecBuilder`] that can be used to build the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let spec = ArgSpec::build()
  ///     .boolean("arg1")
  ///     .done()
  ///     .unwrap();
  /// ```
  pub fn build() -> ArgSpecBuilder {
    ArgSpecBuilder {
      args: HashMap::new(),
      err: None,
    }
  }

  /// Determines if an argument of a given name and [`ArgType`] exists
  /// within the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .string("username")
  ///     .done()
  ///     .unwrap();
  /// if spec.has_arg("username", ArgType::String) {
  ///     let args = spec.parse().unwrap();
  ///     if let Some(username) = args.string("username") {
  ///         // do something with username
  ///     }
  /// }
  /// ```
  pub fn has_arg(&self, name: impl Into<String>, ty: ArgType) -> bool {
    if let Some(_t) = self.args.get(&name.into()) {
      matches!(ty, _t)
    } else {
      false
    }
  }

  /// Parses the command-line arguments and Returns [`Ok(Args)`] if there
  /// were no parse errors and [`Err(Error)`] if otherwise.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let spec = ArgSpec::build()
  ///     .boolean("vsync")
  ///     .done()
  ///     .unwrap();
  /// match spec.parse() {
  ///     Ok(args) => {
  ///         // do stuff with the arguments
  ///     }
  ///     Err(err) => eprintln!("{:?}", err),
  /// }
  /// ```
  pub fn parse(&self) -> Result<Args> {
    let mut bools: HashMap<String, bool> = HashMap::new();
    let mut ints: HashMap<String, i64> = HashMap::new();
    let mut uints: HashMap<String, u64> = HashMap::new();
    let mut strs: HashMap<String, String> = HashMap::new();
    let mut bool_arrays: HashMap<String, Box<[bool]>> = HashMap::new();
    let mut int_arrays: HashMap<String, Box<[i64]>> = HashMap::new();
    let mut uint_arrays: HashMap<String, Box<[u64]>> = HashMap::new();
    let mut str_arrays: HashMap<String, Box<[String]>> = HashMap::new();
    let mut free_args: Vec<String> = Vec::new();
    let mut args = std::env::args().skip(1).peekable();

    while let Some(arg) = args.next() {
      let mut chars = arg.chars().peekable();
      if chars.peek() == Some(&'-') {
        chars.next();
        if chars.peek() == Some(&'-') {
          chars.next();
        }
        let arg_name: String = chars.collect();
        let arg_type = *self
          .args
          .get(&arg_name)
          .ok_or(Error::UnknownArgument(arg_name.clone()))?;
        match arg_type {
          ArgType::Boolean => parse_bool(arg_name, &mut args, &mut bools)?,
          ArgType::Integer => parse_arg(arg_name, arg_type, &mut args, &mut ints)?,
          ArgType::UInteger => parse_arg(arg_name, arg_type, &mut args, &mut uints)?,
          ArgType::String => parse_arg(arg_name, arg_type, &mut args, &mut strs)?,
          ArgType::BooleanArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut bool_arrays)?
          }
          ArgType::IntegerArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut int_arrays)?
          }
          ArgType::UIntegerArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut uint_arrays)?
          }
          ArgType::StringArray(n) => {
            parse_array_arg(arg_name, arg_type, n, &mut args, &mut str_arrays)?
          }
        }
      } else {
        free_args.push(chars.collect());
      }
    }

    Ok(Args::new(
      bools,
      ints,
      uints,
      strs,
      bool_arrays,
      int_arrays,
      uint_arrays,
      str_arrays,
      free_args,
    ))
  }
}

trait HashMapExt<V> {
  fn insert_arg(&mut self, name: String, v: V) -> Result<()>;
}

impl<V> HashMapExt<V> for HashMap<String, V> {
  fn insert_arg(&mut self, name: String, v: V) -> Result<()> {
    if self.contains_key(&name) {
      Err(Error::RepeatedArgument(name))
    } else {
      self.insert(name, v);
      Ok(())
    }
  }
}

trait StringExt {
  fn is_valid_arg_name(&self) -> bool;
}

impl StringExt for String {
  fn is_valid_arg_name(&self) -> bool {
    !(self.starts_with('-')
      || self.contains(|c: char| c.is_whitespace())
      || self.contains(|c: char| c.is_control()))
  }
}

fn parse_arg<T: std::str::FromStr>(
  arg_name: String,
  arg_type: ArgType,
  args: &mut ArgIter,
  dict: &mut HashMap<String, T>,
) -> Result<()> {
  let arg_str = args
    .next()
    .ok_or(Error::MissingParameter(arg_name.clone(), arg_type))?;
  dict.insert_arg(
    arg_name.clone(),
    arg_str.parse::<T>().or(Err(Error::InvalidParameter(
      arg_name.clone(),
      arg_type,
      arg_str,
    )))?,
  )?;
  Ok(())
}

fn parse_bool(
  arg_name: String,
  args: &mut ArgIter,
  bools: &mut HashMap<String, bool>,
) -> Result<()> {
  let value = if args.peek() == Some(&"false".to_string()) {
    args.next();
    false
  } else if args.peek() == Some(&"true".to_string()) {
    args.next();
    true
  } else {
    true
  };
  bools.insert_arg(arg_name, value)
}

fn parse_array_arg<T: std::str::FromStr>(
  arg_name: String,
  arg_type: ArgType,
  array_size: usize,
  args: &mut ArgIter,
  dict: &mut HashMap<String, Box<[T]>>,
) -> Result<()> {
  let mut params: Vec<T> = Vec::with_capacity(array_size);
  for _ in 0..array_size {
    let arg_str = args
      .next()
      .ok_or(Error::MissingParameter(arg_name.clone(), arg_type))?;
    params.push(arg_str.parse::<T>().or(Err(Error::InvalidParameter(
      arg_name.clone(),
      arg_type,
      arg_str,
    )))?);
  }
  dict.insert_arg(arg_name, params.into_boxed_slice())
}

impl std::fmt::Debug for ArgSpec {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    f.debug_map().entries(self.args.iter()).finish()
  }
}

/// Enumerates all data types that are handled by [`ArgSpec::parse()`].
#[derive(Copy, Clone, Debug)]
pub enum ArgType {
  Boolean,
  BooleanArray(usize),
  Integer,
  IntegerArray(usize),
  UInteger,
  UIntegerArray(usize),
  String,
  StringArray(usize),
}

/// Builder type for [`ArgSpec`].
pub struct ArgSpecBuilder {
  args: HashMap<String, ArgType>,
  err: Option<Error>,
}

impl ArgSpecBuilder {
  /// Completes building the [`ArgSpec`]. Returns [`Ok(ArgSpec)`] if no build
  /// errors and [`Err(Error)`] if otherwise.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let spec = ArgSpec::build()
  ///     .uinteger("chickens")
  ///     .done().
  ///     unwrap();
  /// ```
  pub fn done(self) -> Result<ArgSpec> {
    if let Some(err) = self.err {
      Err(err)
    } else {
      Ok(ArgSpec { args: self.args })
    }
  }

  /// Little Wrapper function that will complete building the [`ArgSpec`]
  /// and immediately call [`parse()`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .boolean("fullscreen")
  ///     .boolean("vsync")
  ///     .string("username")
  ///     .parse()
  ///     .unwrap();
  /// ```
  pub fn parse(self) -> Result<Args> {
    self.done()?.parse()
  }

  /// Adds an argument the the [`ArgSpec`] with a given name and type.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .arg("flag", ArgType::Boolean)
  ///     .done()
  ///     .unwrap();
  /// ```
  pub fn arg(mut self, name: impl Into<String>, ty: ArgType) -> ArgSpecBuilder {
    let name_str = name.into();
    if self.err.is_none() && !self.args.contains_key(&name_str) {
      if name_str.is_valid_arg_name() {
        self.args.insert(name_str, ty);
      } else {
        self.err = Some(Error::InvalidArgumentName(name_str));
      }
    } else {
      self.err = Some(Error::RedeclaredArgument(name_str));
    }
    self
  }

  /// Adds a boolean argument to the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  ///  let spec = ArgSpec::build()
  ///     .boolean("vsync")
  ///     .done()
  ///     .unwrap();
  ///  assert_eq!(spec.has_arg("vsync", ArgType::Boolean), true);
  /// ```
  pub fn boolean(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::Boolean)
  }

  /// Adds an i64 argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .integer("num-bananas")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("num-bananas", ArgType::Integer), true);
  /// ```
  pub fn integer(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::Integer)
  }

  /// Adds a u64 argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .uinteger("screen-width")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("screen-width", ArgType::UInteger), true);
  /// ```
  pub fn uinteger(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::UInteger)
  }

  /// Adds a String argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .string("MOTD")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("MOTD", ArgType::String), true);
  /// ```
  pub fn string(self, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::String)
  }

  /// Adds a boolean array argument to the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  ///  let spec = ArgSpec::build()
  ///     .boolean_array(2, "bools")
  ///     .done()
  ///     .unwrap();
  ///  assert_eq!(spec.has_arg("bools", ArgType::BooleanArray(2)), true);
  /// ```
  pub fn boolean_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::BooleanArray(size))
  }

  /// Adds an i64 array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .integer_array(3, "position")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("position", ArgType::IntegerArray(3)), true);
  /// ```
  pub fn integer_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::IntegerArray(size))
  }

  /// Adds a u64 array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .uinteger_array(2, "size")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("size", ArgType::UIntegerArray(2)), true);
  /// ```
  pub fn uinteger_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::UIntegerArray(size))
  }

  /// Adds a String array argument to the [`ArgSpec`].
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::{ArgSpec, ArgType};
  ///
  /// let spec = ArgSpec::build()
  ///     .string_array(2, "name")
  ///     .done()
  ///     .unwrap();
  /// assert_eq!(spec.has_arg("name", ArgType::StringArray(2)), true);
  /// ```
  pub fn string_array(self, size: usize, name: impl Into<String>) -> ArgSpecBuilder {
    self.arg(name, ArgType::StringArray(size))
  }
}
