//! Utility for simple and declarative style command line argument parsing.
//!
//! easy-args is meant to be used to set up simple command-line argumenst for
//! your applications and give them back in an easy to process way.
//!
//! # Getting Started
//!
//! ```
//! // First you must define an [`ArgSpec`] which will determine what the
//! // command-line arguments are for your program and will be used by the parser to
//! // do some simple checks.
//!
//! // You make an [`ArgSpec`] with the builder pattern.
//!
//! use easy_args::spec::ArgSpec;
//!
//! let spec = ArgSpec::build()
//!     .boolean("windowed")
//!     .string("mode")
//!     .done()
//!     .unwrap();
//!
//! // Second you call [`ArgSpecs`]'s [`parse()`] method to retrieve the command-line
//! // arguments in a processed form.
//!
//! let args = spec.parse().unwrap();
//! if args.boolean("windowed") == Some(&true) {
//!     // Put application into windowed mode
//! }
//! ```
//!
//! And that's it! The arguments have been parsed and processed and can be
//! accessed via [`Args`]'s getter methods.
//!
//! [`ArgSpec`] also has a [`parse()`] method so you don't have to make a
//! throwaway variable.
//!
//! ```
//! use easy_args::spec::ArgSpec;
//!
//! let args = ArgSpec::build()
//!     .boolean("windowed")
//!     .string("mode")
//!     .parse()
//!     .unwrap();
//! ```

pub mod args;
pub mod errors;
pub mod spec;

#[cfg(test)]
mod tests {
  use super::errors::*;
  use super::spec::*;

  #[test]
  fn spec_builder() -> Result<()> {
    let spec = ArgSpec::build()
      .boolean("b")
      .integer("n")
      .uinteger("u")
      .string("name")
      .done()?;

    assert!(spec.has_arg("b", ArgType::Boolean));
    assert!(spec.has_arg("n", ArgType::Integer));
    assert!(spec.has_arg("u", ArgType::UInteger));
    assert!(spec.has_arg("name", ArgType::String));
    assert!(!spec.has_arg("none", ArgType::Boolean));

    Ok(())
  }

  #[test]
  fn parse() -> Result<()> {
    let args = ArgSpec::build()
      .boolean("b")
      .integer("n")
      .uinteger("u")
      .string("name")
      .parse()?;
    assert_eq!(args.free_args().len(), 0);
    Ok(())
  }
}
