use crate::spec::ArgType;
use std::fmt::{Debug, Formatter};

/// The error types for argument parsing.
///
/// If user inputs arguments incorrectly, [`ArgSpec::parse()`] will return [`Err(Error)`].
pub enum Error {
  UnknownArgument(String),
  MissingParameter(String, ArgType),
  InvalidParameter(String, ArgType, String),
  RedeclaredArgument(String),
  RepeatedArgument(String),
  InvalidArgumentName(String),
}

impl Debug for Error {
  fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
    match self {
      Error::UnknownArgument(name) => write!(f, "Unknown argument {}.", name),
      Error::MissingParameter(name, ty) => write!(
        f,
        "Missing {:?} parameter value for argument '{}'.",
        ty, name
      ),
      Error::InvalidParameter(name, ty, given) => write!(
        f,
        "Expected {:?} value for argument '{}' but found '{}'.",
        ty, name, given
      ),
      Error::RedeclaredArgument(name) => {
        write!(f, "Argument '{}' has already been declared.", name)
      }
      Error::RepeatedArgument(name) => {
        write!(f, "Argument '{}' has already been assigned a value.", name)
      }
      Error::InvalidArgumentName(name) => write!(
        f, 
        "'{}' is not a valid argument name. Argument names must not begin with a dash or contain any whitespace or control characters.", 
        name),
    }
  }
}

/// Convient [`Result`] type where [`E`] is [`Error`].
pub type Result<T> = std::result::Result<T, Error>;
