use std::collections::HashMap;

/// Holds all the command-line arguments given by the user.
///
/// Each argument is contained within a [`HashMap`] that can be index by the
/// argument's name.
#[derive(Debug)]
pub struct Args {
  bools: HashMap<String, bool>,
  ints: HashMap<String, i64>,
  uints: HashMap<String, u64>,
  strs: HashMap<String, String>,
  bool_arrays: HashMap<String, Box<[bool]>>,
  int_arrays: HashMap<String, Box<[i64]>>,
  uint_arrays: HashMap<String, Box<[u64]>>,
  str_arrays: HashMap<String, Box<[String]>>,
  free_args: Vec<String>,
}

impl Args {
  pub(crate) fn new(
    bools: HashMap<String, bool>,
    ints: HashMap<String, i64>,
    uints: HashMap<String, u64>,
    strs: HashMap<String, String>,
    bool_arrays: HashMap<String, Box<[bool]>>,
    int_arrays: HashMap<String, Box<[i64]>>,
    uint_arrays: HashMap<String, Box<[u64]>>,
    str_arrays: HashMap<String, Box<[String]>>,
    free_args: Vec<String>,
  ) -> Self {
    Args {
      bools,
      ints,
      uints,
      strs,
      bool_arrays,
      int_arrays,
      uint_arrays,
      str_arrays,
      free_args,
    }
  }

  /// Determines if an argument of a given name was set by the user in the
  /// command-line.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .uinteger("window-height")
  ///     .parse()
  ///     .unwrap();
  /// if args.is_set("window-height") {
  ///     // resize the window height
  /// }
  /// ```
  pub fn is_set(&self, name: impl Into<String>) -> bool {
    let n = name.into();
    self
      .bools
      .keys()
      .chain(self.ints.keys())
      .chain(self.uints.keys())
      .chain(self.strs.keys())
      .chain(self.bool_arrays.keys())
      .chain(self.int_arrays.keys())
      .chain(self.uint_arrays.keys())
      .chain(self.str_arrays.keys())
      .find(|&k| *k == n)
      .is_some()
  }

  /// Returns a reference to the boolean value that corresponds with the
  /// given argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .boolean("fullscreen")
  ///     .parse()
  ///     .unwrap();
  /// if args.boolean("fullscreen") == Some(&true) {
  ///     // go fullscreen
  /// }
  /// ```
  pub fn boolean(&self, name: impl Into<String>) -> Option<&bool> {
    self.bools.get(&name.into())
  }

  /// Returns a reference to the i64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .integer("leaves")
  ///     .parse()
  ///     .unwrap();
  /// let num_leaves_in_pile = *args.integer("leaves").unwrap_or(&0);
  /// ```
  pub fn integer(&self, name: impl Into<String>) -> Option<&i64> {
    self.ints.get(&name.into())
  }

  /// Returns a reference to the u64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .uinteger("size")
  ///     .parse()
  ///     .unwrap();
  /// let size = *args.uinteger("size").unwrap_or(&0);
  /// ```
  pub fn uinteger(&self, name: impl Into<String>) -> Option<&u64> {
    self.uints.get(&name.into())
  }

  /// Returns a reference to the String value that corresponds with the
  /// given argument name.
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .string("username")
  ///     .parse()
  ///     .unwrap();
  /// let username = args.string("username").unwrap_or(&"Guest".to_string());
  /// ```
  pub fn string(&self, name: impl Into<String>) -> Option<&String> {
    self.strs.get(&name.into())
  }

  /// Returns a reference to the boolean slice that corresponds with the
  /// given argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .boolean_array(5, "flags")
  ///     .parse()
  ///     .unwrap();
  /// if let Some(flags) = args.boolean_array("flags") {
  ///     // do something with flags
  /// }
  /// ```
  pub fn boolean_array(&self, name: impl Into<String>) -> Option<&[bool]> {
    Some(self.bool_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the i64 slice that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .integer_array(3, "position")
  ///     .parse()
  ///     .unwrap();
  /// if let Some([x, y, z]) = args.integer_array("position") {
  ///     // do something with the position
  /// }
  /// ```
  pub fn integer_array(&self, name: impl Into<String>) -> Option<&[i64]> {
    Some(self.int_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the u64 slice that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .uinteger_array(2, "size")
  ///     .parse()
  ///     .unwrap();
  /// if let Some([width, height]) = args.uinteger_array("size") {
  ///     // do something with screen size
  /// }
  /// ```
  pub fn uinteger_array(&self, name: impl Into<String>) -> Option<&[u64]> {
    Some(self.uint_arrays.get(&name.into())?.as_ref())
  }

  /// Returns a reference to the String slice that corresponds with the
  /// given argument name.
  ///
  /// # Exmaple
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build()
  ///     .string_array(2, "login-details")
  ///     .parse()
  ///     .unwrap();
  /// if let Some([username, password]) = args.string_array("login-details") {
  ///     // do something with username and password
  /// }
  /// ```
  pub fn string_array(&self, name: impl Into<String>) -> Option<&[String]> {
    Some(self.str_arrays.get(&name.into())?.as_ref())
  }

  /// A Vector of all strings passed as command-line arguments that weren't
  /// arguments of the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// use easy_args::spec::ArgSpec;
  ///
  /// let args = ArgSpec::build().parse().unwrap();
  /// for arg in args.free_args() {
  ///     println!("What the heck is this? '{}'.", arg);
  /// }
  /// ```
  pub fn free_args(&self) -> &Vec<String> {
    &self.free_args
  }
}
