//! Utility for simple and declarative style command line argument parsing.
//!
//! easy-args is meant to be used to set up simple command-line argumenst for
//! your applications and give them back in an easy to process way.
//!
//! # Getting Started
//!
//! First you must define an [`ArgSpec`] which will determine what the command-line
//! arguments are for your program and will be used by the parser to do some
//! simple checks.
//!
//! You make an [`ArgSpec`] with the builder pattern.
//!
//! ```
//! let arg_spec = ArgSpec::build()
//!     .boolean("windowed")
//!     .string("mode")
//!     .done();
//! ```
//!
//! Second you call [`ArgSpecs`]'s [`parse()`] method to retrieve the command-line
//! arguments in a processed form.
//!
//! ```
//! let args = arg_spec.parse()?;
//! if let Some(_) = args.boolean("windowed") {
//!     // Put application into windowed mode
//! }
//! ```
//!
//! And that's it! The arguments have been parsed and processed and can be
//! accessed via [`Args`]'s getter methods.
//!
//! [`ArgSpec`] also has a [`parse()`] method so you don't have to make a
//! throwaway variable.
//!
//! ```
//! let args = ArgSpec::build()
//!     .boolean("windowed")
//!     .string("mode")
//!     .parse()?;
//! ```

pub mod args;
pub mod errors;
pub mod spec;

#[cfg(test)]
mod tests {
  use super::errors::*;
  use super::spec::*;

  #[test]
  fn spec_builder() {
    let arg_spec = ArgSpec::build()
      .boolean("b")
      .integer("n")
      .unsigned_integer("u")
      .string("name")
      .done();

    assert!(arg_spec.has_arg("b", ArgType::Boolean));
    assert!(arg_spec.has_arg("n", ArgType::Integer));
    assert!(arg_spec.has_arg("u", ArgType::UInteger));
    assert!(arg_spec.has_arg("name", ArgType::String));
    assert!(!arg_spec.has_arg("none", ArgType::Boolean));
  }

  #[test]
  fn parse() -> Result<()> {
    let args = ArgSpec::build()
      .boolean("b")
      .integer("n")
      .unsigned_integer("u")
      .string("name")
      .parse()?;
    assert_eq!(args.loan_args().len(), 0);
    Ok(())
  }
}
