use std::collections::HashMap;

/// Holds all the command-line arguments given by the user.
///
/// Each argument is contained within a [`HashMap`] that can be index by the
/// argument's name.
#[derive(Debug)]
pub struct Args {
  bools: HashMap<String, bool>,
  ints: HashMap<String, i64>,
  uints: HashMap<String, u64>,
  strs: HashMap<String, String>,
  loan_args: Vec<String>,
}

impl Args {
  pub(crate) fn new(
    bools: HashMap<String, bool>,
    ints: HashMap<String, i64>,
    uints: HashMap<String, u64>,
    strs: HashMap<String, String>,
    loan_args: Vec<String>,
  ) -> Self {
    Args {
      bools,
      ints,
      uints,
      strs,
      loan_args,
    }
  }

  /// Determines if an argument of a given name was set by the user in the
  /// command-line.
  ///
  /// # Example
  ///
  /// ```
  /// if args.is_set("window-height") {
  ///     resize_window(args.integer("window-height").unwrap());
  /// }
  /// ```
  pub fn is_set(&self, name: impl Into<String>) -> bool {
    let n = name.into();
    self
      .bools
      .keys()
      .chain(self.ints.keys())
      .chain(self.uints.keys())
      .chain(self.strs.keys())
      .find(|&k| *k == n)
      .is_some()
  }

  /// Returns a reference to the boolean value that corresponds with the
  /// given argument name.
  ///
  /// # Example
  ///
  /// ```
  /// if args.boolean("windowed") == Some(&true) {
  ///     set_windowed_mode();
  /// }
  /// ```
  pub fn boolean(&self, name: impl Into<String>) -> Option<&bool> {
    self.bools.get(&name.into())
  }

  /// Returns a reference to the i64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// let num_leaves_in_pile = *args.integer("leaves").unwrap_or(&0);
  /// ```
  pub fn integer(&self, name: impl Into<String>) -> Option<&i64> {
    self.ints.get(&name.into())
  }

  /// Returns a reference to the u64 value that corresponds with the given
  /// argument name.
  ///
  /// # Example
  ///
  /// ```
  /// let size = *args.unsigned_integer("size").unwrap_or(&0);
  /// ```
  pub fn unsigned_integer(&self, name: impl Into<String>) -> Option<&u64> {
    self.uints.get(&name.into())
  }

  /// Returns a reference to the String value that corresponds with the
  /// given argument name.
  ///
  /// # Exmaple
  ///
  /// ```
  /// let username args.string("username").unwrap_or(&"Guest".to_string())
  /// ```
  pub fn string(&self, name: impl Into<String>) -> Option<&String> {
    self.strs.get(&name.into())
  }

  /// A Vector of all strings passed as command-line arguments that weren't
  /// arguments of the [`ArgSpec`].
  ///
  /// # Example
  ///
  /// ```
  /// for arg in args.loan_args() {
  ///     println!("What the heck is this? '{}'.", arg);
  /// }
  /// ```
  pub fn loan_args(&self) -> &Vec<String> {
    &self.loan_args
  }
}
