use crate::args::*;
use crate::errors::{Result, *};
use std::collections::HashMap;

pub struct ArgSpec {
  args: HashMap<String, ArgType>,
}

type ArgIter = std::iter::Peekable<std::iter::Skip<std::env::Args>>;

impl ArgSpec {
  pub fn build() -> ArgSpecBuilder {
    ArgSpecBuilder {
      args: HashMap::new(),
    }
  }
  pub fn has_arg(&self, name: impl Into<String>, ty: ArgType) -> bool {
    if let Some(_t) = self.args.get(&name.into()) {
      matches!(ty, _t)
    } else {
      false
    }
  }
  pub fn parse(&self) -> Result<Args> {
    let mut bools: HashMap<String, bool> = HashMap::new();
    let mut ints: HashMap<String, i64> = HashMap::new();
    let mut uints: HashMap<String, u64> = HashMap::new();
    let mut strs: HashMap<String, String> = HashMap::new();
    let mut loan_args: Vec<String> = Vec::new();
    let mut args = std::env::args().skip(1).peekable();

    while let Some(arg) = args.next() {
      let mut chars = arg.chars().peekable();
      if chars.peek() == Some(&'-') {
        chars.next();
        if chars.peek() == Some(&'-') {
          chars.next();
        }
        let arg_name: String = chars.collect();
        let arg_type = *self
          .args
          .get(&arg_name)
          .ok_or(ParseError::UnknownArgument(arg_name.clone()))?;
        match arg_type {
          ArgType::Boolean => parse_bool(arg_name, &mut args, &mut bools),
          ArgType::Integer => parse_arg(arg_name, arg_type, &mut args, &mut ints)?,
          ArgType::UInteger => parse_arg(arg_name, arg_type, &mut args, &mut uints)?,
          ArgType::String => parse_arg(arg_name, arg_type, &mut args, &mut strs)?,
        }
      } else {
        loan_args.push(chars.collect());
      }
    }

    Ok(Args::new(bools, ints, uints, strs, loan_args))
  }
}

fn parse_arg<T: std::str::FromStr>(
  arg_name: String,
  arg_type: ArgType,
  args: &mut ArgIter,
  dict: &mut HashMap<String, T>,
) -> Result<()> {
  let arg_str = args
    .next()
    .ok_or(ParseError::MissingParameter(arg_name.clone(), arg_type))?;
  dict.insert(
    arg_name.clone(),
    arg_str.parse::<T>().or(Err(ParseError::InvalidParameter(
      arg_name.clone(),
      arg_type,
      arg_str,
    )))?,
  );
  Ok(())
}

fn parse_bool(arg_name: String, args: &mut ArgIter, bools: &mut HashMap<String, bool>) {
  let value = if args.peek() == Some(&"false".to_string()) {
    args.next();
    false
  } else if args.peek() == Some(&"true".to_string()) {
    args.next();
    true
  } else {
    true
  };
  bools.insert(arg_name, value);
}

impl std::fmt::Debug for ArgSpec {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    f.debug_map().entries(self.args.iter()).finish()
  }
}

#[derive(Copy, Clone, Debug)]
pub enum ArgType {
  Boolean,
  Integer,
  UInteger,
  String,
}

pub struct ArgSpecBuilder {
  args: HashMap<String, ArgType>,
}

impl ArgSpecBuilder {
  pub fn done(self) -> ArgSpec {
    ArgSpec { args: self.args }
  }
  pub fn parse(self) -> Result<Args> {
    self.done().parse()
  }
  pub fn boolean(mut self, name: impl Into<String>) -> ArgSpecBuilder {
    self.args.insert(name.into(), ArgType::Boolean);
    self
  }
  pub fn integer(mut self, name: impl Into<String>) -> ArgSpecBuilder {
    self.args.insert(name.into(), ArgType::Integer);
    self
  }
  pub fn unsigned_integer(mut self, name: impl Into<String>) -> ArgSpecBuilder {
    self.args.insert(name.into(), ArgType::UInteger);
    self
  }
  pub fn string(mut self, name: impl Into<String>) -> ArgSpecBuilder {
    self.args.insert(name.into(), ArgType::String);
    self
  }
}
