mod csv;
mod json;

use failure::Error;
use indexmap::IndexMap;
use std::io::prelude::*;

use crate::formats::csv::{Csv as CsvFormat, ID as CsvId};
use crate::formats::json::{Json as JsonFormat, ID as JsonId};
use crate::readers::{CachedReader, CACHE_LEN};

pub const DEFAULT_FORMAT: &str = JsonId;

pub trait Format {
	fn add_arguments<'a>(&self, args: clap::Command<'a>) -> clap::Command<'a>;
	fn set_arguments(&mut self, matches: &clap::ArgMatches) -> Result<(), Error>;
	fn get_extensions(&self) -> &'static [&'static str];
	fn is_valid_header(&self, header: &[u8]) -> Result<bool, Error>;
	fn parse(&self, input: &mut dyn Read) -> Result<serde_json::Value, Error>;
	fn write(&self, values: Vec<serde_json::Value>) -> Result<(), Error>;
}

pub fn load_formats() -> IndexMap<&'static str, Box<dyn Format>> {
	let mut formats: IndexMap<&'static str, Box<dyn Format>> = IndexMap::new();
	formats.insert(JsonId, Box::new(JsonFormat {}));
	formats.insert(CsvId, Box::new(CsvFormat::default()));

	formats
}

#[allow(clippy::borrowed_box)]
pub fn guess_format<'a>(
	ext: &Option<String>,
	reader: &mut CachedReader,
	formats: &'a IndexMap<&'static str, Box<dyn Format>>,
) -> Option<&'a Box<dyn Format>> {
	if let Some(ref ext) = ext {
		for (_, format) in formats {
			for pe in format.get_extensions() {
				if ext == pe {
					return Some(format);
				}
			}
		}
	}

	let mut header = [0; CACHE_LEN];
	if reader.read(&mut header).is_ok() {
		reader.rewind();

		for (_, format) in formats {
			if let Ok(is_header) = format.is_valid_header(&header) {
				if is_header {
					return Some(format);
				}
			}
		}
	}

	None
}
