use crate::errors::UnsuccessfulResponseError;
use crate::requests::*;
use crate::responses::posts::Posts;
use reqwest::{StatusCode, Url};
use std::error::Error;

pub struct Client {
    client: reqwest::Client,
    base_url: String,
    login: String,
    api_key: String,
}

impl Client {

    /// Creates an e621/e926 client
    ///
    /// # Examples
    ///
    /// ```
    /// let client = Client::new(
    ///     String::from("https://e621.net"),
    ///     String::from("MyProject/1.0 (by username on e621)"),
    ///     String::from("username"),
    ///     String::from("apikey")
    /// ).unwrap();
    /// ```
    pub fn new(
        base_url: String,
        user_agent: String,
        login: String,
        api_key: String,
    ) -> Result<Client, Box<dyn Error>> {
        let client = reqwest::Client::builder().user_agent(&user_agent).build()?;

        Ok(Client {
            client,
            base_url,
            login,
            api_key,
        })
    }


    async fn post(
        &self,
        url_path: &str,
        params: String,
    ) -> Result<String, Box<dyn std::error::Error>> {
        let result = self
            .client
            .get(Url::parse(format!("{}{}", &self.base_url, url_path).as_str()).unwrap())
            .basic_auth(&self.login, Some(&self.api_key))
            .body(params)
            .send()
            .await?;

        if result.status() == StatusCode::OK {
            Ok(result.text().await?)
        } else {
            Err(Box::new(UnsuccessfulResponseError {
                http_status_code: result.status(),
                body: result.text().await?,
            }))
        }
    }

    /// Returns posts
    ///
    /// # Arguments
    ///
    /// * `post_list_options` - Options for the post list request
    ///
    /// # Examples
    ///
    /// ```
    /// let client = Client::new(
    ///     String::from("https://e621.net"),
    ///     String::from("MyProject/1.0 (by username on e621)"),
    ///     String::from("username"),
    ///     String::from("apikey")
    /// ).unwrap();
    ///
    /// let res = client.post_list(
    ///    PostsListOptions {
    ///        limit: Some(32),
    ///        tags: Some(String::from("lucario")),
    ///        page: None
    ///    }
    /// ).await;
    /// ```
    pub async fn post_list(
        self,
        post_list_options: PostsListOptions,
    ) -> Result<Posts, Box<dyn Error>> {
        Ok(serde_json::from_str(
            self.post(
                "/posts.json",
                serde_urlencoded::to_string(post_list_options)?,
            )
            .await?
            .as_str(),
        )?)
    }
}
