
use log::*;

use std::{fs, io};
use std::sync::atomic::{AtomicU8, Ordering};

use crate::config::CONFIG;
use std::net::Ipv4Addr;
use std::process::Command;
use std::io::{Error, ErrorKind};


/// Write `/etc/nsd/*.dyn.zone` files with the IP received over UDP.
/// Config has an array of supported hostnames, host_index (the first byte of the random data portion is used to identify the host)
pub fn write_zone(ipv4: Ipv4Addr, host_index: u8) -> bool {
    let hostname = CONFIG.hostname(host_index);
    if hostname.is_none() {
        error!("unsupported host index {}", host_index);
        return false;
    }
    write_zone_file(hostname.unwrap(), ipv4, &CONFIG.zone_dir);
    true
}

/// Write `/etc/nsd/*.dyn.zone` files with the IP received over UDP and the mapped hostname.
pub fn write_zone_file(hostname: &String, ipv4: Ipv4Addr, zone_dir: &String) {

    let mut zone_entry = String::from("");
    zone_entry.push_str(hostname);
    let mut i = 32 - hostname.len();
    while i > 0 {
        zone_entry.push_str(" ");
        i -= 1;
    }
    zone_entry.push_str(" 5m  IN  A     ");
    zone_entry.push_str(ipv4.to_string().as_str());
    zone_entry.push_str("\n");

    let path = zone_dir.clone() + "/" + hostname + ".dyn.zone";
    if fs::write(&path, zone_entry).is_ok() {
        info!("updated ip for {} {}", hostname, ipv4.to_string());
    } else {
        error!("ip update failed for {} path={}", hostname, path);
    }

}

/// merge `/etc/nsd/*.dyn.zone` files, using the current config
pub fn zone_merge() -> io::Result<String> {
    let zone_index = ZONE.zone_index.load(Ordering::Relaxed);
    if 0 == zone_index {
        zone_merge_header(&CONFIG.zone_header)
    } else if let Some(alt_zone_header) = &CONFIG.alt_zone_header(zone_index) {
        zone_merge_header(alt_zone_header)
    } else {
        Err(Error::new(ErrorKind::Other, "".to_string()))
    }
}

/// merge `/etc/nsd/*.dyn.zone` files, using the named config
pub fn zone_merge_header(zone_header: &String) -> io::Result<String> {
    // Custom merge script
    if let Some(zone_merge) = &CONFIG.zone_merge {
        return match Command::new(zone_merge).output() {
            Ok(_) => Ok("merged".to_string()),
            Err(e) => Err(e),
        }
    }
    // default merge algo
    else {
        // equivalent of `cat zone_header *.dyn.zone zone_footer > zone_file`

        let path = CONFIG.zone_dir.clone() + "/" + zone_header;

        let mut zone_content = fs::read_to_string(&path)?;

        // dynamic zone entries
        zone_content.push_str("\n; start dynnsd\n\n");
        for host_record in fs::read_dir(&CONFIG.zone_dir)? {
            let host_record = host_record?;
            let host_path = host_record.path();
            if host_path.is_file()  {
                let file_name = host_path.file_name().unwrap();
                if file_name.to_str().unwrap().ends_with(".dyn.zone") {
                    let host_data = fs::read_to_string(host_path)?;
                    zone_content.push_str(host_data.as_str());
                }
            }
        }

        // optional zone footer
        if let Some(zone_footer_file) = &CONFIG.zone_footer {
            let path = CONFIG.zone_dir.clone() + "/" + zone_footer_file;
            let zone_footer = fs::read_to_string( &path)?;
            zone_content.push_str(zone_footer.as_str());
        } else {
            zone_content.push_str("\n; end dynnsd\n");
        }

        // write the contents
        let path = CONFIG.zone_dir.clone() + "/" + &CONFIG.zone_file;

        fs::write(&path, zone_content)?;

        Ok("merged".to_string())
    }
}

pub struct ZoneState {
    pub zone_index: AtomicU8,
}

lazy_static! {
    pub static ref ZONE: ZoneState = ZoneState { zone_index: AtomicU8::new(0) };
}

pub fn zone_change(zone_index: u8) -> io::Result<String> {
    if let Some(alt_zone_header) = &CONFIG.alt_zone_header(zone_index) {
        &ZONE.zone_index.store(zone_index, Ordering::Relaxed);
        return zone_merge_header(alt_zone_header);
    }
    Err(Error::new(ErrorKind::NotFound, "".to_string()))
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::util::init_log4rs;
    use std::str::FromStr;

    #[test]
    fn test_zone_file() {
        init_log4rs();
        let ipv4 = Ipv4Addr::from_str("10.0.3.23").unwrap();
        write_zone(ipv4, 0);
        write_zone_file(&String::from("myhost"), ipv4, &String::from("./etc/nsd"));
        write_zone_file(&String::from("otherhost"), ipv4, &String::from("./etc/nsd"));
        match zone_merge() {
            Ok(_) => {},
            Err(_) => panic!("zone_merge failed"),
        }
    }
}