//! dynnsd server component handling UDP messages.
//!


use std::net::{UdpSocket, IpAddr, Ipv4Addr};
use std::collections::HashMap;

use log::*;

use dynnsd::packet::{PACKET_LEN, validate_packet, create_ip_response_packet, create_ip_created_packet};
use dynnsd::config::CONFIG;
use dynnsd::zone::{write_zone, zone_merge, zone_change};
use dynnsd::nsd::hup_nsd;
use dynnsd::util::init_log4rs;
use std::fs;

const CARGO_PKG_VERSION: &'static str = env!("CARGO_PKG_VERSION");

/// Start a UDP server that handles incoming packets.
pub fn main() {

    init_log4rs();

    info!("starting dynnsd {}", CARGO_PKG_VERSION);

    // write pid ignore errors
    match fs::write("/run/dynnsd.pid", std::process::id().to_string()) {
        Ok(_) => {},
        Err(_) => warn!("could not write pid file"),
    }

    if CONFIG.uid > 0 {
        unsafe {
            libc::setuid(CONFIG.uid);
        }
    }

    let secret = &CONFIG.secret;
    let mut bound_ips: HashMap<u8, Ipv4Addr> = HashMap::new();

    if let Ok(socket) = UdpSocket::bind(&CONFIG.bind_address) {
        info!("bound to {}", socket.local_addr().unwrap());
        let mut packet = [0; PACKET_LEN];
        loop {
            if let Ok((len, sock_address)) = socket.recv_from(&mut packet) {
                if let IpAddr::V4(ipv4) = sock_address.ip() {
                    match validate_packet(len, packet, secret) {
                        Ok((typ, ip)) => {

                            if typ == dynnsd::REQUEST {
                                info!("recv IP_REQUEST");
                                // `IP_REQUEST` client does not know its ip yet, send `IP_RESPONSE` back to the client (signed)
                                socket.send_to(&create_ip_response_packet(ipv4.octets(), secret), &sock_address).ok();
                            }
                            else if typ == dynnsd::CONFIRM {
                                info!("recv IP_CONFIRM");
                                // `IP_CONFIRM` client knows their IP
                                let host_index = packet[8];
                                let binding = bound_ips.get(&host_index);
                                if binding.is_none() || ! binding.unwrap().eq(&ipv4) {
                                    write_zone(ipv4, host_index);
                                    match zone_merge() {
                                        Ok(_) => {
                                            hup_nsd();
                                            bound_ips.insert(host_index, ipv4);
                                        }
                                        Err(e) => error!("error writing zone {:?}", e)
                                    }
                                }

                                // reply ok
                                socket.send_to(&create_ip_created_packet(host_index, secret), &sock_address).ok();
                            }
                            else if typ == dynnsd::CHANGE {
                                info!("recv ZONE_CHANGE");
                                let zone_index = ip[0];
                                match zone_change(zone_index) {
                                    Ok(_) => {
                                        hup_nsd();
                                    },
                                    Err(e) => error!("error changing zone {:?}", e)
                                }
                            }
                            else if typ == dynnsd::HUP {
                                info!("recv ZONE_HUP");
                                hup_nsd();
                                // clear bindings on the assumption that we get a HUP because the zone files have been overwritten
                                // TODO verify this and if it is the case rewrite dynamic addresses
                                bound_ips.clear();

                            }
                        }
                        Err(_) => {
                            debug!("bad packet");
                        }
                    }
                }
            }
        }
    } else {
        panic!("cannot bind");
    }


}
