
use sha2::{Sha256,Digest};
use crate::{IP_LEN, HASH_LEN, HDR_LEN, IP_REQUEST, IP_CONFIRM, IP_CREATED, IP_RESPONSE, ZONE_CHANGE, ZONE_CHANGED, ZONE_HUP};
use rand::RngCore;

pub const PACKET_LEN: usize = HDR_LEN + IP_LEN + HASH_LEN + HASH_LEN;

pub fn packet_type() -> u8 {
    0
}

/// Create a signed `IP_REQUEST` packet, client wants to know its IP address.
pub fn create_ip_request_packet(secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {

    let ip = [0u8; IP_LEN];
    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);

    let mut sha256 = Sha256::new();
    sha256.input(IP_REQUEST);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet: [u8; PACKET_LEN] = [0; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&IP_REQUEST);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());

    return packet;
}

/// Create a signed `IP_RESPONSE` packet, server responds with the clients IP
pub fn create_ip_response_packet(ip: [u8; IP_LEN], secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {

    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);

    let mut sha256 = Sha256::new();
    sha256.input(IP_RESPONSE);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet: [u8; PACKET_LEN] = [0; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&IP_RESPONSE);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());
    return packet;
}

/// Create a signed `IP_CONFIRM` packet, client sends this to all the DNS nodes
pub fn create_ip_confirm_packet(ip: [u8; IP_LEN], host_index: u8, secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {

    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);
    random[0] = host_index;

    let mut sha256 = Sha256::new();
    sha256.input(IP_CONFIRM);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet: [u8; PACKET_LEN] = [0; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&IP_CONFIRM);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());

    return packet;
}

/// Create a signed `IP_CREATED` packet
pub fn create_ip_created_packet(host_index: u8, secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {
    let ip = [host_index, 0, 0, 0];
    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);

    let mut sha256 = Sha256::new();
    sha256.input(IP_CREATED);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet = [0u8; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&IP_CREATED);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());

    return packet;
}

/// Create a signed `ZONE_CHANGE` packet
pub fn create_zone_change(zone_index: u8, secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {
    let ip = [zone_index, 0, 0, 0];
    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);

    let mut sha256 = Sha256::new();
    sha256.input(ZONE_CHANGE);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet = [0u8; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&ZONE_CHANGE);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());

    return packet;
}

/// Create a signed `ZONE_CHANGED` packet
pub fn create_zone_changed(zone_index: u8, secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {
    let ip = [zone_index, 0, 0, 0];
    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);

    let mut sha256 = Sha256::new();
    sha256.input(ZONE_CHANGED);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet = [0u8; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&ZONE_CHANGED);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());

    return packet;
}

/// Create a signed `ZONE_HUP` packet
pub fn create_zone_hup_packet(secret: &[u8; HASH_LEN]) -> [u8; PACKET_LEN] {
    let ip = [0, 0, 0, 0];
    let mut random = [0u8; HASH_LEN];
    rand::thread_rng().fill_bytes(&mut random);

    let mut sha256 = Sha256::new();
    sha256.input(ZONE_HUP);
    sha256.input(ip);
    sha256.input(random);
    sha256.input(secret);

    let mut packet = [0u8; PACKET_LEN];
    packet[0..HDR_LEN].copy_from_slice(&ZONE_HUP);
    packet[HDR_LEN..HDR_LEN+IP_LEN].copy_from_slice(&ip);
    packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN].copy_from_slice(&random);
    packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].copy_from_slice(sha256.result().as_slice());

    return packet;
}

/// validate a packets signature
/// returns the message type & ip address if it is signed
pub fn validate_packet(len: usize, packet: [u8; PACKET_LEN], secret: &[u8; HASH_LEN]) -> Result<(u8, [u8; IP_LEN]), ()> {
    if len != PACKET_LEN || 23 != packet[0] {
        return Err(());
    }

    let mut sha256 = Sha256::new();
    sha256.input(&packet[0..HDR_LEN]);
    sha256.input(&packet[HDR_LEN..HDR_LEN+IP_LEN]);
    sha256.input(&packet[HDR_LEN+IP_LEN..HDR_LEN+IP_LEN+HASH_LEN]);
    sha256.input(secret);

    // compare iterators over u8
    return if sha256.result().iter().eq(packet[HDR_LEN+IP_LEN+HASH_LEN..PACKET_LEN].iter()) {

        let mut ip: [u8; IP_LEN] = [0; IP_LEN];
        ip.copy_from_slice(&packet[HDR_LEN..HDR_LEN + IP_LEN]);

        Ok((packet[1], ip))
    } else {
        Err(())
    }

}


#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_happy() {
        let mut ip: [u8; IP_LEN] = [0; IP_LEN];
        ip[0] = 10;
        ip[IP_LEN - 1] = 23;
        let mut random: [u8; HASH_LEN] = [0; HASH_LEN];
        random[0] = 23;
        random[HASH_LEN - 1] = 24;
        let secret: [u8; HASH_LEN] = [0; HASH_LEN];

        let packet = create_ip_request_packet(&secret);
        assert!(validate_packet(packet.len(), packet, &secret).ok().is_some());

        let packet = create_ip_response_packet([10 as u8, 0, 0, 23], &secret);
        assert!(validate_packet(packet.len(), packet, &secret).ok().is_some());

        let packet = create_ip_confirm_packet([10 as u8, 0, 0, 23], 0, &secret);
        assert!(validate_packet(packet.len(), packet, &secret).ok().is_some());

        let packet = create_ip_created_packet( 0, &secret);
        assert!(validate_packet(packet.len(), packet, &secret).ok().is_some());
    }
}
