
use log::*;

use crate::{HASH_LEN};
use serde_derive::Deserialize;
use sha2::{Sha256, Digest};
use std::fs;

static SERVER_CONFIG_FILE: &str = "dynnsd.toml";
static CLIENT_CONFIG_FILE: &str = "dynnsd-client.toml";

lazy_static! {
    pub static ref CONFIG: ServerConfig = init_server_config();
    pub static ref CLIENT_CONFIG: ClientConfig = init_client_config();
}

/// Initialize server configuration is called by lazy_static when anything references CONFIG
fn init_server_config() -> ServerConfig {

    let mut conf_path = String::from("./etc");
    if let Ok(var) = std::env::var("DYNNSD_CONF") {
        conf_path = String::from(var.as_str());
    }

    conf_path.push('/');
    conf_path.push_str(SERVER_CONFIG_FILE);

    match fs::read_to_string(conf_path) {
        Ok(conf_data) => {
            let mut conf: ServerConfig = toml::from_str(&conf_data).unwrap();
            conf.init();
            return conf;
        }
        Err(e) => {
            error!("loading config failed: {} {}", SERVER_CONFIG_FILE, e);
            std::process::exit(2);
        }
    }
}

/// Initialize server configuration is called by lazy_static when anything references CLIENT_CONFIG
fn init_client_config() -> ClientConfig {

    let mut conf_path = String::from("./etc");
    if let Ok(var) = std::env::var("DYNNSD_CONF") {
        conf_path = String::from(var.as_str());
    }

    conf_path.push('/');
    conf_path.push_str(CLIENT_CONFIG_FILE);

    match fs::read_to_string(conf_path.clone()) {
        Ok(conf_data) => {
            if let Ok(conf) = toml::from_str(&conf_data) {
                let mut conf: ClientConfig = conf;
                conf.init();
                return conf;
            }
            else {
                panic!("could not parse {}", conf_path);
            }
        }
        Err(e) => {
            error!("loading config failed: {} {}", conf_path, e);
            std::process::exit(2);
        }
    }
}

#[derive(Deserialize, Debug)]
#[serde(default)]
pub struct ServerConfig {
    /// user id of the process
    pub uid: u32,
    /// socket to bind the UDP server, defaults to 127.0.0.1:53535
    pub bind_address: String,
    /// Shared secret
    pub secret_string: String,
    /// number of times to hash the secret on boot to harden the password and fix its length
    pub secret_iterations: u32,
    /// hardened secret
    pub secret: [u8; HASH_LEN],
    /// path to nsd's pid file, process is HUPed when a new DNS entry is added, defaults to /run/nsd.pid
    pub nsd_pid: String,
    /// zone directory to create A records, defaults to /etc/nsd/
    pub zone_dir: String,
    /// name of the zone file to create
    pub zone_file: String,
    /// main zone header file
    pub zone_header: String,
    /// optional zone footer file
    pub zone_footer: Option<String>,
    /// optional script to merge zone files together
    /// defaults to the equivalent of cat zone_header *.dyn.zone zone_footer > zone_file
    pub zone_merge: Option<String>,
    /// 0 indexed list of hostnames
    hostnames: Vec<String>,
    /// 0 indexed list of zone files
    alt_zone_header: Vec<String>,
}

impl Default for ServerConfig {
    fn default() -> Self {
        ServerConfig {
            uid: 0,
            bind_address: "127.0.0.1:53535".to_string(),
            secret_string: String::from("12345678901234567890123456789012"),
            secret_iterations: 5,
            secret: [0; HASH_LEN],
            nsd_pid: "/run/nsd.pid".to_string(),
            zone_dir: String::from("/etc/nsd"),
            zone_file: "example.com.zone".to_string(),
            zone_header: "example.com.static.zone".to_string(),
            zone_footer: None,
            zone_merge: None,
            hostnames: vec![],
            alt_zone_header: vec![]
        }
    }
}

impl ServerConfig {
    
    pub fn init(&mut self) {
        // config validation
        if self.hostnames.len() == 0 {
            error!("at least one hostname is required");
            std::process::exit(2);
        }
        if self.secret_string.eq("12345678901234567890123456789012") {
            warn!("running with default password!");
        }
        if self.secret_string.len() < 6 {
            warn!("weak password!");
        }
        if ! fs::read_to_string(&self.nsd_pid).is_ok()  {
            info!("cannot read nsd pid file (yet)");
        }
        for hostname in &self.hostnames {
            if ! sanename::validate(hostname.as_str()) {
                warn!("hostnames should be sanenames {}", hostname);
            }
        }
        self.secret = harden_password(&self.secret_string, self.secret_iterations);
    }

    pub fn hostname(&self, idx: u8) -> Option<&String> {
        if let Some(hostname) = self.hostnames.get(idx as usize) {
            return Some(&hostname);
        }
        None
    }

    pub fn alt_zone_header(&self, idx: u8) -> Option<&String> {
        if let Some(alt_zone_header) = self.alt_zone_header.get(idx as usize) {
            return Some(&alt_zone_header);
        }
        None
    }
}



#[derive(Deserialize, Debug)]
#[serde(default)]
pub struct ClientConfig {
    /// ip and port of the dynnsd servers
    pub dynnsd_addresses: Vec<String>,
    /// index in the server config that applies to this client
    pub host_index: u8,
    /// socket to bind the UDP server, defaults to 0.0.0.0:53536
    pub bind_address: String,
    /// shared secret
    pub secret_string: String,
    /// number of times to hash the secret on boot to harden the password and fix its length
    pub secret_iterations: u32,
    /// hardened secret
    pub secret: [u8; HASH_LEN],
    /// seconds wait between udp message retries
    pub retry_wait: u64,
}

impl Default for ClientConfig {
    fn default() -> Self {
        ClientConfig {
            dynnsd_addresses: vec!["ns1".to_string(), "ns2".to_string()],
            host_index: 0,
            bind_address: "0.0.0.0:53536".to_string(),
            secret_string: String::from("12345678901234567890123456789012"),
            secret_iterations: 5,
            secret: [0; HASH_LEN],
            retry_wait: 5,
        }
    }
}

impl ClientConfig {

    pub fn init(&mut self) {
        // config validation
        if self.dynnsd_addresses.len() == 0 {
            error!("at least one dynnsd server is required");
            std::process::exit(2);
        }
        if self.secret_string.eq("12345678901234567890123456789012") {
            warn!("running with default password!");
        }
        if self.secret_string.len() < 6 {
            warn!("weak password!");
        }
        self.secret = harden_password(&self.secret_string, self.secret_iterations);
    }

}

fn harden_password(secret_string: &String, iterations: u32) -> [u8; HASH_LEN]{
    let mut secret = [0u8; HASH_LEN];
    let mut result_mut = [0u8; HASH_LEN];

    result_mut[0..secret_string.len()].copy_from_slice(secret_string.as_bytes());

    let mut sha256 = Sha256::new();
    sha256.input(result_mut);
    result_mut.copy_from_slice(sha256.result().as_slice());

    let mut i = iterations;
    while i > 0 {
        sha256 = Sha256::new();
        sha256.input(result_mut);
        result_mut.copy_from_slice(sha256.result().as_slice());
        i -= 1;
    }
    secret.copy_from_slice(&result_mut);
    secret
}



#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_happy() {
        println!("{:?}", init_server_config());
        println!("{:?}", init_client_config());
    }
}