
use log::*;

use std::net::{UdpSocket, SocketAddrV4, SocketAddr};
use std::str::FromStr;
use std::time::Duration;

use dynnsd::config::CLIENT_CONFIG;
use dynnsd::packet::{PACKET_LEN, validate_packet, create_ip_request_packet, create_ip_confirm_packet};
use dynnsd::{IP_LEN};
use dynnsd::util::init_log4rs;
use std::thread::sleep;

const CARGO_PKG_VERSION: &'static str = env!("CARGO_PKG_VERSION");

/// Client should be run from cron  (compiled as dynnsd-client)
/// Executes loops until it knows its own IP and has told bit to both remote DNS servers.
fn main() {

    init_log4rs();

    info!("starting dynnsd-client {}", CARGO_PKG_VERSION);

    let mut loops = 100;

    // initialize client configuration
    let secret = &CLIENT_CONFIG.secret;
    let host_index: u8 = CLIENT_CONFIG.host_index;
    let dynnsd_addresses: Vec<String> = CLIENT_CONFIG.dynnsd_addresses.clone();
    let mut dynnsd_sockets: Vec<SocketAddrV4> = vec![];

    for address in dynnsd_addresses {
        match SocketAddrV4::from_str(&address) {
            Ok(dynnsd_socket) => dynnsd_sockets.push(dynnsd_socket),
            Err(e) => {
                error!("invalid server socket '{}' {}", address, e);
                std::process::exit(2);
            },
        }
    }

    // bind to a UDP socket for sending
    if let Ok(client_socket) = UdpSocket::bind(&CLIENT_CONFIG.bind_address) {
        debug!("outbound {}", &CLIENT_CONFIG.bind_address);
        let mut ip_out = [0u8; IP_LEN];
        let mut packet_in = [0u8; PACKET_LEN];

        match client_socket.set_read_timeout(Some(Duration::new(CLIENT_CONFIG.retry_wait, 0))) {
            Ok(_) => {}
            Err(_) => {
                // timeout is needed for the loops to work, no guarantee of a reply with udp
                error!("cant set timeout for udp");
                std::process::exit(1);
            }
        }

        while loops > 0 {
            // send the same `IP_REQUEST` message to both dynnsd servers with zeroed ip_out
            for server_socket in &dynnsd_sockets {
                client_socket.send_to(&create_ip_request_packet(secret), server_socket).ok();
                debug!("send IP_REQUEST to {:?}:{}", server_socket.ip(), server_socket.port());
            }

            // handle `IP_RESPONSE`, only one response is handled
            if let Ok((len, sock_address)) = client_socket.recv_from(&mut packet_in) {
                if len == PACKET_LEN {
                    if let Ok((typ, ip_in)) = validate_packet(len, packet_in, secret) {
                        if typ == dynnsd::RESPONSE {
                            // valid response we now know our IP address
                            ip_out.copy_from_slice(&ip_in);
                            debug!("IP_RESPONSE {:?}", ip_in);
                            break;
                        }
                    }
                    warn!("invalid packet from {:?}", sock_address);
                }
            }
            sleep(Duration::new(1, 0));
            loops = loops -1;
        }

        // send `IP_CONFIRM` in a loop until we get `IP_CREATED` from both servers
        while loops > 0 {
            for server_socket in &dynnsd_sockets {
                client_socket.send_to(&create_ip_confirm_packet(ip_out, host_index, secret), server_socket).ok();
            }
            if let Ok((len, sock_address_in)) = client_socket.recv_from(&mut packet_in) {
                if len == PACKET_LEN {
                    match validate_packet(len, packet_in, secret) {
                        Ok((typ, ip)) => {
                            if typ == dynnsd::CREATED {
                                if let SocketAddr::V4(sock_address_in) = sock_address_in {
                                    &dynnsd_sockets.retain(|address| !address.eq(&sock_address_in));
                                    info!("dynnsd updated {:?}", sock_address_in);
                                }
                            }
                            else if typ == dynnsd::RESPONSE {
                                debug!("IP_RESPONSE not needed");
                            }
                            else {
                                debug!("response not expected {}/{:?}", typ, ip);
                            }
                        }
                        _=> warn!("invalid packet from {:?}", sock_address_in)
                    }
                }
            }
            if dynnsd_sockets.len() == 0 {
                std::process::exit(0);
            }
            sleep(Duration::new(1, 0));
            loops = loops -1;
        }

        error!("did not complete in 100 loops");
        std::process::exit(1);

    } else {
        panic!("cannot bind udp socket");
    }
}

