
use log::*;

use std::net::{UdpSocket, SocketAddrV4};
use std::str::FromStr;
use std::time::Duration;

use dynnsd::config::CLIENT_CONFIG;
use dynnsd::packet::create_zone_hup_packet;
use dynnsd::util::init_log4rs;

/// Client should be run when ever zone files have changed (compiled as dynnsd-hup)
/// Calls a remote dynnsd instance and instructs it to re-read config
/// see dynnsd-client.toml
fn main() {

    init_log4rs();


    // initialize client configuration
    let secret = &CLIENT_CONFIG.secret;
    let dynnsd_addresses: Vec<String> = CLIENT_CONFIG.dynnsd_addresses.clone();
    let mut dynnsd_sockets: Vec<SocketAddrV4> = vec![];

    for address in dynnsd_addresses {
        match SocketAddrV4::from_str(&address) {
            Ok(dynnsd_socket) => dynnsd_sockets.push(dynnsd_socket),
            Err(e) => {
                error!("invalid server socket '{}' {}", address, e);
                std::process::exit(2);
            },
        }
    }

    // bind to a UDP socket for sending
    if let Ok(client_socket) = UdpSocket::bind(&CLIENT_CONFIG.bind_address) {
        debug!("outbound {}", &CLIENT_CONFIG.bind_address);

        match client_socket.set_read_timeout(Some(Duration::new(CLIENT_CONFIG.retry_wait, 0))) {
            Ok(_) => {}
            Err(_) => {
                // timeout is needed for the loops to work, no guarantee of a reply with udp
                error!("cant set timeout for udp");
                std::process::exit(1);
            }
        }

        for server_socket in &dynnsd_sockets {
            client_socket.send_to(&create_zone_hup_packet(secret), server_socket).ok();
            debug!("send ZONE_HUP to {:?}:{}", server_socket.ip(), server_socket.port());
        }

        info!("sent ZONE_HUPs ");
        std::process::exit(0);

    } else {
        panic!("cannot bind udp socket");
    }
}

