# Dynamize

In order for a trait to be usable as a trait object it needs to fulfill
[several requirements](https://doc.rust-lang.org/reference/items/traits.html#object-safety).
For example:

```rust ignore
trait Client {
    type Error;

    fn get(&self, url: String) -> Result<Vec<u8>, Self::Error>;
}

impl Client for HttpClient { type Error = HttpError; ...}
impl Client for FtpClient { type Error = FtpError; ...}

let client: HttpClient = ...;
let object = &client as &dyn Client;
```

The last line of the above code fails to compile with:

> error\[E0191\]: the value of the associated type `Error` (from trait `Client`)
> must be specified

Sometimes you however want a trait object to be able to encompass trait
implementations with different associated type values. This crate provides an
attribute macro to achieve that. To use dynamize you only have to make some
small changes:

```rust ignore
#[dynamize::dynamize]
trait Client {
    type Error: Into<SuperError>;

    fn get(&self, url: String) -> Result<Vec<u8>, Self::Error>;
}

let client: HttpClient = ...;
let object = &client as &dyn DynClient;
```

1. You add the `#[dynamize::dynamize]` attribute to your trait.
2. You specify a trait bound for each associated type.

Dynamize defines a new trait for you, named after your trait but with the `Dyn`
prefix, so e.g. `Client` becomes `DynClient` in our example. The new
"dynamized" trait can then be used without having to specify the associated
type.

## How does this work?

For the above example dynamize generates the following code:

```rust ignore
trait DynClient {
    fn get(&self, url: String) -> Result<Vec<u8>, SuperError>;
}

impl<__to_be_dynamized: Client> DynClient for __to_be_dynamized {
    fn get(&self, url: String) -> Result<Vec<u8>, SuperError> {
        Client::get(self, url).map_err(|x| x.into())
    }
}
```

As you can see in the dynamized trait the associated type was replaced with the
destination type of the `Into` bound. The magic however happens afterwards:
dynamize generates a blanket implementation: each type implementing `Client`
automatically also implements `DynClient`!

## How does this actually work?

The destination type of an associated type is
determined by looking at its first trait bound:

* `Into<T>` is mapped to `T`

* `SomeTrait` is mapped to `Box<dyn SomeTrait>`  
  (for this `SomeTrait` of course needs to be object-safe)

Dynamize can convert associated types in:

* return types, e.g. `fn example(&self) -> Self::A`
* callback parameters, e.g. `fn example<F: Fn(Self::A)>(&self, f: F)`

Dynamize also understands if you wrap associated types in the following types:

* `Option<_>`
* `Result<_, _>`
* `some::module::Result<_>` (assumed to be a Result type alias)
* `&mut dyn Iterator<Item = _>`

Note that since these are resolved recursively you can actually nest these
arbitrarily so e.g. the following also just works:

```rs
fn example(&self) -> Result<Option<Self::Item>, Self::Error>;
```

## How does dynamize deal with method generics?

In order to be object-safe methods must not have generics, so dynamize simply
moves them to the trait definition:

```rs
trait Gen {
    fn foobar<A>(&self, a: A) -> A;
}
```

becomes

```rs
trait DynGen<A> {
    fn foobar(&self, a: A) -> A;
}
```

If two method type parameters have the same name, dynamize enforces that they
also have the same bounds and only adds the parameter once to the trait.

## Dynamize supports async

Dynamize supports async out of the box. Since Rust however does not yet support
async functions in traits, you'll have to additionally use another library like
[async-trait](https://crates.io/crates/async-trait), for example:

```rust ignore
#[dynamize::dynamize]
#[dyn_trait_attr(async_trait)]
#[blanket_impl_attr(async_trait)]
#[async_trait]
trait Client: Sync {
    type Error: std::error::Error;

    async fn get(&self, url: String) -> Result<Vec<u8>, Self::Error>;
}
```

The `#[dyn_trait_attr(...)]` attribute lets you attach macro attributes to the
generated dynamized trait.  The `#[blanket_impl_attr(...)]` attribute lets you
attach macro attributes to the generated blanket implementation. Note that it
is important that the dynamize attribute comes before the `async_trait`
attribute.
