use proc_macro2::Span;
use quote::{quote, ToTokens};
use syn::spanned::Spanned;
use syn::{GenericArgument, Ident, Path, PathArguments, PathSegment, TraitItemType, Type};

use crate::parse_trait_sig::TypeTransform;
use crate::syn_utils::{find_in_type, lifetime_bounds, trait_bounds};
use crate::AssocTypeMatcher;

#[derive(Debug)]
pub enum AssocTypeParseError {
    AssocTypeInBound,
    GenericAssociatedType,
    NoIntoBound,
}

#[derive(Debug, Clone)]
pub struct BoxType {
    trait_name: Path,
    placeholder_lifetime: bool,
}

impl ToTokens for BoxType {
    fn to_tokens(&self, tokens: &mut proc_macro2::TokenStream) {
        let path = &self.trait_name;
        match self.placeholder_lifetime {
            true => tokens.extend(quote! {Box<dyn #path + '_>}),
            false => tokens.extend(quote! {Box<dyn #path>}),
        }
    }
}

pub enum DestType<'a> {
    Into(&'a Type),
    Box(BoxType),
}

impl DestType<'_> {
    pub fn get_dest(&self) -> Type {
        match self {
            DestType::Into(ty) => (*ty).clone(),
            DestType::Box(b) => Type::Verbatim(quote!(#b)),
        }
    }

    pub fn type_transformation(&self) -> TypeTransform {
        match self {
            DestType::Into(_) => TypeTransform::Into,
            DestType::Box(b) => TypeTransform::Box(b.clone()),
        }
    }
}

pub fn parse_assoc_type(
    assoc_type: &TraitItemType,
) -> Result<(&Ident, DestType), (Span, AssocTypeParseError)> {
    if let Some(bound) = trait_bounds(&assoc_type.bounds).next() {
        if let PathSegment {
            ident,
            arguments: PathArguments::AngleBracketed(args),
        } = bound.path.segments.first().unwrap()
        {
            if ident == "Into" && args.args.len() == 1 {
                if let GenericArgument::Type(into_type) = args.args.first().unwrap() {
                    // provide a better error message for type A: Into<Self::B>
                    if find_in_type(into_type, &AssocTypeMatcher).is_some() {
                        return Err((into_type.span(), AssocTypeParseError::AssocTypeInBound));
                    }

                    // TODO: support lifetime GATs (see the currently failing tests/gats.rs)
                    if !assoc_type.generics.params.is_empty() {
                        return Err((
                            assoc_type.generics.params.span(),
                            AssocTypeParseError::GenericAssociatedType,
                        ));
                    }

                    return Ok((&assoc_type.ident, DestType::Into(into_type)));
                }
            }
        }
        return Ok((
            &assoc_type.ident,
            DestType::Box(BoxType {
                trait_name: bound.path.clone(),
                placeholder_lifetime: !lifetime_bounds(&assoc_type.bounds)
                    .any(|l| l.ident == "static"),
            }),
        ));
    }
    Err((assoc_type.span(), AssocTypeParseError::NoIntoBound))
}

#[cfg(test)]
mod tests {
    use quote::quote;
    use syn::{TraitItemType, Type};

    use crate::parse_assoc_type::{parse_assoc_type, AssocTypeParseError, DestType};

    #[test]
    fn ok() {
        let type1: TraitItemType = syn::parse2(quote! {
            type A: Into<String>;
        })
        .unwrap();

        assert!(matches!(
            parse_assoc_type(&type1),
            Ok((id, DestType::Into(Type::Path(path))))
            if id == "A" && path.path.is_ident("String")
        ));
    }

    #[test]
    fn err_no_bound() {
        let type1: TraitItemType = syn::parse2(quote! {
            type A;
        })
        .unwrap();

        assert!(matches!(
            parse_assoc_type(&type1),
            Err((_, AssocTypeParseError::NoIntoBound))
        ));
    }

    #[test]
    fn err_assoc_type_in_bound() {
        let type1: TraitItemType = syn::parse2(quote! {
            type A: Into<Self::B>;
        })
        .unwrap();

        assert!(matches!(
            parse_assoc_type(&type1),
            Err((_, AssocTypeParseError::AssocTypeInBound))
        ));
    }

    #[test]
    fn err_gat_type() {
        let type1: TraitItemType = syn::parse2(quote! {
            type A<X>: Into<Foobar<X>>;
        })
        .unwrap();

        assert!(matches!(
            parse_assoc_type(&type1),
            Err((_, AssocTypeParseError::GenericAssociatedType))
        ));
    }

    #[test]
    fn err_gat_lifetime() {
        let type1: TraitItemType = syn::parse2(quote! {
            type A<'a>: Into<Foobar<'a>>;
        })
        .unwrap();

        assert!(matches!(
            parse_assoc_type(&type1),
            Err((_, AssocTypeParseError::GenericAssociatedType))
        ));
    }
}
