use std::str::FromStr;
use std::io::{BufRead, BufReader, Read, Write };

#[derive(std::fmt::Debug)]
pub enum LogLevel {
    Error,
    Warn,
    Info,
    Debug,
    Trace,
}

impl FromStr for LogLevel {
    type Err = ();
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s.to_lowercase().as_str() {
            "--error" => Ok(Self::Error),
            "--warn" => Ok(Self::Warn),
            "--info" => Ok(Self::Info),
            "--debug" => Ok(Self::Debug),
            "--trace" => Ok(Self::Trace),
            _ => Err(()),
        }
    }
}

pub(crate) fn get_file_path<'a>(index: &'a str, name: String) -> String {
    match index {
        "readme" => name + "/README.md",
        "config" => name + "/config",
        "cargo" => name + "/Cargo.toml",
        "entity" => name + "/src/entity.rs",
        "parser" => name + "/src/parser.rs",
        "spider" => name + "/src/spider.rs",
        "middleware" => name + "/src/middleware.rs",
        "pipeline" => name + "/src/pipeline.rs",
        "lib" => name + "/src/lib.rs",
        _ => panic!(),
    }
}

pub(crate) fn get_file_intro(index: &str) -> &str {
    match index {
        "readme" => {
            r#"<!-- 
-This is a markdown file generated by dyer-cli
- Instructions of the project specified here 
--!>"#
        }
        "lib" => {
            r#"pub mod entity;
pub mod spider;
pub mod middleware;
pub mod pipeline;
pub mod parser; "#
        }
        "entity" => {
            r#"use serde::{Deserialize, Serialize};
use dyer::dyer_macros::entity;

/*
 * the Entity to be used
 *
 *#[derive(Deserialize, Serialize, Debug, Clone)]
 *pub struct Item1 {
 *    pub field1: String,
 *    pub field2: i32,
 *}
 */

/* serve as a placeholder for all entities, and generic parameter of dyer::App
 * attribute #[entity(entities)] mark the enum and use it as container to all data to be collected
 */
#[entity(entities)]
#[derive(Serialize, Debug, Clone)]
pub enum Entities {
    //Item1(Item1),
}

// serve as a appendix/complement to dyer::Task,
// leave it empty if not necessary
// attribute #[entity(targ)] mark the struct and use it as generic type for `Task`
#[entity(targ)]
#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct Targ {}

// serve as a appendix/complement to dyer::Profile
// leave it empty as default
// attribute #[entity(parg)] mark the struct and use it as generic type for `Profile`
#[entity(parg)]
#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct Parg {}"#
        }
        "middleware" => {
            r#"use crate::entity::{Entities, Parg, Targ};
use dyer::App;
use dyer::dyer_macros::middleware;

/* attribute #[middleware(attr)] mark the method and use it as that in `MiddleWare`
 * attr could be :
 *    handle_entity/handle_req/handle_task/handle_profile
 *    /handle_res/handle_err/handle_yerr
 */
#[middleware(handle_entity)]
pub async fn handle_entities(_items: &mut Vec<Entities>, _app: &mut App<Entities, Targ, Parg>) {}
"#
        }
        "parser" => {
            r#"use crate::entity::{Entities, Parg, Targ};
use dyer::{ParseResult, Response};
use dyer::dyer_macros::parser;

/* note that call this function to parse via specifying task.parser:
 *     let task = Task::new();
 *     ...
 *     task.parser = "parse_func".into();
 * that means function `parse_func` is called to parse the Response.
 * attribute #[parser] mark the method and use it extract entities from `Response` marked with
 * string "parse_func"
 */
#[parser]
pub fn parse_func(_res: Response<Targ, Parg>) -> ParseResult<Entities, Targ, Parg> {
    ParseResult::new()
}"#
        }
        "pipeline" => {
            r#"use dyer::dyer_macros::pipeline;

 /*
 * something to do before sending entities to pipeline
 * the return type inside `Option` requires complete path(starts with `std` or crate in `Cargo.toml`)
 * attribute #[pipeline(attr)] mark the method and use it as that in `PipeLine` 
 * attr could be:
 *    open_pipeline/close_pipeline/process_entity/process_yerr
 */
#[pipeline(open_pipeline)]
async fn func_name<'a>() -> &'a Option<std::fs::File> {
    &None
}
"#
        }
        "spider" => {
            r#"use crate::entity::{Entities, Parg, Targ};
use crate::parser::*;
use dyer::*;
use dyer::dyer_macros::spider;

type Stem<U> = Result<U, Box<dyn std::error::Error + Send + Sync>>;
type Btem<E, T, P> = dyn Fn(Response<T, P>) -> ParseResult<E, T, P>;

// attribute #[spider] mark the struct and use it as a type implemented trait `Spider`
#[spider]
pub struct MySpider {
    pub start_uri: String,
}

impl Spider<Entities, Targ, Parg> for MySpider {
    // create an instance 
    fn new() -> Self {
        MySpider{
            start_uri: "https://example.com/some/path/to/site".into()
        }
    }

    // preparation before opening spider
    fn open_spider(&self, _app: &mut App<Entities, Targ, Parg>) {}

    /* 
     * `Task` to be executed when starting `dyer`. Note that this function must reproduce a
     * non-empty vector, if not, the whole program will be left at blank.
     */
    fn entry_task(&self) -> Stem<Vec<Task<Targ>>> {
        let mut task = Task::new();
        task.uri = self.start_uri.clone();
        Ok(vec![task])
    }

    /* the generator of `Profile`
     * `dyer` consume the returned `Request`, generate a `Response` fed to the closure
     * to generate a `Profile`
     */
    fn entry_profile<'a>(&self) -> ProfileInfo<'a, Targ, Parg> {
        ProfileInfo {
            req: Some( Request::<Targ, Parg>::new() ),
            parser: None,
        }
    }

    /* set up parser that extracts `Entities` from the `Response`
     * by the name of Task.parser return the parser function
     * parser is indexed by a `String` name, like:
     * task.parser = "parse_quote".to_string();
     */
    fn get_parser<'a>(&self, ind: &str) -> Option<&'a Btem<Entities, Targ, Parg>> {
        plug!(get_parser(ind; parse_func))
    }

    // preparation before closing spider
    fn close_spider(&self, _app: &mut App<Entities, Targ, Parg>) {}
}"#
        }
        "cargo" => {
            r#"[package]
name = "<+name+>"
version = "0.1.0"
authors = ["your name"]
edition = "2018"

# See more keys and their definitions at https://doc.rust-lang.org/cargo/reference/manifest.html

[[bin]]
name = "<+name+>"
path = ".target/main.rs"

[dependencies]
dyer = { version = "2.0" }
serde = { version = "1.0", features = ["derive"] }
tokio = { version = "0.2", features = ["rt-threaded", "macros"]}
simple_logger = "1.11" "#
        },
        "config" => {
            r#"## ArgApp
is_skip: true,
spawn_task_max: 100,
buf_task: 10000,
round_entity: 10,
data_dir: data/
nap: 15.0,
join_gap: 7.0,

## ArgProfile
arg_profile.is_on: false,
arg_profile.profile_min: 0,
arg_profile.profile_max: 0,

## ArgRate
rate.cycle: 600.0,
rate.load: 99.0,
rate.rate_low: 0.333,
rate.err: 0,
rate.interval: 30.0,
"#
        }
        _ => "",
    }
}

pub(crate) fn run_command(cmd: &str, options: Vec<&str>) {
    let stdout = std::process::Command::new(cmd)
        .args(options)
        .stdout(std::process::Stdio::piped())
        .spawn()
        .unwrap()
        .stdout
        .ok_or_else(|| {
            std::io::Error::new(
                std::io::ErrorKind::Other,
                "Could not capture standard output.",
            )
        })
        .unwrap();

    let reader = BufReader::new(stdout);

    reader
        .lines()
        .filter_map(|line| line.ok())
        .filter(|line| line.find("src\\main.rs").is_none())
        .filter(|line| line.find("src/main.rs").is_none())
        .for_each(|line| println!("{}", line));
}

pub(crate) fn change_log_level(level: &str) {
    let mut file = std::fs::OpenOptions::new().read(true).open(".target/main.rs").unwrap();
    let mut buf = String::new();
    file.read_to_string(&mut buf).unwrap();
    drop(file);
    let ll = level.strip_prefix("--").unwrap();
    let l = &( "log::LevelFilter::".to_string() + &to_camelcase(ll) );
    let buf = buf.replace("log::LevelFilter::Error", l);
    let buf = buf.replace("log::LevelFilter::Warn", l);
    let buf = buf.replace("log::LevelFilter::Info", l);
    let buf = buf.replace("log::LevelFilter::Debug", l);
    let buf = buf.replace("log::LevelFilter::Trace", l);
    let buf = buf.replace("log::LevelFilter::Off", l);
    let mut file = std::fs::OpenOptions::new().truncate(true).write(true).open(".target/main.rs").unwrap();
    file.write_all(buf.as_bytes()).unwrap();
}

pub(crate) fn to_camelcase(s: &str) -> String {
    let mut r = String::with_capacity(s.len());
    let mut ch = s[..].chars();
    let e = ch.next().unwrap().to_uppercase();
    r.push(e.to_string().chars().next().unwrap());
    while let Some(t) = ch.next() {
        r.push(t);
    }
    r
    
}
