use std::borrow::Cow;
use std::collections::HashMap;

use chrono::{DateTime, Utc};

use crate::error::DxrError;
use crate::traits::FromDXR;
use crate::values::{Type, Value};

use super::utils::*;

impl FromDXR for Value {
    fn from_dxr(value: &Value) -> Result<Value, DxrError> {
        Ok(value.clone())
    }
}

impl FromDXR for i32 {
    fn from_dxr(value: &Value) -> Result<i32, DxrError> {
        match value.inner() {
            Type::Integer(int) => Ok(*int),
            t => Err(DxrError::wrong_type(t.name(), "i4")),
        }
    }
}

#[cfg(feature = "i8")]
#[cfg_attr(docsrs, doc(cfg(feature = "i8")))]
impl FromDXR for i64 {
    fn from_dxr(value: &Value) -> Result<i64, DxrError> {
        match value.inner() {
            Type::Long(long) => Ok(*long),
            t => Err(DxrError::wrong_type(t.name(), "i8")),
        }
    }
}

impl FromDXR for bool {
    fn from_dxr(value: &Value) -> Result<bool, DxrError> {
        match value.inner() {
            Type::Boolean(boo) => Ok(*boo),
            t => Err(DxrError::wrong_type(t.name(), "boolean")),
        }
    }
}

impl FromDXR for String {
    fn from_dxr(value: &Value) -> Result<String, DxrError> {
        match value.inner() {
            Type::String(string) => Value::string_unescape(string),
            t => Err(DxrError::wrong_type(t.name(), "string")),
        }
    }
}

impl FromDXR for f64 {
    fn from_dxr(value: &Value) -> Result<f64, DxrError> {
        match value.inner() {
            Type::Double(double) => Ok(*double),
            t => Err(DxrError::wrong_type(t.name(), "double")),
        }
    }
}

impl FromDXR for DateTime<Utc> {
    fn from_dxr(value: &Value) -> Result<DateTime<Utc>, DxrError> {
        match value.inner() {
            Type::DateTime(date) => Ok(*date),
            t => Err(DxrError::wrong_type(t.name(), "dateTime.iso8861")),
        }
    }
}

impl FromDXR for Vec<u8> {
    fn from_dxr(value: &Value) -> Result<Vec<u8>, DxrError> {
        match value.inner() {
            Type::Base64(bytes) => Ok(bytes.clone()),
            t => Err(DxrError::wrong_type(t.name(), "base64")),
        }
    }
}

#[cfg(feature = "nil")]
#[cfg_attr(docsrs, doc(cfg(feature = "nil")))]
impl<T> FromDXR for Option<T>
where
    T: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Option<T>, DxrError> {
        if let Type::Nil = value.inner() {
            Ok(None)
        } else {
            Ok(Some(T::from_dxr(value)?))
        }
    }
}

impl<T> FromDXR for Cow<'_, T>
where
    T: FromDXR + Clone,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        Ok(Cow::Owned(T::from_dxr(value)?))
    }
}

impl<T> FromDXR for Box<T>
where
    T: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        Ok(Box::new(T::from_dxr(value)?))
    }
}

impl<T> FromDXR for Vec<T>
where
    T: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Vec<T>, DxrError> {
        let values = match value.inner() {
            Type::Array { data } => Ok(data.inner()),
            t => Err(DxrError::wrong_type(t.name(), "array")),
        };

        values?.iter().map(|value| T::from_dxr(value)).collect()
    }
}

impl<T> FromDXR for HashMap<String, T>
where
    T: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<HashMap<String, T>, DxrError> {
        let values = match value.inner() {
            Type::Struct { members } => Ok(members),
            t => Err(DxrError::wrong_type(t.name(), "struct")),
        };

        values?
            .iter()
            .map(|v| {
                let name = v.name().to_string();
                match T::from_dxr(v.inner()) {
                    Ok(value) => Ok((name, value)),
                    Err(error) => Err(error),
                }
            })
            .collect()
    }
}

// some implementations for exact numbers of values (with possibly different types)
impl FromDXR for () {
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        match value.inner() {
            Type::Array { data } => {
                let values = data.inner();

                match values.len() {
                    0 => Ok(()),
                    n => Err(DxrError::parameter_mismatch(n, 0)),
                }
            },
            #[cfg(feature = "nil")]
            Type::Nil => Ok(()),
            other => Err(DxrError::wrong_type(other.name(), "array | nil")),
        }
    }
}

impl<T> FromDXR for (T,)
where
    T: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_1(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B> FromDXR for (A, B)
where
    A: FromDXR,
    B: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_2(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B, C> FromDXR for (A, B, C)
where
    A: FromDXR,
    B: FromDXR,
    C: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_3(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B, C, D> FromDXR for (A, B, C, D)
where
    A: FromDXR,
    B: FromDXR,
    C: FromDXR,
    D: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_4(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B, C, D, E> FromDXR for (A, B, C, D, E)
where
    A: FromDXR,
    B: FromDXR,
    C: FromDXR,
    D: FromDXR,
    E: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_5(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B, C, D, E, F> FromDXR for (A, B, C, D, E, F)
where
    A: FromDXR,
    B: FromDXR,
    C: FromDXR,
    D: FromDXR,
    E: FromDXR,
    F: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_6(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B, C, D, E, F, G> FromDXR for (A, B, C, D, E, F, G)
where
    A: FromDXR,
    B: FromDXR,
    C: FromDXR,
    D: FromDXR,
    E: FromDXR,
    F: FromDXR,
    G: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_7(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

impl<A, B, C, D, E, F, G, H> FromDXR for (A, B, C, D, E, F, G, H)
where
    A: FromDXR,
    B: FromDXR,
    C: FromDXR,
    D: FromDXR,
    E: FromDXR,
    F: FromDXR,
    G: FromDXR,
    H: FromDXR,
{
    fn from_dxr(value: &Value) -> Result<Self, DxrError> {
        if let Type::Array { data } = value.inner() {
            let values = data.inner();
            values_to_tuple_8(values)
        } else {
            Err(DxrError::wrong_type(value.inner().name(), "array"))
        }
    }
}

// if needed, implementations for more arguments can be implemented
