#![feature(test)]
extern crate test;

extern crate dw_datastore;
extern crate dw_models;
extern crate dw_query;

// TODO: Move me to an appropriate place
#[macro_export]
macro_rules! json_map {
    { $( $key:literal : $value:expr),* } => {{
        use serde_json::Value;
        use serde_json::map::Map;
        #[allow(unused_mut)]
        let mut map : Map<String, Value> = Map::new();
        $(
          map.insert( $key.to_string(), json!($value) );
        )*
        map
    }};
}

#[cfg(test)]
mod query_benchmarks {
    use test::Bencher;

    use chrono::Duration;
    use serde_json::json;
    use serde_json::Map;
    use serde_json::Value;

    use dw_datastore::Datastore;
    use dw_models::Bucket;
    use dw_models::BucketMetadata;
    use dw_models::Event;
    use dw_models::TimeInterval;

    static BUCKETNAME: &str = "testbucket";
    static TIME_INTERVAL: &str = "1980-01-01T00:00:00Z/2080-01-02T00:00:00Z";

    fn setup_datastore() -> Datastore {
        return Datastore::new_in_memory(false);
    }

    fn create_bucket(ds: &Datastore, bucketname: String) {
        let bucket = Bucket {
            bid: None,
            id: bucketname,
            _type: "testtype".to_string(),
            client: "testclient".to_string(),
            hostname: "testhost".to_string(),
            created: Some(chrono::Utc::now()),
            data: json_map! {},
            metadata: BucketMetadata::default(),
            events: None,
            last_updated: None,
        };
        ds.create_bucket(&bucket).unwrap();
    }

    fn insert_events(ds: &Datastore, bucketname: &str, num_events: i64) {
        let mut possible_data = Vec::<Map<String, Value>>::new();
        for i in 0..20 {
            possible_data.push(json_map! {"number": i});
        }
        let mut event_list = Vec::new();
        for i in 0..num_events {
            let e = Event {
                id: None,
                timestamp: chrono::Utc::now() + Duration::seconds(i),
                duration: Duration::seconds(10),
                data: possible_data[i as usize % 20].clone(),
            };
            event_list.push(e);
        }
        ds.insert_events(bucketname, &event_list).unwrap();
    }

    #[bench]
    fn bench_assign(b: &mut Bencher) {
        let ds = setup_datastore();
        let interval = TimeInterval::new_from_string(TIME_INTERVAL).unwrap();
        b.iter(|| {
            let code = String::from("return a=1;");
            match dw_query::query(&code, &interval, &ds).unwrap() {
                dw_query::DataType::None() => (),
                ref data => panic!("Wrong datatype, {:?}", data),
            };
        });
    }

    #[bench]
    fn bench_many_events(b: &mut Bencher) {
        let ds = setup_datastore();
        create_bucket(&ds, BUCKETNAME.to_string());
        insert_events(&ds, &BUCKETNAME, 5000);

        let interval = TimeInterval::new_from_string(TIME_INTERVAL).unwrap();
        b.iter(|| {
            let code = String::from(
                "
                events = query_bucket(\"testbucket\");
                return events;
                ",
            );
            dw_query::query(&code, &interval, &ds).unwrap();
        });
    }
}
