
extern crate dvcompute_utils;

use dvcompute_utils::collections::im::OrdMap;

#[test]
fn test_map_empty() {
    let xs = OrdMap::<isize, f64>::empty();
    assert_eq!(xs.is_empty(), true);
}

#[test]
fn test_map_insert_once() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(1, 1.0);
    assert_eq!(xs.collect(), vec![(1, 1.0)]);
}

#[test]
fn test_map_insert_twice() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(1, 1.0);
    assert_eq!(xs.collect(), vec![(1, 1.0), (2, 2.0)]);
}

#[test]
fn test_map_insert_triple() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    assert_eq!(xs.collect(), vec![(1, 1.0), (2, 2.0), (3, 3.0)]);
}

#[test]
fn test_map_insert_negative() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(-5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(-3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(-1, -1.0);
    assert_eq!(xs.collect(), vec![(-5, -5.0), (-3, -3.0), (-1, -1.0),
        (1, 1.0), (2, 2.0), (3, 3.0)]);
}

#[test]
fn test_map_get_existent() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(-5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(-3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(-1, -1.0);
    let y = xs.get(&2);
    assert_eq!(y.is_some() && *(y.unwrap()) == 2.0, true);
}

#[test]
fn test_map_cannon_get_nonexistent() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(-5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(-3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(-1, -1.0);
    let y = xs.get(&10);
    assert_eq!(y.is_none(), true);
}

#[test]
fn test_map_empty_extrema() {
    let xs = OrdMap::<isize, f64>::empty();
    let xy_min = xs.first_key_value();
    let xy_max = xs.last_key_value();
    assert_eq!(xy_min.is_none(), true);
    assert_eq!(xy_max.is_none(), true);
}

#[test]
fn test_map_extrema() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    let xy_min = xs.first_key_value();
    let xy_max = xs.last_key_value();
    assert_eq!(xy_min.map(|xy| (xy.0.clone(), xy.1.clone())), Some((1, 1.0)));
    assert_eq!(xy_max.map(|xy| (xy.0.clone(), xy.1.clone())), Some((3, 3.0)));
}

#[test]
fn test_map_negative_extrema() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(-5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(-3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(-1, -1.0);
    let xy_min = xs.first_key_value();
    let xy_max = xs.last_key_value();
    assert_eq!(xy_min.map(|xy| (xy.0.clone(), xy.1.clone())), Some((-5, -5.0)));
    assert_eq!(xy_max.map(|xy| (xy.0.clone(), xy.1.clone())), Some((3, 3.0)));
}

#[test]
fn test_map_balance() {
    let n = 100000;
    let mut xs = OrdMap::<isize, f64>::empty();
    for x in 0 .. n {
        let y = x as f64;
        xs = xs.insert(x, y);
    }
    assert_eq!(xs.depth() as f64 <= 2.0 * (xs.size() as f64).log2() + 1.0, true);
}

#[test]
fn test_map_cut_traverse() {
    let mut xs = OrdMap::<isize, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(-5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(-3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(-1, -1.0);

    let mut vs = Vec::new();
    let mut f = |k: &isize, v: &f64| {
        if *k > 1 {
            Some(*k)
        } else {
            vs.push((*k, *v));
            None
        }
    };

    let k = xs.find(&mut f);

    assert_eq!(k, Some(2));
    assert_eq!(vs, vec![(-5, -5.0), (-3, -3.0), (-1, -1.0), (1, 1.0)]);
}
