
extern crate dvcompute_utils;

use dvcompute_utils::collections::im::IntMap;

#[test]
fn test_map_empty() {
    let xs = IntMap::<u64, f64>::empty();
    assert_eq!(xs.is_empty(), true);
}

#[test]
fn test_map_insert_once() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(1, 1.0);
    assert_eq!(xs.collect(), vec![(1, 1.0)]);
}

#[test]
fn test_map_insert_twice() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(1, 1.0);
    assert_eq!(xs.collect(), vec![(1, 1.0), (2, 2.0)]);
}

#[test]
fn test_map_insert_triple() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    assert_eq!(xs.collect(), vec![(1, 1.0), (2, 2.0), (3, 3.0)]);
}

#[test]
fn test_map_remove_empty() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.remove(&2);
    assert_eq!(xs.collect(), vec![]);
}

#[test]
fn test_map_remove_once() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.remove(&2);
    assert_eq!(xs.collect(), vec![(1, 1.0), (3, 3.0)]);
}

#[test]
fn test_map_remove_twice() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.remove(&2);
    xs = xs.remove(&100);
    assert_eq!(xs.collect(), vec![(1, 1.0), (3, 3.0)]);
}

#[test]
fn test_map_remove_twice_again() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.remove(&2);
    xs = xs.remove(&3);
    assert_eq!(xs.collect(), vec![(1, 1.0)]);
}

#[test]
fn test_map_remove_triple() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.remove(&2);
    xs = xs.remove(&3);
    xs = xs.remove(&1);
    assert_eq!(xs.collect(), vec![]);
}

#[test]
fn test_map_get_existent() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(1, -1.0);
    let y = xs.get(&2);
    assert_eq!(y.is_some() && *(y.unwrap()) == 2.0, true);
}

#[test]
fn test_map_cannon_get_nonexistent() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(5, 5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(1, -1.0);
    let y = xs.get(&10);
    assert_eq!(y.is_none(), true);
}

#[test]
fn test_map_empty_extrema() {
    let xs = IntMap::<u64, f64>::empty();
    let xy_min = xs.first_key_value();
    let xy_max = xs.last_key_value();
    assert_eq!(xy_min.is_none(), true);
    assert_eq!(xy_max.is_none(), true);
}

#[test]
fn test_map_extrema() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(1, 1.0);
    let xy_min = xs.first_key_value();
    let xy_max = xs.last_key_value();
    assert_eq!(xy_min.map(|xy| (xy.0.clone(), xy.1.clone())), Some((1, 1.0)));
    assert_eq!(xy_max.map(|xy| (xy.0.clone(), xy.1.clone())), Some((3, 3.0)));
}

#[test]
fn test_map_stress() {
    let mut xs = IntMap::<u64, f64>::empty();
    for x in 0 .. 1000 {
        let y = x as f64;
        xs = xs.insert(x, y);
    }
    for x in 0 .. 1000 {
        if x != 5 {
            xs = xs.remove(&x);
        }
    }
    assert_eq!(xs.collect(), vec![(5, 5.0)]);
}

#[test]
fn test_map_cut_traverse() {
    let mut xs = IntMap::<u64, f64>::empty();
    xs = xs.insert(2, 2.0);
    xs = xs.insert(5, -5.0);
    xs = xs.insert(3, 3.0);
    xs = xs.insert(3, -3.0);
    xs = xs.insert(1, 1.0);
    xs = xs.insert(1, -1.0);

    let mut vs = Vec::new();
    let mut f = |k: &u64, v: &f64| {
        if *k > 2 {
            Some(*k)
        } else {
            vs.push((*k, *v));
            None
        }
    };

    let k = xs.find(&mut f);

    assert_eq!(k, Some(3));
    assert_eq!(vs, vec![(1, -1.0), (2, 2.0)]);
}
