// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::f32;
use std::f64;
use std::i32;
use std::i64;
use std::isize;

/// Specifies a data type from which values we can gather the statistics.
pub trait StatData where Self: Copy + PartialOrd {

    /// Return the minimum possible value.
    fn min() -> Self;

    /// Return the minimum possible value.
    fn max() -> Self;

    /// Return a whatever value which can be considered as the current value.
    fn any() -> Self;

    /// Convert a sample into f64.
    fn into_f64(sample: Self) -> f64;
}

impl StatData for i32 {

    #[inline]
    fn min() -> i32 {
        i32::MIN
    }

    #[inline]
    fn max() -> i32 {
        i32::MAX
    }

    #[inline]
    fn any() -> i32 {
        0
    }

    #[inline]
    fn into_f64(sample: i32) -> f64 {
        sample as f64
    }
}

impl StatData for i64 {

    #[inline]
    fn min() -> i64 {
        i64::MIN
    }

    #[inline]
    fn max() -> i64 {
        i64::MAX
    }

    #[inline]
    fn any() -> i64 {
        0
    }

    #[inline]
    fn into_f64(sample: i64) -> f64 {
        sample as f64
    }
}

impl StatData for isize {

    #[inline]
    fn min() -> isize {
        isize::MIN
    }

    #[inline]
    fn max() -> isize {
        isize::MAX
    }

    #[inline]
    fn any() -> isize {
        0
    }

    #[inline]
    fn into_f64(sample: isize) -> f64 {
        sample as f64
    }
}

impl StatData for f32 {

    #[inline]
    fn min() -> f32 {
        f32::NEG_INFINITY
    }

    #[inline]
    fn max() -> f32 {
        f32::INFINITY
    }

    #[inline]
    fn any() -> f32 {
        f32::NAN
    }

    #[inline]
    fn into_f64(sample: f32) -> f64 {
        sample as f64
    }
}

impl StatData for f64 {

    #[inline]
    fn min() -> f64 {
        f64::NEG_INFINITY
    }

    #[inline]
    fn max() -> f64 {
        f64::INFINITY
    }

    #[inline]
    fn any() -> f64 {
        f64::NAN
    }

    #[inline]
    fn into_f64(sample: f64) -> f64 {
        sample
    }
}
