// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::result;
use std::ops::Deref;
use std::cell::RefCell;

use dvcompute::simulation;
use dvcompute::simulation::Run;
use dvcompute::simulation::simulation::*;
use dvcompute::simulation::simulation::ops::*;
use dvcompute::simulation::parameter::*;
use dvcompute::simulation::parameter::ops::*;
use dvcompute::simulation::event::*;
use dvcompute::simulation::event::ops::*;
use dvcompute::simulation::observable::*;
use dvcompute::simulation::observable::source::*;
use dvcompute::simulation::ref_comp::*;
use dvcompute::simulation::queue;
use dvcompute::simulation::strategy::*;

use dvcompute_gpss::simulation::queue::*;
use dvcompute_gpss::simulation::facility::*;
use dvcompute_gpss::simulation::storage::*;

use dvcompute_utils::simulation::stats::*;
use dvcompute_utils::collections::im::OrdMap;

use self::locale::*;

/// Result localization.
pub mod locale;

/// Input / Output for results.
pub mod io;

/// An error that may occur when receiving the simulation results.
pub type ResultError = String;

/// The result of receiving the simulation data.
pub type Result<T> = result::Result<T, ResultError>;

/// Represents a provider of the simulation results. It is usually something, or
/// a container of something which can be simulated to get data.
pub trait ResultProvider {

    /// Return the source of simulation results by the specified provider, name and description.
    fn get_source(self, name: ResultName, descr: ResultDescription) -> ResultSource
        where Self: Sized
    {
        let name_path = vec![name.clone()];
        let title = result_name_into_title(name.clone());
        let id = ResultId::UserDefined(UserDefinedResult { name: name.clone(), descr, title });
        let id_path = vec![id.clone()];

        self.get_source_by_id(name, name_path, id, id_path)
    }

    /// Return the source of simulation results by the specified provider, name, description and title.
    fn get_source3(self, name: ResultName, descr: ResultDescription, title: ResultDescription) -> ResultSource
        where Self: Sized
    {
        let name_path = vec![name.clone()];
        let id = ResultId::UserDefined(UserDefinedResult { name: name.clone(), descr, title });
        let id_path = vec![id.clone()];

        self.get_source_by_id(name, name_path, id, id_path)
    }

    /// Return the source of simulation results by the specified provider, name, its name path,
    /// identifier and the corresponding identifier path.
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource;
}

/// It associates the result sources with their names.
pub type ResultSourceMap = OrdMap<ResultName, ResultSource>;

/// The result source.
#[derive(Clone)]
pub enum ResultSource {

    /// The source consisting of a single item.
    Item(Rc<dyn ResultItem>),

    /// An object-like source.
    Object(Rc<ResultObject>),

    /// A vector-like source.
    Vec(Rc<ResultVec>),

    /// This is a separator text.
    Separator(Rc<ResultSeparator>)
}

impl ResultSource {

    /// Return a summarised and usually more short version of the result source by expanding
    /// the main properties or excluding auxiliary properties if required.
    pub fn summary(&self) -> ResultSource {
        match &self {
            &ResultSource::Item(ref x) => x.summary(),
            &ResultSource::Object(ref x) => (x.summary)(),
            &ResultSource::Vec(ref x) => (x.summary)(),
            &ResultSource::Separator(x) => ResultSource::Separator(x.clone())
        }
    }

    /// Return a signal emitted by the source.
    pub fn observable(&self) -> ResultObservable {
        match &self {
            &ResultSource::Item(ref x) => x.observable(),
            &ResultSource::Object(ref x) => (x.observable)(),
            &ResultSource::Vec(ref x) => (x.observable)(),
            &ResultSource::Separator(_) => ResultObservable::empty()
        }
    }

    /// Flatten the result source.
    pub fn flatten(&self) -> Vec<Rc<dyn ResultItem>> {
        match self {
            &ResultSource::Item(ref x) => {
                vec![x.clone()]
            },
            &ResultSource::Object(ref x) => {
                let mut ys = Vec::new();
                for p in x.props.iter() {
                    let mut zs = p.source.flatten();
                    ys.append(&mut zs);
                }
                ys
            },
            &ResultSource::Vec(ref x) => {
                let mut ys = Vec::new();
                for i in x.items.iter() {
                    let mut zs = i.flatten();
                    ys.append(&mut zs);
                }
                ys
            },
            &ResultSource::Separator(_) => {
                vec![]
            }
        }
    }

    /// Return the result source name.
    pub fn get_name(&self) -> ResultName {
        match self {
            &ResultSource::Item(ref x) => x.get_name().clone(),
            &ResultSource::Object(ref x) => x.name.clone(),
            &ResultSource::Vec(ref x) => x.name.clone(),
            &ResultSource::Separator(_) => String::new()
        }
    }

    /// Expand the result source by returning a more detailed version while expanding the properties as possible.
    pub fn expand(&self) -> ResultSource {
        match self {
            &ResultSource::Item(ref x) => x.expand(),
            &ResultSource::Object(ref x) => {
                let mut ys = Vec::with_capacity(x.props.len());
                for p in x.props.iter() {
                    let p2 = ResultProperty {
                        label: p.label.clone(),
                        id: p.id.clone(),
                        source: p.source.expand()
                    };
                    ys.push(p2);
                }
                let z = ResultObject {
                    name: x.name.clone(),
                    id: x.id.clone(),
                    type_id: x.type_id.clone(),
                    props: ys,
                    observable: x.observable.clone(),
                    summary: x.summary.clone()
                };
                ResultSource::Object(Rc::new(z))
            },
            &ResultSource::Vec(ref x) => {
                let mut ys = Vec::with_capacity(x.items.len());
                for i in x.items.iter() {
                    ys.push(i.expand());
                }
                let z = ResultVec {
                    name: x.name.clone(),
                    id: x.id.clone(),
                    items: ys,
                    subscript: x.subscript.clone(),
                    observable: x.observable.clone(),
                    summary: x.summary.clone()
                };
                ResultSource::Vec(Rc::new(z))
            },
            &ResultSource::Separator(ref x) => {
                let z = ResultSeparator { text: x.text.clone() };
                ResultSource::Separator(Rc::new(z))
            }
        }
    }

    /// Represent the result source as integer numbers.
    pub fn get_int_vals(&self) -> Result<Vec<ResultValue<isize>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_int_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as vectors of integer numbers.
    pub fn get_int_vec_vals(&self) -> Result<Vec<ResultValue<Vec<isize>>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_int_vec_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as statistics based on integer numbers.
    pub fn get_int_stats_vals(&self) -> Result<Vec<ResultValue<SamplingStats<isize>>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_int_stats_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as time-persistent statistics based on integer numbers.
    pub fn get_int_timing_stats_vals(&self) -> Result<Vec<ResultValue<TimingStats<isize>>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_int_timing_stats_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as floating point numbers.
    pub fn get_floating_vals(&self) -> Result<Vec<ResultValue<f64>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_floating_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as vectors of floating point numbers.
    pub fn get_floating_vec_vals(&self) -> Result<Vec<ResultValue<Vec<f64>>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_floating_vec_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as statistics based on floating point numbers.
    pub fn get_floating_stats_vals(&self) -> Result<Vec<ResultValue<SamplingStats<f64>>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_floating_stats_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as time-persistent statistics based on floating point numbers.
    pub fn get_floating_timing_stats_vals(&self) -> Result<Vec<ResultValue<TimingStats<f64>>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_floating_timing_stats_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result source as `String` values.
    pub fn get_string_vals(&self) -> Result<Vec<ResultValue<String>>> {
        let xs = self.flatten();
        let mut ys = Vec::with_capacity(xs.len());
        for x in xs {
            ys.push(x.get_string_val()?);
        }
        result::Result::Ok(ys)
    }

    /// Return a source by the specified statistics.
    pub fn from_stats<E>(cont: ResultValue<SamplingStats<E>>) -> ResultSource
        where ResultValue<E>: ResultItem,
              ResultValue<SamplingStats<E>>: ResultItem,
              E: StatData + Copy + Clone + 'static
    {
        let props = vec![
            cont.clone().map_prop("count", ResultId::SamplingStatsCount, move |x| { x.count as isize }),
            cont.clone().map_prop("mean", ResultId::SamplingStatsMean, move |x| { x.mean }),
            cont.clone().map_prop("mean2", ResultId::SamplingStatsMean2, move |x| { x.mean2 }),
            cont.clone().map_prop("std", ResultId::SamplingStatsDeviation, move |x| { x.deviation() }),
            cont.clone().map_prop("var", ResultId::SamplingStatsVariance, move |x| { x.variance() }),
            cont.clone().map_prop("min", ResultId::SamplingStatsMin, move |x| { x.min }),
            cont.clone().map_prop("max", ResultId::SamplingStatsMax, move |x| { x.max })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::SamplingStats,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_stats(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a source by the specified statistics.
    pub fn summary_from_stats<E>(cont: ResultValue<SamplingStats<E>>) -> ResultSource
        where ResultValue<E>: ResultItem,
              ResultValue<SamplingStats<E>>: ResultItem,
              E: Clone + 'static
    {
        ResultSource::Item(Rc::new(cont.get_string_val().unwrap()))
    }

    /// Return a source by the specified time persisten variable statistics.
    pub fn from_timing_stats<E>(cont: ResultValue<TimingStats<E>>) -> ResultSource
        where ResultValue<E>: ResultItem,
              ResultValue<TimingStats<E>>: ResultItem,
              E: StatData + Copy + Clone + 'static
    {
        let props = vec![
            cont.clone().map_prop("count", ResultId::TimingStatsCount, move |x| { x.count as isize }),
            cont.clone().map_prop("mean", ResultId::TimingStatsMean, move |x| { x.mean() }),
            cont.clone().map_prop("std", ResultId::TimingStatsDeviation, move |x| { x.deviation() }),
            cont.clone().map_prop("var", ResultId::TimingStatsVariance, move |x| { x.variance() }),
            cont.clone().map_prop("min", ResultId::TimingStatsMin, move |x| { x.min }),
            cont.clone().map_prop("max", ResultId::TimingStatsMax, move |x| { x.max }),
            cont.clone().map_prop("min_time", ResultId::TimingStatsMinTime, move |x| { x.min_time }),
            cont.clone().map_prop("max_time", ResultId::TimingStatsMaxTime, move |x| { x.max_time }),
            cont.clone().map_prop("start_time", ResultId::TimingStatsStartTime, move |x| { x.start_time }),
            cont.clone().map_prop("last_time", ResultId::TimingStatsLastTime, move |x| { x.last_time }),
            cont.clone().map_prop("sum", ResultId::TimingStatsSum, move |x| { x.sum }),
            cont.clone().map_prop("sum2", ResultId::TimingStatsSum2, move |x| { x.sum2 })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::TimingStats,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_timing_stats(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a source by the specified time persistent variable statistics.
    pub fn summary_from_timing_stats<E>(cont: ResultValue<TimingStats<E>>) -> ResultSource
        where ResultValue<E>: ResultItem,
              ResultValue<TimingStats<E>>: ResultItem,
              E: Clone + 'static
    {
        ResultSource::Item(Rc::new(cont.get_string_val().unwrap()))
    }

    /// Return a source by the specified queue.
    pub fn from_queue(cont: ResultContainer<Queue>) -> ResultSource {
        let props = vec![
            cont.clone().prop("queue_empty",
                ResultId::QueueEmpty,
                move |x| { Queue::is_empty(x) },
                move |x| { Queue::content_changed_(&x) }),
            cont.clone().prop("queue_content",
                ResultId::QueueContent,
                move |x| { Queue::content(x) },
                move |x| { Queue::content_changed_(&x) }),
            cont.clone().prop("queue_content_stats",
                ResultId::QueueContentStats,
                move |x| { Queue::content_stats(x) },
                move |x| { Queue::content_changed_(&x) }),
            cont.clone().prop("enqueue_count",
                ResultId::EnqueueCount,
                move |x| { Queue::enqueue_count(x) },
                move |x| { Queue::enqueue_count_changed_(&x) }),
            cont.clone().prop("enqueue_zero_enty_count",
                ResultId::EnqueueZeroEntryCount,
                move |x| { Queue::enqueue_zero_entry_count(x) },
                move |x| { Queue::enqueue_zero_entry_count_changed_(&x) }),
            cont.clone().prop("queue_wait_time",
                ResultId::QueueWaitTime,
                move |x| { Queue::wait_time(x) },
                move |x| { Queue::wait_time_changed_(&x) }),
            cont.clone().prop("queue_non_zero_entry_wait_time",
                ResultId::QueueNotZeroEntryWaitTime,
                move |x| { Queue::non_zero_entry_wait_time(x) },
                move |x| { Queue::non_zero_entry_wait_time_changed_(&x) }),
            cont.clone().prop("queue_rate",
                ResultId::QueueRate,
                move |x| { Queue::rate(x) },
                move |x| { Queue::rate_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Queue,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_queue(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a summary source by the specified queue.
    pub fn summary_from_queue(cont: ResultContainer<Queue>) -> ResultSource {
        let props = vec![
            cont.clone().prop("queue_content_stats",
                ResultId::QueueContentStats,
                move |x| { Queue::content_stats(x) },
                move |x| { Queue::content_changed_(&x) }),
            cont.clone().prop("enqueue_count",
                ResultId::EnqueueCount,
                move |x| { Queue::enqueue_count(x) },
                move |x| { Queue::enqueue_count_changed_(&x) }),
            cont.clone().prop("enqueue_zero_enty_count",
                ResultId::EnqueueZeroEntryCount,
                move |x| { Queue::enqueue_zero_entry_count(x) },
                move |x| { Queue::enqueue_zero_entry_count_changed_(&x) }),
            cont.clone().prop("queue_wait_time",
                ResultId::QueueWaitTime,
                move |x| { Queue::wait_time(x) },
                move |x| { Queue::wait_time_changed_(&x) }),
            cont.clone().prop("queue_non_zero_entry_wait_time",
                ResultId::QueueNotZeroEntryWaitTime,
                move |x| { Queue::non_zero_entry_wait_time(x) },
                move |x| { Queue::non_zero_entry_wait_time_changed_(&x) }),
            cont.clone().prop("queue_rate",
                ResultId::QueueRate,
                move |x| { Queue::rate(x) },
                move |x| { Queue::rate_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Queue,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_queue(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a source by the specified facility.
    pub fn from_facility<T: 'static>(cont: ResultContainer<Facility<T>>) -> ResultSource {
        let props = vec![
            cont.clone().prop("queue_count",
                ResultId::FacilityQueueCount,
                move |x| { Facility::queue_count(x) },
                move |x| { Facility::queue_count_changed_(&x) }),
            cont.clone().prop("queue_count_stats",
                ResultId::FacilityQueueCountStats,
                move |x| { Facility::queue_count_stats(x) },
                move |x| { Facility::queue_count_changed_(&x) }),
            cont.clone().prop("total_wait_time",
                ResultId::FacilityTotalWaitTime,
                move |x| { Facility::total_wait_time(x) },
                move |x| { Facility::wait_time_changed_(&x) }),
            cont.clone().prop("wait_time",
                ResultId::FacilityWaitTime,
                move |x| { Facility::wait_time(x) },
                move |x| { Facility::wait_time_changed_(&x) }),
            cont.clone().prop("total_holding_time",
                ResultId::FacilityTotalHoldingTime,
                move |x| { Facility::total_holding_time(x) },
                move |x| { Facility::holding_time_changed_(&x) }),
            cont.clone().prop("holding_time",
                ResultId::FacilityHoldingTime,
                move |x| { Facility::holding_time(x) },
                move |x| { Facility::holding_time_changed_(&x) }),
            cont.clone().integ_prop("interrupted",
                ResultId::FacilityInterrupted,
                move |x| { Facility::is_interrupted(x) }),
            cont.clone().prop("count",
                ResultId::FacilityCount,
                move |x| { Facility::count(x) },
                move |x| { Facility::count_changed_(&x) }),
            cont.clone().prop("count_stats",
                ResultId::FacilityCountStats,
                move |x| { Facility::count_stats(x) },
                move |x| { Facility::count_changed_(&x) }),
            cont.clone().prop("capture_count",
                ResultId::FacilityCaptureCount,
                move |x| { Facility::capture_count(x) },
                move |x| { Facility::capture_count_changed_(&x) }),
            cont.clone().prop("util_count",
                ResultId::FacilityUtilCount,
                move |x| { Facility::util_count(x) },
                move |x| { Facility::util_count_changed_(&x) }),
            cont.clone().prop("util_count_stats",
                ResultId::FacilityUtilCountStats,
                move |x| { Facility::util_count_stats(x) },
                move |x| { Facility::util_count_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Facility,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_facility(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a summary source by the specified facility.
    pub fn summary_from_facility<T: 'static>(cont: ResultContainer<Facility<T>>) -> ResultSource {
        let props = vec![
            cont.clone().prop("queue_count_stats",
                ResultId::FacilityQueueCountStats,
                move |x| { Facility::queue_count_stats(x) },
                move |x| { Facility::queue_count_changed_(&x) }),
            cont.clone().prop("wait_time",
                ResultId::FacilityWaitTime,
                move |x| { Facility::wait_time(x) },
                move |x| { Facility::wait_time_changed_(&x) }),
            cont.clone().prop("holding_time",
                ResultId::FacilityHoldingTime,
                move |x| { Facility::holding_time(x) },
                move |x| { Facility::holding_time_changed_(&x) }),
            cont.clone().prop("count_stats",
                ResultId::FacilityCountStats,
                move |x| { Facility::count_stats(x) },
                move |x| { Facility::count_changed_(&x) }),
            cont.clone().prop("capture_count",
                ResultId::FacilityCaptureCount,
                move |x| { Facility::capture_count(x) },
                move |x| { Facility::capture_count_changed_(&x) }),
            cont.clone().prop("util_count_stats",
                ResultId::FacilityUtilCountStats,
                move |x| { Facility::util_count_stats(x) },
                move |x| { Facility::util_count_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Facility,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_facility(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a source by the specified storage.
    pub fn from_storage<T: 'static>(cont: ResultContainer<Storage<T>>) -> ResultSource {
        let props = vec![
            cont.clone().const_prop("capacity",
                ResultId::StorageCapacity,
                move |x| { x.capacity() }),
            cont.clone().integ_prop("empty",
                ResultId::StorageEmpty,
                move |x| { Storage::is_empty(x) }),
            cont.clone().integ_prop("full",
                ResultId::StorageFull,
                move |x| { Storage::is_full(x) }),
            cont.clone().prop("queue_count",
                ResultId::StorageQueueCount,
                move |x| { Storage::queue_count(x) },
                move |x| { Storage::queue_count_changed_(&x) }),
            cont.clone().prop("queue_count_stats",
                ResultId::StorageQueueCountStats,
                move |x| { Storage::queue_count_stats(x) },
                move |x| { Storage::queue_count_changed_(&x) }),
            cont.clone().prop("total_wait_time",
                ResultId::StorageTotalWaitTime,
                move |x| { Storage::total_wait_time(x) },
                move |x| { Storage::wait_time_changed_(&x) }),
            cont.clone().prop("wait_time",
                ResultId::StorageWaitTime,
                move |x| { Storage::wait_time(x) },
                move |x| { Storage::wait_time_changed_(&x) }),
            cont.clone().integ_prop("average_holding_time",
                ResultId::StorageAverageHoldingTime,
                move |x| { Storage::average_holding_time(x) }),
            cont.clone().prop("content",
                ResultId::StorageContent,
                move |x| { Storage::content(x) },
                move |x| { Storage::content_changed_(&x) }),
            cont.clone().prop("content_stats",
                ResultId::StorageContentStats,
                move |x| { Storage::content_stats(x) },
                move |x| { Storage::content_changed_(&x) }),
            cont.clone().prop("use_count",
                ResultId::StorageUseCount,
                move |x| { Storage::use_count(x) },
                move |x| { Storage::use_count_changed_(&x) }),
            cont.clone().prop("used_content",
                ResultId::StorageUsedContent,
                move |x| { Storage::used_content(x) },
                move |x| { Storage::used_content_changed_(&x) }),
            cont.clone().prop("util_count",
                ResultId::StorageUtilCount,
                move |x| { Storage::util_count(x) },
                move |x| { Storage::util_count_changed_(&x) }),
            cont.clone().prop("util_count_stats",
                ResultId::StorageUtilCountStats,
                move |x| { Storage::util_count_stats(x) },
                move |x| { Storage::util_count_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Storage,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_storage(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a summary source by the specified storage.
    pub fn summary_from_storage<T: 'static>(cont: ResultContainer<Storage<T>>) -> ResultSource {
        let props = vec![
            cont.clone().const_prop("capacity",
                ResultId::StorageCapacity,
                move |x| { x.capacity() }),
            cont.clone().prop("queue_count_stats",
                ResultId::StorageQueueCountStats,
                move |x| { Storage::queue_count_stats(x) },
                move |x| { Storage::queue_count_changed_(&x) }),
            cont.clone().prop("wait_time",
                ResultId::StorageWaitTime,
                move |x| { Storage::wait_time(x) },
                move |x| { Storage::wait_time_changed_(&x) }),
            cont.clone().integ_prop("average_holding_time",
                ResultId::StorageAverageHoldingTime,
                move |x| { Storage::average_holding_time(x) }),
            cont.clone().prop("content_stats",
                ResultId::StorageContentStats,
                move |x| { Storage::content_stats(x) },
                move |x| { Storage::content_changed_(&x) }),
            cont.clone().prop("use_count",
                ResultId::StorageUseCount,
                move |x| { Storage::use_count(x) },
                move |x| { Storage::use_count_changed_(&x) }),
            cont.clone().prop("used_content",
                ResultId::StorageUsedContent,
                move |x| { Storage::used_content(x) },
                move |x| { Storage::used_content_changed_(&x) }),
            cont.clone().prop("util_count_stats",
                ResultId::StorageUtilCountStats,
                move |x| { Storage::util_count_stats(x) },
                move |x| { Storage::util_count_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Storage,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_storage(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a source by the specified bounded queue.
    pub fn from_bounded_queue<SI, SM, SO, T>(cont: ResultContainer<queue::stats::Queue<SI, SM, SO, T>>) -> ResultSource
        where SI: QueueStrategy + 'static,
              SM: QueueStrategy + 'static,
              SO: QueueStrategy + 'static,
              T: Clone + 'static
    {
        let props = vec![
            cont.clone().prop("queue_empty",
                ResultId::QueueEmpty,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::is_empty(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::is_empty_changed_(&x) }),
            cont.clone().prop("queue_full",
                ResultId::QueueFull,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::is_full(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::is_full_changed_(&x) }),
            cont.clone().const_prop("capacity",
                ResultId::QueueMaxCount,
                move |x| { x.max_count() }),
            cont.clone().prop("queue_count",
                ResultId::QueueCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::count_changed_(&x) }),
            cont.clone().prop("queue_count_stats",
                ResultId::QueueCountStats,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::count_stats(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::count_changed_(&x) }),
            cont.clone().prop("enqueue_count",
                ResultId::EnqueueCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_count_changed_(&x) }),
            cont.clone().prop("enqueue_lost_count",
                ResultId::EnqueueLostCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_lost_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_lost_count_changed_(&x) }),
            cont.clone().prop("enqueue_store_count",
                ResultId::EnqueueStoreCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_store_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_store_count_changed_(&x) }),
            cont.clone().prop("dequeue_count",
                ResultId::DequeueCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_count_changed_(&x) }),
            cont.clone().prop("dequeue_extract_count",
                ResultId::DequeueExtractCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_extract_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_extract_count_changed_(&x) }),
            cont.clone().prop("queue_load_factor",
                ResultId::QueueLoadFactor,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::load_factor(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::load_factor_changed_(&x) }),
            cont.clone().integ_prop("enqueue_rate",
                ResultId::EnqueueRate,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_rate(x) }),
            cont.clone().integ_prop("enqueue_store_rate",
                ResultId::EnqueueStoreRate,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::store_rate(x) }),
            cont.clone().integ_prop("dequeue_rate",
                ResultId::DequeueRate,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_rate(x) }),
            cont.clone().integ_prop("dequeue_extract_rate",
                ResultId::DequeueExtractRate,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_extract_rate(x) }),
            cont.clone().prop("queue_wait_time",
                ResultId::QueueWaitTime,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::wait_time(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::wait_time_changed_(&x) }),
            cont.clone().prop("queue_total_wait_time",
                ResultId::QueueTotalWaitTime,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::total_wait_time(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::total_wait_time_changed_(&x) }),
            cont.clone().prop("enqueue_wait_time",
                ResultId::EnqueueWaitTime,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_wait_time(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_wait_time_changed_(&x) }),
            cont.clone().prop("dequeue_wait_time",
                ResultId::DequeueWaitTime,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_wait_time(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_wait_time_changed_(&x) }),
            cont.clone().prop("queue_rate",
                ResultId::QueueRate,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::rate(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::rate_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Queue,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_bounded_queue(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return the summary by the specified bounded queue.
    pub fn summary_from_bounded_queue<SI, SM, SO, T>(cont: ResultContainer<queue::stats::Queue<SI, SM, SO, T>>) -> ResultSource
        where SI: QueueStrategy + 'static,
              SM: QueueStrategy + 'static,
              SO: QueueStrategy + 'static,
              T: Clone + 'static
    {
        let props = vec![
            cont.clone().const_prop("capacity",
                ResultId::QueueMaxCount,
                move |x| { x.max_count() }),
            cont.clone().prop("queue_count_stats",
                ResultId::QueueCountStats,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::count_stats(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::count_changed_(&x) }),
            cont.clone().prop("enqueue_count",
                ResultId::EnqueueCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_count_changed_(&x) }),
            cont.clone().prop("enqueue_lost_count",
                ResultId::EnqueueLostCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_lost_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_lost_count_changed_(&x) }),
            cont.clone().prop("enqueue_store_count",
                ResultId::EnqueueStoreCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_store_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::enqueue_store_count_changed_(&x) }),
            cont.clone().prop("dequeue_count",
                ResultId::DequeueCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_count_changed_(&x) }),
            cont.clone().prop("dequeue_extract_count",
                ResultId::DequeueExtractCount,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_extract_count(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::dequeue_extract_count_changed_(&x) }),
            cont.clone().prop("queue_load_factor",
                ResultId::QueueLoadFactor,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::load_factor(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::load_factor_changed_(&x) }),
            cont.clone().prop("queue_wait_time",
                ResultId::QueueWaitTime,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::wait_time(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::wait_time_changed_(&x) }),
            cont.clone().prop("queue_rate",
                ResultId::QueueRate,
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::rate(x) },
                move |x| { queue::stats::Queue::<SI, SM, SO, T>::rate_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Queue,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_bounded_queue(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return a source by the specified unbounded queue.
    pub fn from_unbounded_queue<SM, SO, T>(cont: ResultContainer<queue::unbounded::stats::Queue<SM, SO, T>>) -> ResultSource
        where SM: QueueStrategy + 'static,
              SO: QueueStrategy + 'static,
              T: Clone + 'static
    {
        let props = vec![
            cont.clone().prop("queue_empty",
                ResultId::QueueEmpty,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::is_empty(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::is_empty_changed_(&x) }),
            cont.clone().prop("queue_count",
                ResultId::QueueCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::count_changed_(&x) }),
            cont.clone().prop("queue_count_stats",
                ResultId::QueueCountStats,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::count_stats(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::count_changed_(&x) }),
            cont.clone().prop("enqueue_store_count",
                ResultId::EnqueueStoreCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::enqueue_store_count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::enqueue_store_count_changed_(&x) }),
            cont.clone().prop("dequeue_count",
                ResultId::DequeueCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_count_changed_(&x) }),
            cont.clone().prop("dequeue_extract_count",
                ResultId::DequeueExtractCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_extract_count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_extract_count_changed_(&x) }),
            cont.clone().integ_prop("enqueue_store_rate",
                ResultId::EnqueueStoreRate,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::store_rate(x) }),
            cont.clone().integ_prop("dequeue_rate",
                ResultId::DequeueRate,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_rate(x) }),
            cont.clone().integ_prop("dequeue_extract_rate",
                ResultId::DequeueExtractRate,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_extract_rate(x) }),
            cont.clone().prop("queue_wait_time",
                ResultId::QueueWaitTime,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::wait_time(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::wait_time_changed_(&x) }),
            cont.clone().prop("dequeue_wait_time",
                ResultId::DequeueWaitTime,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_wait_time(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_wait_time_changed_(&x) }),
            cont.clone().prop("queue_rate",
                ResultId::QueueRate,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::rate(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::rate_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Queue,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_unbounded_queue(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return the summary by the specified unbounded queue.
    pub fn summary_from_unbounded_queue<SM, SO, T>(cont: ResultContainer<queue::unbounded::stats::Queue<SM, SO, T>>) -> ResultSource
        where SM: QueueStrategy + 'static,
              SO: QueueStrategy + 'static,
              T: Clone + 'static
    {
        let props = vec![
            cont.clone().prop("queue_count_stats",
                ResultId::QueueCountStats,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::count_stats(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::count_changed_(&x) }),
            cont.clone().prop("enqueue_store_count",
                ResultId::EnqueueStoreCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::enqueue_store_count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::enqueue_store_count_changed_(&x) }),
            cont.clone().prop("dequeue_count",
                ResultId::DequeueCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_count_changed_(&x) }),
            cont.clone().prop("dequeue_extract_count",
                ResultId::DequeueExtractCount,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_extract_count(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::dequeue_extract_count_changed_(&x) }),
            cont.clone().prop("queue_wait_time",
                ResultId::QueueWaitTime,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::wait_time(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::wait_time_changed_(&x) }),
            cont.clone().prop("queue_rate",
                ResultId::QueueRate,
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::rate(x) },
                move |x| { queue::unbounded::stats::Queue::<SM, SO, T>::rate_changed_(&x) })
        ];
        let source = ResultObject {
            name: cont.name.clone(),
            id: cont.id.clone(),
            type_id: ResultId::Queue,
            observable: cont.observable.clone(),
            summary: Rc::new(move || { ResultSource::summary_from_unbounded_queue(cont.clone()) }),
            props: props
        };
        let source = Rc::new(source);
        ResultSource::Object(source)
    }

    /// Return an arbitrary text as a separator source.
    pub fn from_text(text: String) -> ResultSource {
        let source = ResultSeparator { text: text };
        let source = Rc::new(source);
        ResultSource::Separator(source)
    }

    /// Return the time result source.
    pub fn time() -> ResultSource {
        let source = EventFn::new(move || { time_event() });
        let name = String::from("t");
        let name_path = vec![name.clone()];
        let id = ResultId::Time;
        let id_path = vec![id.clone()];
        source.get_source_by_id(name, name_path, id, id_path)
    }
}

/// Represents a trait for actual representation of the item.
pub trait ResultItem {

    /// Return the item name.
    fn get_name(&self) -> ResultName;

    /// Return the item name path.
    fn get_name_path(&self) -> Vec<ResultName>;

    /// Return the item identifier.
    fn get_id(&self) -> ResultId;

    /// Return the item identifier path.
    fn get_id_path(&self) -> Vec<ResultId>;

    /// Whether the item emits a signal.
    fn observable(&self) -> ResultObservable;

    /// Return an expanded version of the item, for example,
    /// when the statistics item is exanded to an object
    /// having the corresponded properties for count, average,
    /// deviation, minimum, maximum and so on.
    fn expand(&self) -> ResultSource;

    /// Return usually a short version of the item, i.e. its summary,
    /// but values of some data types such as statistics can be
    /// implicitly expanded to an object with the corresponding
    /// properties.
    fn summary(&self) -> ResultSource;

    /// Try to return integer numbers in time points.
    fn as_int_val(&self) -> Option<ResultValue<isize>>;

    /// Try to return a vector of integer numbers in time points.
    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>>;

    /// Try to return statistics based on integer numbers.
    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>>;

    /// Try to return timing statistics based on integer numbers.
    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>>;

    /// Try to return double floating point numbers in time points.
    fn as_floating_val(&self) -> Option<ResultValue<f64>>;

    /// Try to return a vector of double floating point numbers in time points.
    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>>;

    /// Try to return statistics based on double floating point numbers.
    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>>;

    /// Try to return timing statistics based on double floating point numbers.
    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>>;

    /// Try to return the `String` representation in time points.
    fn as_string_val(&self) -> Option<ResultValue<String>>;

    /// Return integer numbers in time points.
    fn get_int_val(&self) -> Result<ResultValue<isize>> {
        match self.as_int_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of integer numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return a vector of integer numbers in time points.
    fn get_int_vec_val(&self) -> Result<ResultValue<Vec<isize>>> {
        match self.as_int_vec_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of vectors of integer numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return statistics based on integer numbers.
    fn get_int_stats_val(&self) -> Result<ResultValue<SamplingStats<isize>>> {
        match self.as_int_stats_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of statistics based on integer numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return the timing statistics based on integer numbers.
    fn get_int_timing_stats_val(&self) -> Result<ResultValue<TimingStats<isize>>> {
        match self.as_int_timing_stats_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of the time-persistent statistics based on integer numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return double floating point numbers in time points.
    fn get_floating_val(&self) -> Result<ResultValue<f64>> {
        match self.as_floating_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of double floating point numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return a vector of double floating point numbers in time points.
    fn get_floating_vec_val(&self) -> Result<ResultValue<Vec<f64>>> {
        match self.as_floating_vec_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of vectors of double floating point numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return statistics based on double floating point numbers.
    fn get_floating_stats_val(&self) -> Result<ResultValue<SamplingStats<f64>>> {
        match self.as_floating_stats_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of statistics based on double floating point numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return the timing statistics based on integer numbers.
    fn get_floating_timing_stats_val(&self) -> Result<ResultValue<TimingStats<f64>>> {
        match self.as_floating_timing_stats_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of the time-persistent statistics based on double floating point numbers", self.get_name());
                result::Result::Err(err)
            }
        }
    }

    /// Return `String` values in time points.
    fn get_string_val(&self) -> Result<ResultValue<String>> {
        match self.as_string_val() {
            Some(x) => result::Result::Ok(x),
            None => {
                let err = format!("Cannot represent {} as a source of String values", self.get_name());
                result::Result::Err(err)
            }
        }
    }
}

/// The simulation results represented by an object having properties.
pub struct ResultObject {

    /// The object name.
    pub name: ResultName,

    /// The object identifier.
    pub id: ResultId,

    /// The object type identifier.
    pub type_id: ResultId,

    /// The object properties.
    pub props: Vec<ResultProperty>,

    /// A combined signal if present.
    pub observable: Rc<dyn Fn() -> ResultObservable>,

    /// A short version of the object, i.e. its summary.
    pub summary: Rc<dyn Fn() -> ResultSource>
}

/// The object property containing the simulation results.
pub struct ResultProperty {

    /// The property short label.
    pub label: ResultName,

    /// The property identifier.
    pub id: ResultId,

    /// The simulation results supplied by the property.
    pub source: ResultSource
}

/// The simulation results represented by a vector.
pub struct ResultVec {

    /// The vector name.
    pub name: ResultName,

    /// The vector identifier.
    pub id: ResultId,

    /// The results supplied by the vector items.
    pub items: Vec<ResultSource>,

    /// The subscript used as a suffix to create item names.
    pub subscript: Vec<ResultName>,

    /// A combined signal if present.
    pub observable: Rc<dyn Fn() -> ResultObservable>,

    /// A short version of the vector, i.e. summary.
    pub summary: Rc<dyn Fn() -> ResultSource>
}

impl ResultVec {

    /// Calculate the result vector signal.
    pub fn observable(items: &[ResultSource]) -> ResultObservable {
        items
            .iter()
            .fold(ResultObservable::empty(), move |acc, item| {
                acc.merge(item.observable())
            })
    }

    /// Calculate the result vector summary.
    pub fn summary(vec: Rc<ResultVec>) -> ResultSource {
        let items = {
            vec.items
                .iter()
                .map(|x| { x.summary() })
                .collect::<Vec<ResultSource>>()
        };
        ResultVec::with_items(vec, items)
    }

    /// Create a new instance with other items.
    fn with_items(vec: Rc<ResultVec>, items: Vec<ResultSource>) -> ResultSource {
        let name = vec.name.clone();
        let id = vec.id.clone();
        let subscript = vec.subscript.clone();
        let observable = {
            let items = items.clone();
            Rc::new(move || { ResultVec::observable(&items) })
        };
        let summary = {
            let items = items.clone();
            Rc::new(move || {
                ResultVec::with_items(vec.clone(), items.clone())
            })
        };
        let vec = ResultVec { name, id, items, subscript, observable, summary };
        let vec = Rc::new(vec);
        ResultSource::Vec(vec)
    }
}

/// It separates the simulation results when printing.
#[derive(Clone)]
pub struct ResultSeparator {

    /// The separator text.
    pub text: String
}

/// A parameterised value that actually represents a generalised result item that has no parametric type.
#[derive(Clone)]
pub struct ResultValue<E> {

    /// The value name.
    pub name: ResultName,

    /// The value name path.
    pub name_path: Vec<ResultName>,

    /// The value identifier.
    pub id: ResultId,

    /// The value identifier path.
    pub id_path: Vec<ResultId>,

    /// Simulation data supplied by the value.
    pub data: Rc<dyn Fn() -> ResultData<E>>,

    /// Whether the value emits a signal when changing simulation data.
    pub observable: Rc<dyn Fn() -> ResultObservable>
}

impl<E> ResultValue<E> {

    /// Map the contained data by applying the corresponding function.
    pub fn map<F, B>(self, f: F) -> ResultValue<B>
        where F: FnOnce(E) -> B + Clone + 'static,
              B: 'static,
              E: 'static
    {
        let ResultValue { name, name_path, id, id_path, data, observable } = self;
        let data = {
            Rc::new(move || {
                data().map(f.clone()).into_boxed()
            })
        };
        ResultValue {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: data,
            observable: observable
        }
    }

    /// Apply the functional composition.
    pub fn and_then<F, B>(self, f: F) -> ResultValue<B>
        where F: FnOnce(ResultData<E>) -> ResultData<B> + Clone + 'static,
              B: 'static,
              E: 'static
    {
        let ResultValue { name, name_path, id, id_path, data, observable } = self;
        let data = {
            Rc::new(move || {
                (f.clone())(data())
            })
        };
        ResultValue {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: data,
            observable: observable
        }
    }

    /// Convert the result value to a container.
    pub fn into_container(self) -> ResultContainer<ResultData<E>> {
        let ResultValue { name, name_path, id, id_path, data, observable } = self;
        let data = Rc::new(data());
        ResultContainer { name, name_path, id, id_path, data, observable }
    }

    /// Create a new raw property source by the specified result value.
    pub fn raw_prop_source<F, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F) -> ResultSource
        where F: FnOnce(ResultData<E>) -> ResultData<B> + Clone + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        let ResultValue { name, name_path, id: _, id_path, data, observable } = self;
        let item = ResultValue {
            name: name + "." + prop_name,
            name_path: {
                let mut name_path = name_path;
                name_path.push(prop_name.to_string());
                name_path
            },
            id: prop_id.clone(),
            id_path: {
                let mut id_path = id_path;
                id_path.push(prop_id);
                id_path
            },
            data: {
                let data = data.clone();
                Rc::new(move || { (prop_data.clone())(data()) })
            },
            observable: observable
        };
        ResultSource::Item(Rc::new(item))
    }

    /// Create a new raw property by the specified result value.
    pub fn raw_prop<F, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F) -> ResultProperty
        where F: FnOnce(ResultData<E>) -> ResultData<B> + Clone + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        let label  = prop_name.to_string();
        let id     = prop_id.clone();
        let source = self.raw_prop_source(prop_name, prop_id, prop_data);
        ResultProperty {
            label: label,
            id: id,
            source: source
        }
    }

    /// Create by the specified result value a mapped property which is recomputed each time again and again.
    pub fn map_prop<F, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F) -> ResultProperty
        where F: FnOnce(E) -> B + Clone + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        self.raw_prop(prop_name, prop_id,
            move |comp| { comp.map(prop_data).into_boxed() })
    }
}

/// A container of the simulation results such as queue, facility or array.
pub struct ResultContainer<E> {

    /// The container name.
    pub name: ResultName,

    /// The container name path.
    pub name_path: Vec<ResultName>,

    /// The container identifier.
    pub id: ResultId,

    /// The container identifier path.
    pub id_path: Vec<ResultId>,

    /// The container data.
    pub data: Rc<E>,

    /// Whether the container emits a signal when changing simulation data.
    pub observable: Rc<dyn Fn() -> ResultObservable>
}

impl<E> ResultContainer<E> {

    /// Map the contained data by applying the corresponding function.
    pub fn map<F, B>(val: Rc<Self>, f: F) -> ResultContainer<B>
        where F: FnOnce(&E) -> B
    {
        let data = Rc::new(f(&val.data));
        ResultContainer {
            name: val.name.clone(),
            name_path: val.name_path.clone(),
            id: val.id.clone(),
            id_path: val.id_path.clone(),
            data: data,
            observable: val.observable.clone()
        }
    }

    /// Create a new raw property source by the specified container.
    pub fn raw_prop_source<F1, F2, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F1, prop_observable: F2) -> ResultSource
        where F1: FnOnce(Rc<E>) -> ResultData<B> + Clone + 'static,
              F2: FnOnce(Rc<E>) -> ResultObservable + Clone + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        let ResultContainer { name, name_path, id: _, id_path, data, observable: _ } = self;
        let item = ResultValue {
            name: name + "." + prop_name,
            name_path: {
                let mut name_path = name_path;
                name_path.push(prop_name.to_string());
                name_path
            },
            id: prop_id.clone(),
            id_path: {
                let mut id_path = id_path;
                id_path.push(prop_id);
                id_path
            },
            data: {
                let data = data.clone();
                Rc::new(move || { (prop_data.clone())(data.clone()) })
            },
            observable: {
                Rc::new(move || { (prop_observable.clone())(data.clone()) })
            }
        };
        ResultSource::Item(Rc::new(item))
    }

    /// Create a new raw property by the specified container.
    pub fn raw_prop<F1, F2, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F1, prop_observable: F2) -> ResultProperty
        where F1: FnOnce(Rc<E>) -> ResultData<B> + Clone + 'static,
              F2: FnOnce(Rc<E>) -> ResultObservable + Clone + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        let label  = prop_name.to_string();
        let id     = prop_id.clone();
        let source = self.raw_prop_source(prop_name, prop_id, prop_data, prop_observable);
        ResultProperty {
            label: label,
            id: id,
            source: source
        }
    }

    /// Create a new constant property source by the specified container.
    pub fn const_prop<F, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F) -> ResultProperty
        where F: FnOnce(Rc<E>) -> B + Clone + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        self.raw_prop(prop_name, prop_id,
            move |e| { return_event(prop_data(e)).into_boxed() },
            move |_| { ResultObservable::Empty })
    }

    /// Create by the specified container a property that changes in the integration time points,
    // or it is supposed to be such one.
    pub fn integ_prop<F, M, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F) -> ResultProperty
        where F: FnOnce(Rc<E>) -> M + Clone + 'static,
              M: Event<Item = B> + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        self.raw_prop(prop_name, prop_id,
            move |e| { prop_data(e).into_boxed() },
            move |_| { ResultObservable::Unknown })
    }

    /// Create a new property by the specified container.
    pub fn prop<F1, F2, M, O, B>(self, prop_name: &str, prop_id: ResultId, prop_data: F1, prop_observable: F2) -> ResultProperty
        where F1: FnOnce(Rc<E>) -> M + Clone + 'static,
              F2: FnOnce(Rc<E>) -> O + Clone + 'static,
              M: Event<Item = B> + 'static,
              O: Observable<Message = ()> + 'static,
              ResultValue<B>: ResultItem,
              B: 'static,
              E: 'static
    {
        self.raw_prop(prop_name, prop_id,
            move |e| { prop_data(e).into_boxed() },
            move |e| { ResultObservable::Observable(prop_observable(e).into_boxed()) })
    }
}

impl<E> Clone for ResultContainer<E> {

    fn clone(&self) -> Self {
        ResultContainer {
            name: self.name.clone(),
            name_path: self.name_path.clone(),
            id: self.id.clone(),
            id_path: self.id_path.clone(),
            data: self.data.clone(),
            observable: self.observable.clone()
        }
    }
}

/// Represents the very simulation results.
pub type ResultData<E> = EventBox<E>;

/// Convert the timing statistics data to its normalised sampling-based representation.
fn norm_timing_stats_data<A>(data: ResultData<TimingStats<A>>) -> ResultData<SamplingStats<A>>
    where A: StatData + Copy + PartialOrd + 'static
{
    data.flat_map(move |x| {
        cons_event(move |p| {
            let n = p.iteration;
            let y = x.norm(n);
            result::Result::Ok(y)
        })
    }).into_boxed()
}

/// Whether an object containing the results emits a signal notifying about change of data.
pub enum ResultObservable {

    /// There is no signal at all.
    Empty,

    /// The signal is unknown, but the entity probably changes.
    Unknown,

    /// When the signal is precisely specified.
    Observable(ObservableBox<()>),

    /// When the specified signal was combined with unknown signal.
    ObservableMix(ObservableBox<()>)
}

impl ResultObservable {

    /// Construct a new result observable by the specified optional pure observable.
    pub fn from_option(observable: Option<ObservableBox<()>>) -> Self {
        match observable {
            Some(x) => ResultObservable::Observable(x),
            None => ResultObservable::Empty
        }
    }

    /// Return an empty observable.
    pub fn empty() -> Self {
        ResultObservable::Empty
    }

    /// Merge two observables.
    pub fn merge(self, other: Self) -> Self {
        match (self, other) {
            (ResultObservable::Empty, z) => z,

            (ResultObservable::Unknown, ResultObservable::Empty) => ResultObservable::Unknown,
            (ResultObservable::Unknown, ResultObservable::Unknown) => ResultObservable::Unknown,
            (ResultObservable::Unknown, ResultObservable::Observable(x)) => ResultObservable::ObservableMix(x),
            (ResultObservable::Unknown, z@ResultObservable::ObservableMix(_)) => z,

            (z@ResultObservable::Observable(_), ResultObservable::Empty) => z,
            (ResultObservable::Observable(x), ResultObservable::Unknown) => ResultObservable::ObservableMix(x),
            (ResultObservable::Observable(x), ResultObservable::Observable(y)) => ResultObservable::Observable(x.merge(y).into_boxed()),
            (ResultObservable::Observable(x), ResultObservable::ObservableMix(y)) => ResultObservable::ObservableMix(x.merge(y).into_boxed()),

            (z@ResultObservable::ObservableMix(_), ResultObservable::Empty) => z,
            (z@ResultObservable::ObservableMix(_), ResultObservable::Unknown) => z,
            (ResultObservable::ObservableMix(x), ResultObservable::Observable(y)) => ResultObservable::ObservableMix(x.merge(y).into_boxed()),
            (ResultObservable::ObservableMix(x), ResultObservable::ObservableMix(y)) => ResultObservable::ObservableMix(x.merge(y).into_boxed()),
        }
    }

    /// Return a pure signal as a result of combination of the predefined signals
    /// with the specified result signal usually provided by the sources.
    ///
    /// The signal returned is triggered when the source signal is triggered.
    /// The pure signal is also triggered in the integration time points
    /// if the source signal is unknown or it was combined with any unknown signal.
    pub fn pure(self, predefined: &ResultPredefinedObservableSet) -> ObservableBox<()> {
        match self {
            ResultObservable::Empty => {
                predefined.in_start_time().map(|_| ())
                    .into_boxed()
            },
            ResultObservable::Unknown => {
                predefined.in_integ_times().map(|_| ())
                    .into_boxed()
            },
            ResultObservable::Observable(x) => {
                predefined.in_start_time().map(|_| ())
                    .merge(predefined.in_stop_time().map(|_| ()))
                    .merge(x)
                    .into_boxed()
            },
            ResultObservable::ObservableMix(x) => {
                predefined.in_integ_times().map(|_| ())
                    .merge(x)
                    .into_boxed()
            }
        }
    }
}

impl ResultItem for ResultValue<isize> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn summary(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        Some(self.clone())
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        Some(self.clone().map(move |x| { vec![x] }))
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        Some(self.clone().map(move |x| { SamplingStats::from_sample(x) }))
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        Some(self.clone().map(move |x| { x as f64 }))
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        Some(self.clone().map(move |x| { vec![x as f64] }))
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().map(move |x| { SamplingStats::from_sample(x as f64) }))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<f64> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn summary(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        None
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        Some(self.clone())
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        Some(self.clone().map(move |x| { vec![x] }))
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().map(move |x| { SamplingStats::from_sample(x) }))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<Vec<isize>> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn summary(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        Some(self.clone())
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        Some(self.clone().map(move |x| { SamplingStats::from_samples(x.iter()) }))
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        Some(self.clone().map(move |x| { x.iter().map(|y| { *y as f64 }).collect() }))
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().map(move |x| { SamplingStats::from_samples(x.iter().map(|y| { *y as f64 })) }))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{:?}", x) }))
    }
}

impl ResultItem for ResultValue<Vec<f64>> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn summary(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        None
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        Some(self.clone())
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().map(move |x| { SamplingStats::from_samples(x.iter()) }))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{:?}", x) }))
    }
}

impl ResultItem for ResultValue<SamplingStats<isize>> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::from_stats(self.clone())
    }

    fn summary(&self) -> ResultSource {
        ResultSource::summary_from_stats(self.clone())
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        Some(self.clone())
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        None
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().map(move |x| { x.into() }))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<SamplingStats<f64>> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::from_stats(self.clone())
    }

    fn summary(&self) -> ResultSource {
        ResultSource::summary_from_stats(self.clone())
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        None
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        None
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone())
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<TimingStats<isize>> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::from_timing_stats(self.clone())
    }

    fn summary(&self) -> ResultSource {
        ResultSource::summary_from_timing_stats(self.clone())
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        Some(self.clone().and_then(norm_timing_stats_data))
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        Some(self.clone())
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        None
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().map(move |x| { x.into() }).and_then(norm_timing_stats_data))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        Some(self.clone().map(move |x| { x.into() }))
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<TimingStats<f64>> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::from_timing_stats(self.clone())
    }

    fn summary(&self) -> ResultSource {
        ResultSource::summary_from_timing_stats(self.clone())
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        None
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        None
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        Some(self.clone().and_then(norm_timing_stats_data))
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        Some(self.clone())
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<bool> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn summary(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        None
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        None
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        None
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone().map(move |x| { format!("{}", x) }))
    }
}

impl ResultItem for ResultValue<String> {

    fn get_name(&self) -> ResultName {
        self.name.clone()
    }

    fn get_name_path(&self) -> Vec<ResultName> {
        self.name_path.clone()
    }

    fn get_id(&self) -> ResultId {
        self.id.clone()
    }

    fn get_id_path(&self) -> Vec<ResultId> {
        self.id_path.clone()
    }

    fn observable(&self) -> ResultObservable {
        (self.observable)()
    }

    fn expand(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn summary(&self) -> ResultSource {
        ResultSource::Item(Rc::new(self.clone()))
    }

    fn as_int_val(&self) -> Option<ResultValue<isize>> {
        None
    }

    fn as_int_vec_val(&self) -> Option<ResultValue<Vec<isize>>> {
        None
    }

    fn as_int_stats_val(&self) -> Option<ResultValue<SamplingStats<isize>>> {
        None
    }

    fn as_int_timing_stats_val(&self) -> Option<ResultValue<TimingStats<isize>>> {
        None
    }

    fn as_floating_val(&self) -> Option<ResultValue<f64>> {
        None
    }

    fn as_floating_vec_val(&self) -> Option<ResultValue<Vec<f64>>> {
        None
    }

    fn as_floating_stats_val(&self) -> Option<ResultValue<SamplingStats<f64>>> {
        None
    }

    fn as_floating_timing_stats_val(&self) -> Option<ResultValue<TimingStats<f64>>> {
        None
    }

    fn as_string_val(&self) -> Option<ResultValue<String>> {
        Some(self.clone())
    }
}

/// It represents the predefined signals provided by every simulation model.
#[derive(Clone)]
pub struct ResultPredefinedObservableSet {

    /// The source of signal triggered in the integration time points.
    source_in_integ_times: Rc<ObservableSource<f64>>,

    /// The source of signal triggered in the start time.
    source_in_start_time: Rc<ObservableSource<f64>>,

    /// The source of signal triggered in the stop time.
    source_in_stop_time: Rc<ObservableSource<f64>>
}

impl ResultPredefinedObservableSet {

    /// Create a new predefined observable set.
    pub fn new() -> NewResultPredefinedObservableSet {
        NewResultPredefinedObservableSet {}
    }

    /// The signal triggered in the integration time points.
    pub fn in_integ_times(&self) -> ObservableBox<f64> {
        self.source_in_integ_times.publish().into_boxed()
    }

    /// The signal triggered in the start time.
    pub fn in_start_time(&self) -> ObservableBox<f64> {
        self.source_in_start_time.publish().into_boxed()
    }

    /// The signal triggered in the stop time.
    pub fn in_stop_time(&self) -> ObservableBox<f64> {
        self.source_in_stop_time.publish().into_boxed()
    }
}

/// It creates a new predefined observable set.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct NewResultPredefinedObservableSet {}

impl Simulation for NewResultPredefinedObservableSet {

    type Item = ResultPredefinedObservableSet;

    #[doc(hidden)]
    fn call_simulation(self, r: &Run) -> simulation::Result<Self::Item> {
        cons_event(move |p| {
            let source_in_integ_times = new_observable_source_in_integ_times().call_event(p)?;
            let source_in_start_time  = new_observable_source_in_start_time().call_event(p)?;
            let source_in_stop_time   = new_observable_source_in_stop_time().call_event(p)?;
            result::Result::Ok(ResultPredefinedObservableSet {
                source_in_integ_times: source_in_integ_times,
                source_in_start_time: source_in_start_time,
                source_in_stop_time: source_in_stop_time
            })
        }).run_in_start_time_by(false)
            .call_simulation(r)
    }
}

/// It contains the results of simulation.
pub struct ResultSet {

    /// The sources of simulation results as a map of associated names.
    map: ResultSourceMap,

    /// The sources of simulation results an an ordered vector.
    vec: Vec<ResultSource>
}

impl Into<ResultSet> for Vec<ResultSource> {
    fn into(self) -> ResultSet {
        let mut map = OrdMap::empty();
        for source in self.iter() {
            map = map.insert(source.get_name(), source.clone());
        }
        ResultSet {
            map: map,
            vec: self
        }
    }
}

impl Clone for ResultSet {
    fn clone(&self) -> Self {
        let sources: Vec<ResultSource> = self.vec.iter().map(|x| x.clone()).collect();
        sources.into()
    }
}

impl ResultSet {

    /// An empty result set.
    pub fn empty() -> Self {
        let sources: Vec<ResultSource> = vec![];
        sources.into()
    }

    /// Merge two result sets.
    pub fn merge(&self, other: &Self) -> Self {
        let mut sources  = self.vec.clone();
        let mut sources2 = other.vec.clone();
        sources.append(&mut sources2);
        sources.into()
    }

    /// Return a summarised and usually more short version of the result set by expanding
    /// the main properties or excluding auxiliary properties if required.
    pub fn summary(&self) -> Self {
        let sources: Vec<ResultSource> = self.vec.iter().map(|x| x.summary()).collect();
        sources.into()
    }

    /// Return an expanded version of the result set while expanding the properties as possible.
    pub fn expand(&self) -> Self {
        let sources: Vec<ResultSource> = self.vec.iter().map(|x| x.expand()).collect();
        sources.into()
    }

    /// Return a signal emitted by the set.
    pub fn observable(&self) -> ResultObservable {
        self.vec
            .iter()
            .map(|x| x.observable())
            .fold(ResultObservable::Empty, |x, y| x.merge(y))
    }

    /// Get the result sources.
    pub fn get_sources(&self) -> impl Iterator<Item = &ResultSource> {
        self.vec.iter()
    }

    /// Represent the result set as integer numbers.
    pub fn get_int_vals(&self) -> Result<Vec<ResultValue<isize>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_int_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as vectors of integer numbers.
    pub fn get_int_vec_vals(&self) -> Result<Vec<ResultValue<Vec<isize>>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_int_vec_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as statistics based on integer numbers.
    pub fn get_int_stats_vals(&self) -> Result<Vec<ResultValue<SamplingStats<isize>>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_int_stats_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as time-persistent statistics based on integer numbers.
    pub fn get_int_timing_stats_vals(&self) -> Result<Vec<ResultValue<TimingStats<isize>>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_int_timing_stats_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as floating point numbers.
    pub fn get_floating_vals(&self) -> Result<Vec<ResultValue<f64>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_floating_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as vectors of floating point numbers.
    pub fn get_floating_vec_vals(&self) -> Result<Vec<ResultValue<Vec<f64>>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_floating_vec_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as statistics based on floating point numbers.
    pub fn get_floating_stats_vals(&self) -> Result<Vec<ResultValue<SamplingStats<f64>>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_floating_stats_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as time-persistent statistics based on floating point numbers.
    pub fn get_floating_timing_stats_vals(&self) -> Result<Vec<ResultValue<TimingStats<f64>>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_floating_timing_stats_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }

    /// Represent the result set as `String` values.
    pub fn get_string_vals(&self) -> Result<Vec<ResultValue<String>>> {
        let mut ys = Vec::new();
        for x in self.vec.iter() {
            let mut zs = x.get_string_vals()?;
            ys.append(&mut zs);
        }
        result::Result::Ok(ys)
    }
}

/// It transforms the results of simulation.
pub struct ResultTransform {
    f: Rc<dyn Fn(&ResultSet) -> Result<ResultSet>>
}

impl Clone for ResultTransform {
    fn clone(&self) -> Self {
        ResultTransform {
            f: self.f.clone()
        }
    }
}

impl ResultTransform {

    /// Create a new result transform.
    pub fn new<F>(f: F) -> ResultTransform
        where F: Fn(&ResultSet) -> Result<ResultSet> + 'static
    {
        ResultTransform {
            f: Rc::new(f)
        }
    }

    /// Transform the results.
    pub fn call(&self, results: &ResultSet) -> Result<ResultSet> {
        (self.f)(results)
    }

    /// Create an empty result transform that retains the result set as is.
    pub fn empty() -> Self {
        ResultTransform::new(|x| result::Result::Ok(x.clone()))
    }

    /// Merge two result transformations.
    pub fn merge(&self, other: &Self) -> Self {
        let f  = self.f.clone();
        let f2 = other.f.clone();
        ResultTransform::new(move |x| {
            let ys1 = f(x)?;
            let ys2 = f2(x)?;
            result::Result::Ok(ys1.merge(&ys2))
        })
    }

    /// Compose two result transformations.
    pub fn and_then(&self, other: &Self) -> Self {
        let f  = self.f.clone();
        let f2 = other.f.clone();
        ResultTransform::new(move |x| {
            let ys1 = f(x)?;
            f2(&ys1)
        })
    }

    /// Return a summarised and usually more short version of the result set by expanding
    /// the main properties or excluding auxiliary properties if required.
    pub fn summary() -> Self {
        ResultTransform::new(|x| result::Result::Ok(x.summary()))
    }

    /// Return an expanded version of the result set while expanding the properties as possible.
    pub fn expand() -> Self {
        ResultTransform::new(|x| result::Result::Ok(x.expand()))
    }

    /// Take a result by its name.
    pub fn by_name(name: String) -> Self {
        ResultTransform::new(move |x| {
            match x.map.get(&name) {
                Some(ref source) => {
                    let sources: Vec<ResultSource> = vec![source.deref().clone()];
                    result::Result::Ok(sources.into())
                },
                None => {
                    let err = format!("Not found result source with name {}", name);
                    result::Result::Err(err)
                }
            }
        })
    }

    /// Take a result from the object with the specified identifier. It can identify
    /// an item, object property, the object iself, vector or its elements.
    pub fn by_id(id: ResultId) -> Self {
        ResultTransform::new(move |x| {
            fn process_source(result: &mut Vec<ResultSource>, source: &ResultSource, id: &ResultId) -> Result<()> {
                match &source {
                    ResultSource::Item(ref s) if s.get_id() == *id => {
                        result.push(source.clone())
                    },
                    ResultSource::Item(_) => {
                        let err = format!("Expected to find item with id = {:?}", *id);
                        return result::Result::Err(err);
                    },
                    ResultSource::Object(ref s) if s.id == *id => {
                        result.push(source.clone())
                    },
                    ResultSource::Object(ref s) => {
                        let ps = s.props.iter().find(|p| { p.id == *id });
                        match ps {
                            Some(ref p) => result.push(p.source.clone()),
                            None => {
                                let err = format!("Not found property with id = {:?}", *id);
                                return result::Result::Err(err);
                            }
                        }
                    },
                    ResultSource::Vec(ref s) if s.id == *id => {
                        result.push(source.clone())
                    },
                    ResultSource::Vec(ref s) => {
                        for item in s.items.iter() {
                            process_source(result, item, id)?;
                        }
                    },
                    ResultSource::Separator(_) => {
                        let err = format!("Expected to find either item, or object, or vector with id = {:?}", id);
                        return result::Result::Err(err);
                    }
                }
                result::Result::Ok(())
            }
            let mut ys: Vec<ResultSource> = Vec::new();
            for source in x.vec.iter() {
                process_source(&mut ys, source, &id)?;
            }
            result::Result::Ok(ys.into())
        })
    }
}

impl<T> ResultProvider for ParameterFn<T>
    where T: 'static,
          ResultValue<T>: ResultItem
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let item = ResultValue {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: {
                Rc::new(move || {
                    self.next().into_event().into_boxed()
                })
            },
            observable: {
                Rc::new(move || ResultObservable::Unknown)
            }
        };
        ResultSource::Item(Rc::new(item))
    }
}

impl<T> ResultProvider for SimulationFn<T>
    where T: 'static,
          ResultValue<T>: ResultItem
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let item = ResultValue {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: {
                Rc::new(move || {
                    self.next().into_event().into_boxed()
                })
            },
            observable: {
                Rc::new(move || ResultObservable::Unknown)
            }
        };
        ResultSource::Item(Rc::new(item))
    }
}

impl<T> ResultProvider for EventFn<T>
    where T: 'static,
          ResultValue<T>: ResultItem
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let item = ResultValue {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: {
                Rc::new(move || {
                    self.next().into_boxed()
                })
            },
            observable: {
                Rc::new(move || ResultObservable::Unknown)
            }
        };
        ResultSource::Item(Rc::new(item))
    }
}

impl<T> ResultProvider for Rc<RefComp<T>>
    where T: Clone + 'static,
          ResultValue<T>: ResultItem
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let item = ResultValue {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: {
                Rc::new(move || {
                    RefComp::read(self.clone()).into_boxed()
                })
            },
            observable: {
                Rc::new(move || ResultObservable::Unknown)
            }
        };
        ResultSource::Item(Rc::new(item))
    }
}

impl<T> ResultProvider for Vec<T>
    where T: ResultProvider
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let len = self.len();
        let items: Vec<ResultSource> = {
            let name = name.clone();
            self.into_iter()
                .enumerate()
                .map(move |(i, x)| {
                    let idx = format!("[{}]", i);
                    let name = name.clone() + &idx;
                    let name_path = {
                        let mut name_path = name_path.clone();
                        name_path.push(idx);
                        name_path
                    };
                    let id = ResultId::VectorItem(i);
                    let id_path = {
                        let mut id_path = id_path.clone();
                        id_path.push(id.clone());
                        id_path
                    };
                    x.get_source_by_id(name, name_path, id, id_path)
                })
                .collect()
        };
        let subscript: Vec<ResultName> = {
            (0 .. len)
                .map(|i| format!("[{}]", i))
                .collect()
        };
        let memo: Rc<RefCell<Option<Rc<ResultVec>>>> = Rc::new(RefCell::new(None));
        let item = ResultVec {
            name: name,
            id: id,
            items: items.clone(),
            subscript: subscript,
            observable: {
                Rc::new(move || {
                    ResultVec::observable(&items)
                })
            },
            summary: {
                let memo = memo.clone();
                Rc::new(move || {
                    let vec = {
                        let cell = memo.borrow();
                        match *cell {
                            Some(ref x) => x.clone(),
                            None => panic!("Invalid memo cell")
                        }
                    };
                    ResultVec::summary(vec)
                })
            }
        };
        let item = Rc::new(item);
        {
            let mut cell = memo.borrow_mut();
            *cell = Some(item.clone());
        }
        ResultSource::Vec(item)
    }
}

impl ResultProvider for Rc<Queue> {

    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let cont = ResultContainer {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: self.clone(),
            observable: {
                Rc::new(move || ResultObservable::Observable(self.changed_().into_boxed()))
            }
        };
        ResultSource::from_queue(cont)
    }
}

impl<T> ResultProvider for Rc<Facility<T>>
    where T: 'static
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let cont = ResultContainer {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: self.clone(),
            observable: {
                Rc::new(move || ResultObservable::Observable(self.changed_().into_boxed()))
            }
        };
        ResultSource::from_facility(cont)
    }
}

impl<T> ResultProvider for Rc<Storage<T>>
    where T: 'static
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let cont = ResultContainer {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: self.clone(),
            observable: {
                Rc::new(move || ResultObservable::Observable(self.changed_().into_boxed()))
            }
        };
        ResultSource::from_storage(cont)
    }
}

impl<SI, SM, SO, T> ResultProvider for Rc<queue::stats::Queue<SI, SM, SO, T>>
    where SI: QueueStrategy + 'static,
          SM: QueueStrategy + 'static,
          SO: QueueStrategy + 'static,
          T: Clone + 'static
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let cont = ResultContainer {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: self.clone(),
            observable: {
                Rc::new(move || ResultObservable::Observable(self.changed_().into_boxed()))
            }
        };
        ResultSource::from_bounded_queue(cont)
    }
}

impl<SM, SO, T> ResultProvider for Rc<queue::unbounded::stats::Queue<SM, SO, T>>
    where SM: QueueStrategy + 'static,
          SO: QueueStrategy + 'static,
          T: Clone + 'static
{
    fn get_source_by_id(self, name: ResultName, name_path: Vec<ResultName>, id: ResultId, id_path: Vec<ResultId>) -> ResultSource {
        let cont = ResultContainer {
            name: name,
            name_path: name_path,
            id: id,
            id_path: id_path,
            data: self.clone(),
            observable: {
                Rc::new(move || ResultObservable::Observable(self.changed_().into_boxed()))
            }
        };
        ResultSource::from_unbounded_queue(cont)
    }
}
