// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::collections::BTreeMap;

/// A locale to output the simulation results.
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum ResultLocale {

    /// The Russian locale.
    Ru,

    /// The English locale.
    En
}

/// A name used for indentifying the results when generating output.
pub type ResultName = String;

/// A description used for describing the results when generating output.
pub type ResultDescription = String;

/// The result enity identifier.
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum ResultId {

    /// The time computation.
    Time,

    /// The vector item.
    VectorItem(usize),

    /// Observation-based statistics.
    SamplingStats,

    /// The count of samples in the statistics summary.
    SamplingStatsCount,

    /// The average value in the statistics summary.
    SamplingStatsMean,

    /// The average square value in the statistics summary.
    SamplingStatsMean2,

    /// The deviation in the statistics summary.
    SamplingStatsDeviation,

    /// The variance in the statistics summary.
    SamplingStatsVariance,

    /// The minimum value in the statistics summary.
    SamplingStatsMin,

    /// The maximum value in the statistics summary.
    SamplingStatsMax,

    /// A time persistent variable statistics.
    TimingStats,

    /// The count of samples in the statistics summary.
    TimingStatsCount,

    /// The average value in the statistics summary.
    TimingStatsMean,

    /// The deviation in the statistics summary.
    TimingStatsDeviation,

    /// The variance in the statistics summary.
    TimingStatsVariance,

    /// The minimum value in the statistics summary.
    TimingStatsMin,

    /// The maximum value in the statistics summary.
    TimingStatsMax,

    /// The time at which the minimum is attained.
    TimingStatsMinTime,

    /// The time at which the maximum is attained.
    TimingStatsMaxTime,

    /// The start time of sampling.
    TimingStatsStartTime,

    /// The last time of sampling.
    TimingStatsLastTime,

    /// The sum of values.
    TimingStatsSum,

    /// The sum of square values.
    TimingStatsSum2,

    /// The queue identifier.
    Queue,

    /// Whether the queue is empty.
    QueueEmpty,

    /// Whether the queue is full.
    QueueFull,

    /// The queue capacity,
    QueueMaxCount,

    /// The queue size.
    QueueCount,

    /// The queue size statistics.
    QueueCountStats,

    /// The queue content.
    QueueContent,

    /// The queue content statistics.
    QueueContentStats,

    /// The enqueue count.
    EnqueueCount,

    /// The enqueue count statistics.
    EnqueueCountStats,

    /// The enqueue zero entry count.
    EnqueueZeroEntryCount,

    /// The enqueue lost count.
    EnqueueLostCount,

    /// The enqueue store count.
    EnqueueStoreCount,

    /// The dequeue count.
    DequeueCount,

    /// The dequeue extraction count.
    DequeueExtractCount,

    /// The queue load factor.
    QueueLoadFactor,

    /// The enqueue rate.
    EnqueueRate,

    /// The enqueue store rate.
    EnqueueStoreRate,

    /// The dequeue rate.
    DequeueRate,

    /// The dequeue extract rate.
    DequeueExtractRate,

    /// The queue wait time.
    QueueWaitTime,

    /// The queue non-zero entry wait time.
    QueueNotZeroEntryWaitTime,

    /// The queue total wait time.
    QueueTotalWaitTime,

    /// The enqueue wait time.
    EnqueueWaitTime,

    /// The dequeue wait time.
    DequeueWaitTime,

    /// The queue rate.
    QueueRate,

    /// The facility identifier.
    Facility,

    /// The facility queue count.
    FacilityQueueCount,

    /// The facility queue count statistics.
    FacilityQueueCountStats,

    /// The total facility wait time.
    FacilityTotalWaitTime,

    /// The facility wait time.
    FacilityWaitTime,

    /// The total facility holding time.
    FacilityTotalHoldingTime,

    /// The facility holding time.
    FacilityHoldingTime,

    /// The facility interrupted flag.
    FacilityInterrupted,

    /// The facility counter.
    FacilityCount,

    /// The facility counter statistics.
    FacilityCountStats,

    /// The facility capture count.
    FacilityCaptureCount,

    /// The facility utilisation count.
    FacilityUtilCount,

    /// The facility utilisation count statistics.
    FacilityUtilCountStats,

    /// The storage identifier.
    Storage,

    /// The storage capacity.
    StorageCapacity,

    /// Whether the storage is empty.
    StorageEmpty,

    /// Whether the storage is full.
    StorageFull,

    /// The storage queue count.
    StorageQueueCount,

    /// The storage queue count statistics.
    StorageQueueCountStats,

    /// The storage total wait time.
    StorageTotalWaitTime,

    /// The storage wait time.
    StorageWaitTime,

    /// The average storage holding time.
    StorageAverageHoldingTime,

    /// The storage content.
    StorageContent,

    /// The storage content statistics.
    StorageContentStats,

    /// The storage use count.
    StorageUseCount,

    /// The used storage content.
    StorageUsedContent,

    /// The storage utilisation count.
    StorageUtilCount,

    /// The storage utilisation count statistics.
    StorageUtilCountStats,

    /// An user defined description.
    UserDefined(UserDefinedResult),

    /// A localised result.
    LocalisedResult(LocalisedResult)
}

/// It describes the user-defined simulation result.
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct UserDefinedResult {

    /// The user-defined result name.
    pub name: ResultName,

    /// The user-defined result description.
    pub descr: ResultDescription,

    /// The user-defined result title.
    pub title: ResultDescription
}

/// This is a localisation of the specified simulation result.
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct LocalisedResult {

    /// The localised descriptions.
    descrs: BTreeMap<ResultLocale, ResultDescription>,

    /// The localised titles.
    titles: BTreeMap<ResultLocale, ResultDescription>
}

impl LocalisedResult {

    /// Create a new localised result.
    pub fn new(descrs: BTreeMap<ResultLocale, ResultDescription>,
        titles: BTreeMap<ResultLocale, ResultDescription>) -> Self
    {
        Self { descrs, titles }
    }
}

impl ResultId {

    /// Get the result description.
    pub fn get_descr(&self, locale: &ResultLocale) -> String {
        match self {
            &ResultId::Time => String::from("simulation time"),
            &ResultId::VectorItem(ref i) => format!("item {}", i),
            &ResultId::SamplingStats => String::from("observation-based statistics"),
            &ResultId::SamplingStatsCount => String::from("count"),
            &ResultId::SamplingStatsMean => String::from("mean"),
            &ResultId::SamplingStatsMean2 => String::from("mean square"),
            &ResultId::SamplingStatsDeviation => String::from("deviation"),
            &ResultId::SamplingStatsVariance => String::from("variance"),
            &ResultId::SamplingStatsMin => String::from("minimum"),
            &ResultId::SamplingStatsMax => String::from("maximum"),
            &ResultId::TimingStats => String::from("time-persistent statistics"),
            &ResultId::TimingStatsCount => String::from("count"),
            &ResultId::TimingStatsMean => String::from("mean"),
            &ResultId::TimingStatsDeviation => String::from("deviation"),
            &ResultId::TimingStatsVariance => String::from("variance"),
            &ResultId::TimingStatsMin => String::from("minimum"),
            &ResultId::TimingStatsMax => String::from("maximum"),
            &ResultId::TimingStatsMinTime => String::from("time of minimum"),
            &ResultId::TimingStatsMaxTime => String::from("time of maximum"),
            &ResultId::TimingStatsStartTime => String::from("start time"),
            &ResultId::TimingStatsLastTime => String::from("last time"),
            &ResultId::TimingStatsSum => String::from("sum"),
            &ResultId::TimingStatsSum2 => String::from("sum square"),
            &ResultId::Queue => String::from("queue"),
            &ResultId::QueueEmpty => String::from("is the queue empty?"),
            &ResultId::QueueFull => String::from("is the queue full?"),
            &ResultId::QueueMaxCount => String::from("the queue capacity"),
            &ResultId::QueueCount => String::from("the current queue size"),
            &ResultId::QueueCountStats => String::from("the queue size statistics"),
            &ResultId::QueueContent => String::from("the current queue content"),
            &ResultId::QueueContentStats => String::from("the queue content statistics"),
            &ResultId::EnqueueCount => String::from("number of enqueued items"),
            &ResultId::EnqueueCountStats => String::from("statistics for a number of enqueued items"),
            &ResultId::EnqueueZeroEntryCount => String::from("number of zero entry enqueued items"),
            &ResultId::EnqueueLostCount => String::from("number of items lost when tried to enqueue"),
            &ResultId::EnqueueStoreCount => String::from("number of items stored in the queue"),
            &ResultId::DequeueCount => String::from("number of dequeue requests"),
            &ResultId::DequeueExtractCount => String::from("number of extracted items"),
            &ResultId::QueueLoadFactor => String::from("the queue size / capacity"),
            &ResultId::EnqueueRate => String::from("enqueue requests per time"),
            &ResultId::EnqueueStoreRate => String::from("number of stored items per time"),
            &ResultId::DequeueRate => String::from("dequeue requests per time"),
            &ResultId::DequeueExtractRate => String::from("number of extracted items per time"),
            &ResultId::QueueWaitTime => String::from("wait time"),
            &ResultId::QueueNotZeroEntryWaitTime => String::from("wait time without zero entries"),
            &ResultId::QueueTotalWaitTime => String::from("total wait time by including the enque request wait time"),
            &ResultId::EnqueueWaitTime => String::from("the enque request wait time"),
            &ResultId::DequeueWaitTime => String::from("the deque request wait time"),
            &ResultId::QueueRate => String::from("estimated queue rate"),
            &ResultId::Facility => String::from("facility"),
            &ResultId::FacilityQueueCount => String::from("the current queue size"),
            &ResultId::FacilityQueueCountStats => String::from("the queue size statistics"),
            &ResultId::FacilityTotalWaitTime => String::from("total wait time"),
            &ResultId::FacilityWaitTime => String::from("wait time"),
            &ResultId::FacilityTotalHoldingTime => String::from("total holding time"),
            &ResultId::FacilityHoldingTime => String::from("holding time"),
            &ResultId::FacilityInterrupted => String::from("is the facility currently interrupted?"),
            &ResultId::FacilityCount => String::from("the current available count"),
            &ResultId::FacilityCountStats => String::from("the available count statistics"),
            &ResultId::FacilityCaptureCount => String::from("the current capture count"),
            &ResultId::FacilityUtilCount => String::from("utilization"),
            &ResultId::FacilityUtilCountStats => String::from("utilization statistics"),
            &ResultId::Storage => String::from("storage"),
            &ResultId::StorageCapacity => String::from("capacity"),
            &ResultId::StorageEmpty => String::from("is completely unused?"),
            &ResultId::StorageFull => String::from("is completely used?"),
            &ResultId::StorageQueueCount => String::from("the current queue size"),
            &ResultId::StorageQueueCountStats => String::from("the queue size statistics"),
            &ResultId::StorageTotalWaitTime => String::from("total wait time"),
            &ResultId::StorageWaitTime => String::from("wait time"),
            &ResultId::StorageAverageHoldingTime => String::from("average holding time"),
            &ResultId::StorageContent => String::from("the current available content"),
            &ResultId::StorageContentStats => String::from("the available content statistics"),
            &ResultId::StorageUseCount => String::from("the total use count"),
            &ResultId::StorageUsedContent => String::from("the total used content"),
            &ResultId::StorageUtilCount => String::from("utilization"),
            &ResultId::StorageUtilCountStats => String::from("utilization statistics"),
            &ResultId::UserDefined(ref r) => r.descr.clone(),
            &ResultId::LocalisedResult(ref r) => {
                r.descrs.get(locale)
                    .or_else(|| r.descrs.get(&ResultLocale::En))
                    .or_else(|| r.descrs.values().next())
                    .unwrap()
                    .clone()
            }
        }
    }

    /// Get the result title.
    pub fn get_title(&self, locale: &ResultLocale) -> String {
        match self {
            &ResultId::Time => String::from("time"),
            &ResultId::VectorItem(ref i) => format!("item {}", i),
            &ResultId::SamplingStats => String::from("stats"),
            &ResultId::SamplingStatsCount => String::from("count"),
            &ResultId::SamplingStatsMean => String::from("mean"),
            &ResultId::SamplingStatsMean2 => String::from("mean square"),
            &ResultId::SamplingStatsDeviation => String::from("deviation"),
            &ResultId::SamplingStatsVariance => String::from("variance"),
            &ResultId::SamplingStatsMin => String::from("minimum"),
            &ResultId::SamplingStatsMax => String::from("maximum"),
            &ResultId::TimingStats => String::from("time-persistent stats"),
            &ResultId::TimingStatsCount => String::from("count"),
            &ResultId::TimingStatsMean => String::from("mean"),
            &ResultId::TimingStatsDeviation => String::from("deviation"),
            &ResultId::TimingStatsVariance => String::from("variance"),
            &ResultId::TimingStatsMin => String::from("minimum"),
            &ResultId::TimingStatsMax => String::from("maximum"),
            &ResultId::TimingStatsMinTime => String::from("time of minimum"),
            &ResultId::TimingStatsMaxTime => String::from("time of maximum"),
            &ResultId::TimingStatsStartTime => String::from("start time"),
            &ResultId::TimingStatsLastTime => String::from("last time"),
            &ResultId::TimingStatsSum => String::from("sum"),
            &ResultId::TimingStatsSum2 => String::from("sum square"),
            &ResultId::Queue => String::from("queue"),
            &ResultId::QueueEmpty => String::from("empty queue?"),
            &ResultId::QueueFull => String::from("full queue?"),
            &ResultId::QueueMaxCount => String::from("queue capacity"),
            &ResultId::QueueCount => String::from("queue size"),
            &ResultId::QueueCountStats => String::from("queue size stats"),
            &ResultId::QueueContent => String::from("queue content"),
            &ResultId::QueueContentStats => String::from("queue content stats"),
            &ResultId::EnqueueCount => String::from("enque count"),
            &ResultId::EnqueueCountStats => String::from("enque count stats"),
            &ResultId::EnqueueZeroEntryCount => String::from("zero entry enque count"),
            &ResultId::EnqueueLostCount => String::from("count of lost when tried to enque"),
            &ResultId::EnqueueStoreCount => String::from("count of stored in queue"),
            &ResultId::DequeueCount => String::from("deque request count"),
            &ResultId::DequeueExtractCount => String::from("number of extracted items"),
            &ResultId::QueueLoadFactor => String::from("queue load factor"),
            &ResultId::EnqueueRate => String::from("enque rate"),
            &ResultId::EnqueueStoreRate => String::from("stored item count rate"),
            &ResultId::DequeueRate => String::from("deque request rate"),
            &ResultId::DequeueExtractRate => String::from("extracted item count rate"),
            &ResultId::QueueWaitTime => String::from("wait time"),
            &ResultId::QueueNotZeroEntryWaitTime => String::from("non-zero entry wait time"),
            &ResultId::QueueTotalWaitTime => String::from("total wait time"),
            &ResultId::EnqueueWaitTime => String::from("enque wait time"),
            &ResultId::DequeueWaitTime => String::from("deque wait time"),
            &ResultId::QueueRate => String::from("queue rate"),
            &ResultId::Facility => String::from("facility"),
            &ResultId::FacilityQueueCount => String::from("queue size"),
            &ResultId::FacilityQueueCountStats => String::from("queue size stats"),
            &ResultId::FacilityTotalWaitTime => String::from("total wait time"),
            &ResultId::FacilityWaitTime => String::from("wait time"),
            &ResultId::FacilityTotalHoldingTime => String::from("total holding time"),
            &ResultId::FacilityHoldingTime => String::from("holding time"),
            &ResultId::FacilityInterrupted => String::from("interrupted?"),
            &ResultId::FacilityCount => String::from("available count"),
            &ResultId::FacilityCountStats => String::from("available count stats"),
            &ResultId::FacilityCaptureCount => String::from("capture count"),
            &ResultId::FacilityUtilCount => String::from("utilization"),
            &ResultId::FacilityUtilCountStats => String::from("utilization stats"),
            &ResultId::Storage => String::from("storage"),
            &ResultId::StorageCapacity => String::from("capacity"),
            &ResultId::StorageEmpty => String::from("completely unused?"),
            &ResultId::StorageFull => String::from("completely used?"),
            &ResultId::StorageQueueCount => String::from("current queue size"),
            &ResultId::StorageQueueCountStats => String::from("queue size stats"),
            &ResultId::StorageTotalWaitTime => String::from("total wait time"),
            &ResultId::StorageWaitTime => String::from("wait time"),
            &ResultId::StorageAverageHoldingTime => String::from("average holding time"),
            &ResultId::StorageContent => String::from("available content"),
            &ResultId::StorageContentStats => String::from("available content stats"),
            &ResultId::StorageUseCount => String::from("total use count"),
            &ResultId::StorageUsedContent => String::from("total used content"),
            &ResultId::StorageUtilCount => String::from("utilization"),
            &ResultId::StorageUtilCountStats => String::from("utilization stats"),
            &ResultId::UserDefined(ref r) => r.title.clone(),
            &ResultId::LocalisedResult(ref r) => {
                r.titles.get(locale)
                    .or_else(|| r.titles.get(&ResultLocale::En))
                    .or_else(|| r.titles.values().next())
                    .unwrap()
                    .clone()
            }
        }
    }
}

/// Convertt the result name to title.
pub fn result_name_into_title(name: ResultName) -> ResultDescription {
    name.split_whitespace()
        .map(move |word| {
            word.split(move |ch: char| { ch.is_uppercase() })
                .map(move |word: &str| { word.to_lowercase() })
                .fold("".to_string(), move |acc, word| { acc + &word })
        })
        .fold("".to_string(), move |acc, words| { acc + &words })
}
