// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::io;
use std::io::Write;
use std::result::Result;
use std::cell::RefCell;
use std::ops::Deref;

use dvcompute::simulation;
use dvcompute::simulation::error::*;
use dvcompute::simulation::Point;
use dvcompute::simulation::Specs;
use dvcompute::simulation::event::*;

#[cfg(feature="cons_mode")]
use dvcompute::simulation::comm::context::LogicalProcessContext;

use crate::simulation::results::*;

/// An utility to convert the vector to a `String` result.
fn into_string_result(w: Vec<u8>) -> simulation::Result<String> {
    match String::from_utf8(w) {
        Result::Ok(x) => Result::Ok(x),
        Result::Err(e) => {
            let msg = format!("{}", e);
            let err = Error::panic(msg);
            Result::Err(err)
        }
    }
}

/// Show the object representation at the specified time point.
#[inline]
pub fn show_result<M>(comp: M, locale: ResultLocale) -> impl Event<Item = String>
    where M: ResultWrite
{
    delay_event(move || {
        let w = Rc::new(RefCell::new(Vec::new()));
        write_result(comp, w.clone(), locale)
            .flat_map(move |()| {
                cons_event(move |_p| {
                    let w = w.replace(Vec::new());
                    into_string_result(w)
                })
            })
    })
}

/// Show the object representation in the integration time points.
#[inline]
pub fn show_results_in_integ_times<F, M>(f: F, locale: ResultLocale) -> impl Simulation<Item = String>
    where F: Fn() -> M + 'static,
          M: ResultWrite + 'static
{
    delay_simulation(move || {
        let w = Rc::new(RefCell::new(Vec::new()));
        write_results_in_integ_times(f, w.clone(), locale)
            .flat_map(move |()| {
                cons_simulation(move |_r| {
                    let w = w.replace(Vec::new());
                    into_string_result(w)
                })
            })
    })
}

/// Show the object representation in the final time point.
#[inline]
pub fn show_result_in_stop_time<M>(comp: M, locale: ResultLocale) -> impl Simulation<Item = String>
    where M: ResultWrite + 'static
{
    delay_simulation(move || {
        let w = Rc::new(RefCell::new(Vec::new()));
        write_result_in_stop_time(comp, w.clone(), locale)
            .flat_map(move |()| {
                cons_simulation(move |_r| {
                    let w = w.replace(Vec::new());
                    into_string_result(w)
                })
            })
    })
}

/// Print the object representation at the specified time point.
#[inline]
pub fn print_result<M>(comp: M, locale: ResultLocale) -> impl Event<Item = ()>
    where M: ResultWrite
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_result(comp, w, locale)
}

/// Print the object representation in the integration time points.
#[inline]
pub fn print_results_in_integ_times<F, M>(f: F, locale: ResultLocale) -> impl Simulation<Item = ()>
    where F: Fn() -> M + 'static,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_results_in_integ_times(f, w, locale)
}

/// Print the object representation in the final time point.
#[inline]
pub fn print_result_in_stop_time<M>(comp: M, locale: ResultLocale) -> impl Simulation<Item = ()>
    where M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_result_in_stop_time(comp, w, locale)
}

/// Write the object representation at the specified time point.
#[inline]
pub fn write_result<M, W>(comp: M, w: Rc<RefCell<W>>, locale: ResultLocale) -> impl Event<Item = ()>
    where M: ResultWrite,
          W: Write
{
    cons_event(move |p| {
        let mut w = w.borrow_mut();
        let w = &mut *w;
        comp.write(w, &locale, p)
    })
}

/// Write the object representation in the integration time points.
pub fn write_results_in_integ_times<F, M, W>(f: F, w: Rc<RefCell<W>>, locale: ResultLocale) -> impl Simulation<Item = ()>
    where F: Fn() -> M + 'static,
          M: ResultWrite + 'static,
          W: Write + 'static
{
    enqueue_io_events_with_integ_times(move || {
        let comp   = f();
        let w      = w.clone();
        let locale = locale.clone();
        cons_event(move |p| {
            let mut w = w.borrow_mut();
            let w = &mut *w;
            comp.write(w, &locale, p)
        })
        .into_boxed()
    })
    .run_in_start_time()
    .flat_map(move |()| {
        return_event(())
            .run_in_stop_time()
    })
}

/// Write the object representation in the final time point.
pub fn write_result_in_stop_time<M, W>(comp: M, w: Rc<RefCell<W>>, locale: ResultLocale) -> impl Simulation<Item = ()>
    where M: ResultWrite + 'static,
          W: Write + 'static
{
    cons_event(move |p| {
        Result::Ok(p.run.specs.stop_time)
    })
    .flat_map(move |t2| {
        enqueue_io_event(t2, {
            cons_event(move |p| {
                let mut w = w.borrow_mut();
                let w = &mut *w;
                comp.write(w, &locale, p)
            })
            .into_boxed()
        })
    })
    .run_in_start_time()
    .flat_map(move |()| {
        return_event(())
            .run_in_stop_time()
    })
}

/// Show the simulation results in the integration time points.
#[cfg(any(feature="seq_mode", feature="wasm_mode"))]
pub fn show_simulation_results_in_integ_times<S, M>(sim: S, specs: Specs, locale: ResultLocale) -> simulation::Result<String>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(Vec::new()));
    write_simulation_results_in_integ_times(sim, specs, w.clone(), locale)?;
    let w = w.replace(Vec::new());
    into_string_result(w)
}

/// Show the simulation results in the integration time points.
#[cfg(feature="cons_mode")]
pub fn show_simulation_results_in_integ_times<S, M>(sim: S, specs: Specs, ctx: &LogicalProcessContext, locale: ResultLocale) -> simulation::Result<String>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(Vec::new()));
    write_simulation_results_in_integ_times(sim, specs, ctx, w.clone(), locale)?;
    let w = w.replace(Vec::new());
    into_string_result(w)
}

/// Show the simulation results in the final time point.
#[cfg(any(feature="seq_mode", feature="wasm_mode"))]
pub fn show_simulation_result_in_stop_time<S, M>(sim: S, specs: Specs, locale: ResultLocale) -> simulation::Result<String>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(Vec::new()));
    write_simulation_result_in_stop_time(sim, specs, w.clone(), locale)?;
    let w = w.replace(Vec::new());
    into_string_result(w)
}

/// Show the simulation results in the final time point.
#[cfg(feature="cons_mode")]
pub fn show_simulation_result_in_stop_time<S, M>(sim: S, specs: Specs, ctx: &LogicalProcessContext, locale: ResultLocale) -> simulation::Result<String>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(Vec::new()));
    write_simulation_result_in_stop_time(sim, specs, ctx, w.clone(), locale)?;
    let w = w.replace(Vec::new());
    into_string_result(w)
}

/// Print the simulation results in the integration time points.
#[cfg(any(feature="seq_mode", feature="wasm_mode"))]
pub fn print_simulation_results_in_integ_times<S, M>(sim: S, specs: Specs, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_simulation_results_in_integ_times(sim, specs, w, locale)
}

/// Print the simulation results in the integration time points.
#[cfg(feature="cons_mode")]
pub fn print_simulation_results_in_integ_times<S, M>(sim: S, specs: Specs, ctx: &LogicalProcessContext, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_simulation_results_in_integ_times(sim, specs, ctx, w, locale)
}

/// Print the simulation results in the final time point.
#[cfg(any(feature="seq_mode", feature="wasm_mode"))]
pub fn print_simulation_result_in_stop_time<S, M>(sim: S, specs: Specs, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_simulation_result_in_stop_time(sim, specs, w, locale)
}

/// Print the simulation results in the final time point.
#[cfg(feature="cons_mode")]
pub fn print_simulation_result_in_stop_time<S, M>(sim: S, specs: Specs, ctx: &LogicalProcessContext, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static
{
    let w = Rc::new(RefCell::new(io::stdout()));
    write_simulation_result_in_stop_time(sim, specs, ctx, w, locale)
}

/// Write the simulation results in the integration time points.
#[cfg(any(feature="seq_mode", feature="wasm_mode"))]
pub fn write_simulation_results_in_integ_times<S, M, W>(sim: S, specs: Specs, w: Rc<RefCell<W>>, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static,
          W: Write + 'static
{
    sim.flat_map(move |comp| {
        let comp = Rc::new(comp);
        write_results_in_integ_times(move || { comp.clone() }, w, locale)
    })
    .run(specs)
}

/// Write the simulation results in the integration time points.
#[cfg(feature="cons_mode")]
pub fn write_simulation_results_in_integ_times<S, M, W>(sim: S, specs: Specs, ctx: &LogicalProcessContext, w: Rc<RefCell<W>>, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static,
          W: Write + 'static
{
    sim.flat_map(move |comp| {
        let comp = Rc::new(comp);
        write_results_in_integ_times(move || { comp.clone() }, w, locale)
    })
    .run(specs, ctx)
}

/// Write the simulation result in the final time point.
#[cfg(any(feature="seq_mode", feature="wasm_mode"))]
pub fn write_simulation_result_in_stop_time<S, M, W>(sim: S, specs: Specs, w: Rc<RefCell<W>>, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static,
          W: Write + 'static
{
    sim.flat_map(move |comp| {
        write_result_in_stop_time(comp, w, locale)
    })
    .run(specs)
}

/// Write the simulation result in the final time point.
#[cfg(feature="cons_mode")]
pub fn write_simulation_result_in_stop_time<S, M, W>(sim: S, specs: Specs, ctx: &LogicalProcessContext, w: Rc<RefCell<W>>, locale: ResultLocale) -> simulation::Result<()>
    where S: Simulation<Item = M>,
          M: ResultWrite + 'static,
          W: Write + 'static
{
    sim.flat_map(move |comp| {
        write_result_in_stop_time(comp, w, locale)
    })
    .run(specs, ctx)
}

/// Allows writing the `String` representation of the object.
pub trait ResultWrite {

    /// Write the `String` representation at the specified time point.
    fn write<W>(&self, write: &mut W, locale: &ResultLocale, p: &Point) -> simulation::Result<()>
        where W: Write;
}

impl ResultWrite for ResultSet {

    fn write<W>(&self, write: &mut W, locale: &ResultLocale, p: &Point) -> simulation::Result<()>
        where W: Write
    {
        let x1 = ResultSource::from_text(String::from("----------"));
        let x2 = ResultSource::time();

        x1.write(write, locale, p)?;
        x2.write(write, locale, p)?;

        for source in self.get_sources() {
            source.write(write, locale, p)?;
        }

        Result::Ok(())
    }
}

impl ResultWrite for ResultSource {

    fn write<W>(&self, write: &mut W, locale: &ResultLocale, p: &Point) -> simulation::Result<()>
        where W: Write
    {
        write_source(self, write, 0, locale, p)
    }
}

impl<T: ResultWrite> ResultWrite for Rc<T> {

    fn write<W>(&self, write: &mut W, locale: &ResultLocale, p: &Point) -> simulation::Result<()>
        where W: Write
    {
        self.deref().write(write, locale, p)
    }
}

/// Write an indented text representation of the results by the specified source.
fn write_source<W: Write>(source: &ResultSource, w: &mut W, indent: usize, locale: &ResultLocale, p: &Point) -> simulation::Result<()> {
    match source {
        &ResultSource::Item(ref x) => {
            write_source_ex(source, w, indent, x.get_name(), locale, p)
        },
        &ResultSource::Vec(ref x) => {
            write_source_ex(source, w, indent, x.name.clone(), locale, p)
        },
        &ResultSource::Object(ref x) => {
            write_source_ex(source, w, indent, x.name.clone(), locale, p)
        },
        &ResultSource::Separator(ref x) => {
            write_source_ex(source, w, indent, x.text.clone(), locale, p)
        }
    }
}

/// Write an indented and labelled text representation of the results by the specified source.
fn write_source_ex<W: Write>(source: &ResultSource, w: &mut W, indent: usize, label: ResultName, locale: &ResultLocale, p: &Point) -> simulation::Result<()> {
    match source {
        &ResultSource::Item(ref x) => {
            let a = match x.get_string_val() {
                Result::Ok(a) => Result::Ok(a),
                Result::Err(msg) => Result::Err(Error::panic(msg))
            };
            let a = a?;
            let a = (a.data)();
            let a = a.call_event(p)?;
            match write_item(x.deref(), a, w, indent, label, locale, p) {
                Result::Ok(z) => Result::Ok(z),
                Result::Err(err) => Result::Err(Error::io(err))
            }
        },
        &ResultSource::Vec(ref x) => {
            let _ = match write_descr(&x.id, w, indent, label.clone(), locale, p) {
                Result::Ok(z) => Result::Ok(z),
                Result::Err(err) => Result::Err(Error::io(err))
            }?;
            for i in 0..(x.items.len()) {
                let source = &x.items[i];
                let indent = indent + 2;
                let label  = label.clone() + &x.subscript[i];
                write_source_ex(source, w, indent, label, locale, p)?;
            }
            Result::Ok(())
        },
        &ResultSource::Object(ref x) => {
            let _ = match write_descr(&x.id, w, indent, label.clone(), locale, p) {
                Result::Ok(z) => Result::Ok(z),
                Result::Err(err) => Result::Err(Error::io(err))
            }?;
            for i in 0..(x.props.len()) {
                let source = &x.props[i].source;
                let indent = indent + 2;
                let label  = x.props[i].label.clone();
                write_source_ex(source, w, indent, label, locale, p)?;
            }
            Result::Ok(())
        },
        &ResultSource::Separator(_) => {
            let _ = match write_separator(w, indent, label.clone(), p) {
                Result::Ok(z) => Result::Ok(z),
                Result::Err(err) => Result::Err(Error::io(err))
            }?;
            Result::Ok(())
        }
    }
}

/// Write an indented and labelled text representation of the results by the specified item.
fn write_item<W: Write>(item: &dyn ResultItem, rep: String, w: &mut W, indent: usize, label: ResultName, locale: &ResultLocale, _p: &Point) -> io::Result<()> {
    let tab = " ".repeat(indent);
    write!(w, "{}", tab)?;
    write!(w, "// ")?;
    writeln!(w, "{}", item.get_id().get_descr(locale))?;
    write!(w, "{}", tab)?;
    write!(w, "{}", label)?;
    write!(w, " = ")?;
    writeln!(w, "{}", rep)?;
    writeln!(w, "")?;
    Result::Ok(())
}

/// Write the identifier descriptor as indented and labelled.
fn write_descr<W: Write>(id: &ResultId, w: &mut W, indent: usize, label: ResultName, locale: &ResultLocale, _p: &Point) -> io::Result<()> {
    let tab = " ".repeat(indent);
    write!(w, "{}", tab)?;
    write!(w, "// ")?;
    writeln!(w, "{}", id.get_descr(locale))?;
    write!(w, "{}", tab)?;
    write!(w, "{}", label)?;
    writeln!(w, ":")?;
    writeln!(w, "")?;
    Result::Ok(())
}

/// Write the separator as indented and labelled.
fn write_separator<W: Write>(w: &mut W, indent: usize, label: ResultName, _p: &Point) -> io::Result<()> {
    let tab = " ".repeat(indent);
    write!(w, "{}", tab)?;
    write!(w, "{}", label)?;
    writeln!(w, "")?;
    writeln!(w, "")?;
    Result::Ok(())
}
