// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::convert::TryInto;

use dvcompute_network::network::*;

use libc::*;

#[cfg_attr(windows, link(name = "dvcompute_mpi.dll"))]
#[cfg_attr(not(windows), link(name = "dvcompute_mpi"))]
extern {

    /// Get the MPI support.
    #[doc(hidden)]
    pub fn create_extern_mpi_network_support(create_buf: unsafe extern "C" fn(size: c_ulong) -> NetworkBuffer) -> NetworkSupport;

    /// Get the MPI support by the specified run index.
    #[doc(hidden)]
    pub fn create_extern_mpi_network_support_by_index(create_buf: unsafe extern "C" fn(size: c_ulong) -> NetworkBuffer, run_index: c_int) -> NetworkSupport;
}

/// Get the MPI support.
pub fn create_mpi_network_support() -> NetworkSupport {
    unsafe {
        create_extern_mpi_network_support(create_buf)
    }
}

/// Get the MPI support by the specified run index.
pub fn create_mpi_network_support_by_index(run_index: usize) -> NetworkSupport {
    let run_index: c_int = run_index.try_into().expect("Expected the integer run index.");
    unsafe {
        create_extern_mpi_network_support_by_index(create_buf, run_index)
    }
}

/// Create a network buffer by the specified size.
unsafe extern "C" fn create_buf(size: c_ulong) -> NetworkBuffer {
    let size = size as usize;
    let mut vec = Vec::with_capacity(size);
    vec.resize(size, 0);
    NetworkBuffer::new(vec)
}
