// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::ops::Deref;

use dvcompute_dist::simulation::Point;
use dvcompute_dist::simulation::ref_comp::*;
use dvcompute_dist::simulation::strategy::QueueStorage;

use dvcompute_utils::collections::im::*;

/// The FCFS queue storage.
pub struct FCFSStorage<T> {

    /// The underlying queue.
    queue: RefComp<IntMap<isize, Queue<T>>>,
}

impl<T> FCFSStorage<T>
    where T: Clone
{
    /// Create a new storage.
    pub fn new() -> Self {
        FCFSStorage { queue: RefComp::new(IntMap::empty()) }
    }
}

impl<T> QueueStorage for FCFSStorage<T>
    where T: Clone + 'static
{
    type Priority = isize;
    type Item     = T;

    #[inline]
    fn is_empty(&self, p: &Point) -> bool {
        let queue = self.queue.read_at(p);
        queue.is_empty()
    }

    fn pop(&self, p: &Point) -> Option<Self::Item> {
        let results = {
            let queue = self.queue.read_at(p);
            match queue.first_key_value() {
                None => None,
                Some((key, deque)) => {
                    match deque.pop_front() {
                        None => {
                            panic!("Inconsistent queue storage")
                        },
                        Some((item, deque)) => {
                            Some((*key, item, deque))
                        }
                    }
                }
            }
        };
        match results {
            None => None,
            Some((key, item, deque)) => {
                let mut queue = self.queue.read_at(p);
                if deque.is_empty() {
                    queue = queue.remove(&key);
                } else {
                    queue = queue.insert(key, deque);
                }
                self.queue.write_at(queue, p);
                Some(item)
            }
        }
    }

    fn push(&self, _item: Self::Item, _p: &Point) {
        panic!("Not supported operation")
    }

    fn push_with_priority(&self, priority: Self::Priority, item: Self::Item, p: &Point) {
        let key = - priority;
        let deque_to_insert = {
            let queue = self.queue.read_at(p);
            match queue.get(&key) {
                None => {
                    let deque = Queue::empty();
                    let deque = deque.push_back(item);
                    Some(deque)
                },
                Some(deque) => {
                    let deque = deque.push_back(item);
                    Some(deque)
                }
            }
        };
        match deque_to_insert {
            None => {},
            Some(deque) => {
                let mut queue = self.queue.read_at(p);
                queue = queue.insert(key, deque);
                self.queue.write_at(queue, p);
            }
        }
    }

    fn remove_by<F>(&self, predicate: F, p: &Point) -> Option<Self::Item>
        where F: Fn(&Self::Item) -> bool + Clone
    {
        match self.remove_and_find_key(predicate, p) {
            None => None,
            Some((key, item, deque)) => {
                let mut queue = self.queue.read_at(p);
                if deque.is_empty() {
                    queue = queue.remove(&key);
                } else {
                    queue = queue.insert(key, deque);
                }
                self.queue.write_at(queue, p);
                Some(item)
            }
        }
    }

    fn exists<F>(&self, predicate: F, p: &Point) -> bool
        where F: Fn(&Self::Item) -> bool + Clone
    {
        self.exists_impl(predicate, p)
    }

    fn find<F>(&self, predicate: F, p: &Point) -> Option<Self::Item>
        where F: Fn(&Self::Item) -> bool + Clone,
              Self::Item: Clone
    {
        self.find_impl(predicate, p)
    }
}

impl<T: 'static> FCFSStorage<T> {

    /// Remove an item satisfying the predicate and return the corresponding key and new dequeue.
    fn remove_and_find_key<F>(&self, predicate: F, p: &Point) -> Option<(isize, T, Queue<T>)>
        where F: Fn(&T) -> bool + Clone,
              T: Clone
    {
        let queue = self.queue.read_at(p);
        let mut f = |key: &isize, deque: &Queue<T>| {
            match deque.remove_by(predicate.clone()) {
                None => None,
                Some((item, deque)) => {
                    return Some((*key, item, deque))
                }
            }
        };
        queue.find(&mut f)
    }

    /// Test whether there is an item satisfying the predicate.
    fn exists_impl<F>(&self, predicate: F, p: &Point) -> bool
        where F: Fn(&T) -> bool + Clone,
              T: Clone
    {
        let queue = self.queue.read_at(p);
        let mut f = |_: &isize, deque: &Queue<T>| {
            match deque.exists(predicate.clone()) {
                false => None,
                true => Some(())
            }
        };
        queue.find(&mut f).is_some()
    }

    /// Find an item satisfying the predicate.
    fn find_impl<F>(&self, predicate: F, p: &Point) -> Option<T>
        where F: Fn(&T) -> bool + Clone,
              T: Clone
    {
        let queue = self.queue.read_at(p);
        let mut f = |_: &isize, deque: &Queue<T>| {
            match deque.find(predicate.clone()) {
                None => None,
                y@Some(_) => y
            }
        };
        queue.find(&mut f)
    }
}

/// The LCFS queue storage.
pub struct LCFSStorage<T> {

    /// The underlying queue.
    queue: RefComp<IntMap<isize, List<T>>>,
}

impl<T> LCFSStorage<T>
    where T: Clone
{
    /// Create a new storage.
    pub fn new() -> Self {
        LCFSStorage { queue: RefComp::new(IntMap::empty()) }
    }
}

impl<T> QueueStorage for LCFSStorage<T>
    where T: Clone + 'static
{
    type Priority = isize;
    type Item     = T;

    #[inline]
    fn is_empty(&self, p: &Point) -> bool {
        let queue = self.queue.read_at(p);
        queue.is_empty()
    }

    fn pop(&self, p: &Point) -> Option<Self::Item> {
        let results = {
            let queue = self.queue.read_at(p);
            match queue.first_key_value() {
                None => None,
                Some((key, deque)) => {
                    match &deque {
                        &List::Nil => {
                            panic!("Inconsistent queue storage")
                        },
                        &List::Cons(ref head, ref tail) => {
                            let tail = tail.deref();
                            Some((*key, head.clone(), tail.clone()))
                        }
                    }
                }
            }
        };
        match results {
            None => None,
            Some((key, item, deque)) => {
                let mut queue = self.queue.read_at(p);
                if deque.is_empty() {
                    queue = queue.remove(&key);
                } else {
                    queue = queue.insert(key, deque);
                }
                self.queue.write_at(queue, p);
                Some(item)
            }
        }
    }

    fn push(&self, _item: Self::Item, _p: &Point) {
        panic!("Not supported operation")
    }

    fn push_with_priority(&self, priority: Self::Priority, item: Self::Item, p: &Point) {
        let key = - priority;
        let deque_to_insert = {
            let queue = self.queue.read_at(p);
            match queue.get(&key) {
                None => {
                    let deque = List::Cons(item, Rc::new(List::Nil));
                    Some(deque)
                },
                Some(deque) => {
                    let deque = List::Cons(item, Rc::new(deque.clone()));
                    Some(deque)
                }
            }
        };
        match deque_to_insert {
            None => {},
            Some(deque) => {
                let mut queue = self.queue.read_at(p);
                queue = queue.insert(key, deque);
                self.queue.write_at(queue, p);
            }
        }
    }

    fn remove_by<F>(&self, _predicate: F, _p: &Point) -> Option<Self::Item>
        where F: Fn(&Self::Item) -> bool + Clone
    {
        panic!("Not supported operation")
    }

    fn exists<F>(&self, _predicate: F, _p: &Point) -> bool
        where F: Fn(&Self::Item) -> bool + Clone
    {
        panic!("Not supported operation")
    }

    fn find<F>(&self, _predicate: F, _p: &Point) -> Option<Self::Item>
        where F: Fn(&Self::Item) -> bool + Clone,
              Self::Item: Clone
    {
        panic!("Not supported operation")
    }
}
