// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;

use crate::simulation::block::*;
use crate::simulation::block::generator::*;

/// It represents the source of `Block` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct BlockFn<I, O> {
    gen: Rc<dyn Fn() -> BlockBox<I, O>>
}

impl<I, O> BlockFn<I, O> {

    /// Create a new source of computations.
    #[inline]
    pub fn new<F, M>(f: F) -> Self
        where F: Fn() -> M + 'static,
              M: Block<Input = I, Output = O> + 'static
    {
        BlockFn {
            gen: Rc::new(move || { f().into_boxed() })
        }
    }

    /// Get the next computation.
    #[inline]
    pub fn next(&self) -> BlockBox<I, O> {
        (self.gen)()
    }

    /// Bind the current computation with its continuation within the resulting computation.
    #[inline]
    pub fn and_then<O2>(&self, other: &BlockFn<O, O2>) -> BlockFn<I, O2>
        where I: 'static,
              O: 'static,
              O2: 'static
    {
        let gen  = self.gen.clone();
        let gen2 = other.gen.clone();
        BlockFn::new(move || {
            gen().and_then(gen2())
        })
    }
}

/// It represents the source of `GeneratorBlock` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct GeneratorBlockFn<T> {
    gen: Rc<dyn Fn() -> BlockBox<BlockFn<T, ()>, ()>>
}

impl<T> GeneratorBlockFn<T> {

    /// Create a new source of computations.
    #[inline]
    pub fn new<F, M>(f: F) -> Self
        where F: Fn() -> M + 'static,
              M: GeneratorBlock<T> + 'static
    {
        GeneratorBlockFn {
            gen: Rc::new(move || { f().into_boxed() })
        }
    }

    /// Get the next computation.
    #[inline]
    pub fn next(&self) -> BlockBox<BlockFn<T, ()>, ()> {
        (self.gen)()
    }
}
