// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::time::Duration;
use std::ptr;
use std::ffi::CString;
use std::ffi::CStr;

use libc::*;

/// A context of the logical process.
pub type LogicalProcessContext = c_void;

/// The threading mode used for simulation.
#[derive(Clone)]
pub enum LogicalProcessThread {

  /// This is the default single threaded mode that corresponds to `MPI_THREAD_SINGLE`.
  SingleThread,

  /// This is the two threaded mode that corresponds to `MPI_THREAD_FUNNELED`.
  FunneledThread
}

/// The logical process parameters.
#[derive(Clone)]
pub struct LogicalProcessParameters {

    /// The logical process name.
    pub name: String,

    /// The time horizon in modeling time units.
    pub time_horizon: Option<f64>,

    /// The undoable log size threshold used for detecting an overflow.
    pub undoable_log_size_threshold: usize,

    /// The output message queue size threshold used for detecting an overflow.
    pub output_message_queue_size_threshold: usize,

    /// The transient message queue size threshold used for detecting an overflow.
    pub transient_message_queue_size_threshold: usize,

    /// The timeout used for synchronising the operations.
    pub sync_timeout: Duration,

    /// Whether the automatic reconnecting to processes is enabled when occurring IO errors (unstable)
    #[doc(hidden)]
    pub reconnecting_enabled: bool,

    /// The timeout used when intiailizing the logical process.
    pub initializing_timeout: Duration,

    /// The idle loop interval when processing the messages.
    pub idle_loop_interval: Option<Duration>,

    /// The test interval for network messages.
    pub network_test_interval: Duration,

    /// The threshold of network requests to test.
    pub network_test_threshold: usize,

    /// The threshold of network requests to wait.
    pub network_wait_threshold: usize,

    /// The thread mode used for running the logical process.
    pub thread_mode: LogicalProcessThread
}

impl LogicalProcessParameters {

    /// Create new logical process parameters.
    pub fn new() -> LogicalProcessParameters {
        LogicalProcessParameters {
            name: String::from("LP"),
            time_horizon: None,
            sync_timeout: Duration::from_secs(60),
            undoable_log_size_threshold: 10000000,
            output_message_queue_size_threshold: 10000,
            transient_message_queue_size_threshold: 5000,
            reconnecting_enabled: false,
            initializing_timeout: Duration::from_secs(10),
            idle_loop_interval: Some(Duration::from_micros(1)),
            network_test_interval: Duration::from_millis(10),
            network_test_threshold: 256,
            network_wait_threshold: 8192,
            thread_mode: LogicalProcessThread::SingleThread
        }
    }
}

/// A C-friendly representation of the logical process parameters.
#[repr(C)]
pub struct LogicalProcessParametersRepr {

    /// Delete the representation.
    delete_repr: unsafe extern "C" fn(*mut LogicalProcessParametersRepr),

    /// The logical process name.
    name: *mut c_char,

    /// Delete the logical process name.
    delete_name: unsafe extern "C" fn(*mut c_char),

    /// The time horizon in modeling time units if defined.
    time_horizon: f64,

    /// A flag indicating whether the time horizon is defined.
    time_horizon_defined: c_int,

    /// The undoable log size threshold used for detecting an overflow.
    undoable_log_size_threshold: usize,

    /// The output message queue size threshold used for detecting an overflow.
    output_message_queue_size_threshold: usize,

    /// The transient message queue size threshold used for detecting an overflow.
    transient_message_queue_size_threshold: usize,

    /// The timeout used for synchronising the operations.
    sync_timeout: [u8; 16],

    /// Whether the automatic reconnecting to processes is enabled when occurring IO errors (unstable)
    #[doc(hidden)]
    reconnecting_enabled: c_int,

    /// The timeout used when intiailizing the logical process.
    initializing_timeout: [u8; 16],

    /// The idle loop interval when processing the messages.
    idle_loop_interval: [u8; 16],

    /// A flag indicating whether the idle loop interval is defined.
    idle_loop_interval_defined: c_int,

    /// The test interval for network messages.
    network_test_interval: [u8; 16],

    /// The threshold of network requests to test.
    network_test_threshold: usize,

    /// The threshold of network requests to wait.
    network_wait_threshold: usize,

    /// The thread mode used for running the logical process.
    thread_mode: c_int
}

impl LogicalProcessParametersRepr {

    /// Convert to a C-friendly representation.
    pub fn new(ps: LogicalProcessParameters) -> Self {
        let name = CString::new(ps.name.clone()).expect("NulError");
        let name = CString::into_raw(name);
        LogicalProcessParametersRepr {
            delete_repr: delete_lp_repr,
            name: name,
            delete_name: delete_lp_name,
            time_horizon: ps.time_horizon.unwrap_or(0.0),
            time_horizon_defined: if ps.time_horizon.is_some() { 1 } else { 0 },
            undoable_log_size_threshold: ps.undoable_log_size_threshold,
            output_message_queue_size_threshold: ps.output_message_queue_size_threshold,
            transient_message_queue_size_threshold: ps.transient_message_queue_size_threshold,
            sync_timeout: u128::to_ne_bytes(ps.sync_timeout.as_nanos()),
            reconnecting_enabled: if ps.reconnecting_enabled { 1 } else { 0 },
            initializing_timeout: u128::to_ne_bytes(ps.initializing_timeout.as_nanos()),
            idle_loop_interval: u128::to_ne_bytes(ps.idle_loop_interval.unwrap_or(Duration::from_micros(3)).as_nanos()),
            idle_loop_interval_defined: if ps.idle_loop_interval.is_some() { 1 } else { 0 },
            network_test_interval: u128::to_ne_bytes(ps.network_test_interval.as_nanos()),
            network_test_threshold: ps.network_test_threshold,
            network_wait_threshold: ps.network_wait_threshold,
            thread_mode: {
                match ps.thread_mode {
                    LogicalProcessThread::SingleThread => 0,
                    LogicalProcessThread::FunneledThread => 1
                }
            }
        }
    }
}

impl Drop for LogicalProcessParametersRepr {

    fn drop(&mut self) {
        unsafe {
            (self.delete_name)(self.name);
        }
    }
}

unsafe extern "C" fn delete_lp_name(name: *mut c_char) {
    if name != ptr::null_mut() {
        let _ = CString::from_raw(name);
    }
}

unsafe extern "C" fn delete_lp_repr(e: *mut LogicalProcessParametersRepr) {
    if e != ptr::null_mut() {
        let _ = Box::from_raw(e);
    }
}

/// Convert the `LogicalProcessParametersRepr` referenced to by the FFI pointer to an `LogicalProcessParameters`.
pub unsafe fn ffi_logical_process_parameters_repr_into(ps: *mut LogicalProcessParametersRepr) -> LogicalProcessParameters {
    let name = CStr::from_ptr((*ps).name);
    let name = name.to_bytes().to_vec();
    let name = String::from_utf8(name).expect("FromUtf8Error");
    let xs   = LogicalProcessParameters {
        name: name,
        time_horizon: if (*ps).time_horizon_defined != 0 { Some((*ps).time_horizon) } else { None },
        undoable_log_size_threshold: (*ps).undoable_log_size_threshold,
        output_message_queue_size_threshold: (*ps).output_message_queue_size_threshold,
        transient_message_queue_size_threshold: (*ps).transient_message_queue_size_threshold,
        sync_timeout: Duration::from_nanos(u128::from_ne_bytes((*ps).sync_timeout) as u64),
        reconnecting_enabled: (*ps).reconnecting_enabled != 0,
        initializing_timeout: Duration::from_nanos(u128::from_ne_bytes((*ps).initializing_timeout) as u64),
        idle_loop_interval: if (*ps).idle_loop_interval_defined != 0 { Some(Duration::from_nanos(u128::from_ne_bytes((*ps).idle_loop_interval) as u64)) } else { None },
        network_test_interval: Duration::from_nanos(u128::from_ne_bytes((*ps).network_test_interval) as u64),
        network_test_threshold: (*ps).network_test_threshold,
        network_wait_threshold: (*ps).network_wait_threshold,
        thread_mode: {
            if (*ps).thread_mode != 0 { LogicalProcessThread::FunneledThread } else { LogicalProcessThread::SingleThread }
        }
    };
    ((*ps).delete_repr)(ps);
    xs
}
