// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;

/// An immutable priority queue as described in book
/// "Algorithms: a functional programming approach" by
/// Fethi Rabhi and Guy Lapalme.
#[derive(Clone)]
pub enum PriorityQueue<K, V> {

    /// The queue item.
    QueueItem(Rc<PriorityQueueItem<K, V>>),

    /// The empty queue.
    EmptyQueue
}

/// An immutable queue item.
#[derive(Clone)]
pub struct PriorityQueueItem<K, V> {

    /// The queue size.
    len: usize,

    /// The item key.
    key: K,

    /// The item value.
    val: V,

    /// The item rank.
    rank: usize,

    /// The left branch.
    left: PriorityQueue<K, V>,

    /// The right branch.
    right: PriorityQueue<K, V>,
}

impl<K, V> PriorityQueue<K, V> where K: PartialOrd + Copy, V: Clone {

    /// Create a new priority queue.
    #[inline]
    pub fn new() -> Self {
        PriorityQueue::EmptyQueue
    }

    /// The size of the queue.
    #[inline]
    pub fn len(&self) -> usize {
        match self {
            PriorityQueue::QueueItem(ref item) => item.len,
            PriorityQueue::EmptyQueue => 0
        }
    }

    /// Test whether the queue is empty.
    #[inline]
    pub fn is_empty(&self) -> bool {
        match self {
            PriorityQueue::QueueItem(_) => false,
            PriorityQueue::EmptyQueue => true
        }
    }

    /// Enqueue a new item with the specified key and value.
    #[inline]
    pub fn enqueue(&self, key: K, val: V) -> Self {
        let item = PriorityQueueItem {
            len: 1,
            key: key,
            val: val,
            rank: 1,
            left: PriorityQueue::EmptyQueue,
            right: PriorityQueue::EmptyQueue
        };
        let queue = PriorityQueue::QueueItem(Rc::new(item));
        self.merge(&queue)
    }

    /// Dequeue the top element with minimal key.
    #[inline]
    pub fn dequeue(&self) -> Self {
        match self {
            PriorityQueue::QueueItem(ref item) => {
                item.left.merge(&item.right)
            },
            PriorityQueue::EmptyQueue => {
                panic!("The queue is empty!")
            }
        }
    }

    /// Return the minimal key if the priority queue is not empty.
    #[inline]
    pub fn front_key(&self) -> Option<&K> {
        match self {
            PriorityQueue::QueueItem(ref item) => Some(&item.key),
            PriorityQueue::EmptyQueue => None
        }
    }

    /// Return the front value if the priority queue is not empty.
    #[inline]
    pub fn front_val(&self) -> Option<&V> {
        match self {
            PriorityQueue::QueueItem(ref item) => Some(&item.val),
            PriorityQueue::EmptyQueue => None
        }
    }

    /// Return the front key and value if the priority queue is not empty.
    #[inline]
    pub fn front(&self) -> Option<(&K, &V)> {
        match self {
            PriorityQueue::QueueItem(ref item) => Some((&item.key, &item.val)),
            PriorityQueue::EmptyQueue => None
        }
    }

    /// Return the queue rank.
    #[inline]
    fn rank(&self) -> usize {
        match self {
            PriorityQueue::QueueItem(ref item) => item.rank,
            PriorityQueue::EmptyQueue => 0
        }
    }

    /// Construct a new priority queue.
    fn make(key: K, val: V, left: Self, right: Self) -> Self {
        let len = left.len() + right.len() + 1;
        if left.rank() >= right.rank() {
            let item = PriorityQueueItem {
                len: len,
                key: key,
                val: val,
                rank: right.rank() + 1,
                left: left,
                right: right
            };
            PriorityQueue::QueueItem(Rc::new(item))
        } else {
            let item = PriorityQueueItem {
                len: len,
                key: key,
                val: val,
                rank: left.rank() + 1,
                left: right,
                right: left
            };
            PriorityQueue::QueueItem(Rc::new(item))
        }
    }

    /// Merge two queues.
    fn merge(&self, other: &Self) -> Self {
        match (self, other) {
            (PriorityQueue::QueueItem(ref r1), PriorityQueue::QueueItem(ref r2)) => {
                if r1.key <= r2.key {
                    let left  = r1.left.clone();
                    let right = r1.right.merge(other);
                    PriorityQueue::make(r1.key, r1.val.clone(), left, right)
                } else {
                    let left  = r2.left.clone();
                    let right = self.merge(&r2.right);
                    PriorityQueue::make(r2.key, r2.val.clone(), left, right)
                }
            },
            (PriorityQueue::QueueItem(ref r1), PriorityQueue::EmptyQueue) => {
                PriorityQueue::QueueItem(r1.clone())
            },
            (PriorityQueue::EmptyQueue, PriorityQueue::QueueItem(ref r2)) => {
                PriorityQueue::QueueItem(r2.clone())
            },
            (PriorityQueue::EmptyQueue, PriorityQueue::EmptyQueue) => {
                PriorityQueue::EmptyQueue
            }
        }
    }
}
