// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::cell::UnsafeCell;
use std::cell::RefCell;
use std::ops::Deref;

use crate::simulation;
use crate::simulation::point::Point;
use crate::simulation::event::Event;

use dvcompute_utils::collections::im::IntMap;

/// A mutable cell computation synchronized with the event queue.
pub struct RefComp<T> {

    /// The initial item.
    init_item: UnsafeCell<T>,

    /// The items to store.
    items: UnsafeCell<Option<Rc<RefCell<IntMap<u64, RefCell<T>>>>>>
}

impl<T> RefComp<T> {

    /// Create a new mutable cell by the initial value.
    #[inline]
    pub fn new(val: T) -> RefComp<T> {
        RefComp {
            init_item: UnsafeCell::new(val),
            items: UnsafeCell::new(None)
        }
    }

    /// Read the contents of the mutable cell at the specified time point.
    pub fn read_at(&self, p: &Point) -> T
        where T: Clone + 'static
    {
        let mut branch = &p.run.branch;
        loop {
            match &branch.parent {
                &None => {
                    let p = self.init_item.get();
                    unsafe {
                        return (*p).clone();
                    }
                },
                &Some(ref parent) => {
                    let items = self.items.get();
                    unsafe {
                        match &(*items) {
                            &None => {
                                let p = self.init_item.get();
                                return (*p).clone();
                            },
                            &Some(ref items) => {
                                let items0 = items.deref().borrow();
                                match items0.get(&branch.id) {
                                    None => {
                                        branch = parent;
                                    },
                                    Some(ref item) => {
                                        let p = item.deref().borrow();
                                        return (*p).clone()
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    /// Read the contents of the mutable cell within the `Event` computation.
    #[inline]
    pub fn read(comp: Rc<Self>) -> Read<T>
        where T: Clone + 'static
    {
        Read { comp: comp }
    }

    /// Swap the contents of the mutable cell at the specified time point.
    #[inline]
    pub fn swap_at(&self, val: T, p: &Point) -> T
        where T: Clone + 'static
    {
        let tmp = self.read_at(p);
        self.write_at(val, p);
        tmp
    }

    /// Swap the contents of the mutable cell within the `Event` computation.
    #[inline]
    pub fn swap(comp: Rc<Self>, val: T) -> Swap<T>
        where T: Clone + 'static
    {
        Swap { comp: comp, val: val }
    }

    /// Write into the contents of the mutable cell at the specified time point.
    pub fn write_at(&self, val: T, p: &Point)
        where T: Clone + 'static
    {
        let branch = &p.run.branch;
        if branch.parent.is_none() {
            let p = self.init_item.get();
            unsafe {
                *p = val;
            }
        } else {
            let items = self.items.get();
            unsafe {
                match &(*items) {
                    &None => {
                        let items0 = Rc::new(RefCell::new(IntMap::empty()));
                        (*items).replace(items0.clone());
                        create_at(items0, branch.id, val, p);
                    },
                    &Some(ref items) => {
                        let items0 = items.deref().borrow();
                        match items0.get(&branch.id) {
                            None => {
                                drop(items0);
                                create_at(items.clone(), branch.id, val, p);
                            },
                            Some(ref item) => {
                                let mut p = item.deref().borrow_mut();
                                (*p) = val;
                            }
                        }
                    }
                }
            }
        }
    }

    /// Write into the contents of the mutable cell within the `Event` computation.
    #[inline]
    pub fn write(comp: Rc<Self>, val: T) -> Write<T>
        where T: Clone + 'static
    {
        Write { comp: comp, val: val }
    }

    /// Mutate the contents of the mutable cell at the specified time point.
    #[inline]
    pub fn modify_at<F>(&self, f: F, p: &Point)
        where F: FnOnce(&T) -> T,
              T: Clone + 'static
    {
        self.write_at(f(&self.read_at(p)), p);
    }

    /// Mutate the contents of the mutable cell within the `Event` computation.
    #[inline]
    pub fn modify<F>(comp: Rc<Self>, f: F) -> Modify<T, F>
        where F: FnOnce(&T) -> T,
              T: Clone + 'static
    {
        Modify { comp: comp, f: f }
    }
}

/// Create a new entry in the items.
fn create_at<T>(items: Rc<RefCell<IntMap<u64, RefCell<T>>>>, branch_id: u64, val: T, p: &Point)
    where T: Clone + 'static
{
    {
        let mut items = items.deref().borrow_mut();
        *items = (*items).insert(branch_id, RefCell::new(val));
    }
    let mut fins = p.run.finalizers.borrow_mut();
    (*fins).push(Box::new(move || {
        let mut items = items.deref().borrow_mut();
        *items = (*items).remove(&branch_id);
    }));
}

impl<T> PartialEq for RefComp<T> {

    fn eq(&self, other: &Self) -> bool {
        self.init_item.get() == other.init_item.get()
    }
}

impl<T> Eq for RefComp<T> {}

impl<T: Clone> Clone for RefComp<T> {

    fn clone(&self) -> Self {
        let init_item = unsafe {
            let p = self.init_item.get();
            (*p).clone()
        };
        let items = unsafe {
            let p = self.items.get();
            (*p).clone()
        };
        Self {
            init_item: UnsafeCell::new(init_item),
            items: UnsafeCell::new(items)
        }
    }
}

/// It reads the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Read<T> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>
}

impl<T: Clone + 'static> Event for Read<T> {

    type Item = T;

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Read { comp } = self;
        Result::Ok(comp.read_at(p))
    }
}

/// It swaps the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Swap<T> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>,

    /// A new value.
    val: T
}

impl<T: Clone + 'static> Event for Swap<T> {

    type Item = T;

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Swap { comp, val } = self;
        Result::Ok(comp.swap_at(val, p))
    }
}

/// It writes into the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Write<T> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>,

    /// A new value.
    val: T
}

impl<T: Clone + 'static> Event for Write<T> {

    type Item = ();

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Write { comp, val } = self;
        Result::Ok(comp.write_at(val, p))
    }
}

/// It mutates the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Modify<T, F> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>,

    /// A mutator.
    f: F
}

impl<T, F> Event for Modify<T, F>
    where F: FnOnce(&T) -> T,
          T: Clone + 'static
{
    type Item = ();

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Modify { comp, f } = self;
        Result::Ok(comp.modify_at(f, p))
    }
}
