// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;

use crate::simulation::parameter::*;
use crate::simulation::parameter::random::*;
use crate::simulation::process::*;

/// Hold the process for a random time interval distributed uniformly.
#[inline]
pub fn random_uniform_process(min: f64, max: f64) -> impl Process<Item = f64> + Clone {
    random_uniform_parameter(min, max)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval distributed uniformly.
#[inline]
pub fn random_uniform_process_(min: f64, max: f64) -> impl Process<Item = ()> + Clone {
    random_uniform_parameter(min, max)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for an integer random time interval distributed uniformly.
#[inline]
pub fn random_int_uniform_process(min: isize, max: isize) -> impl Process<Item = isize> + Clone {
    random_int_uniform_parameter(min, max)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt as f64)
                .map(move |()| dt)
        })
}

/// Hold the process for an integer random time interval distributed uniformly.
#[inline]
pub fn random_int_uniform_process_(min: isize, max: isize) -> impl Process<Item = ()> + Clone {
    random_int_uniform_parameter(min, max)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt as f64)
        })
}

/// Hold the process for a random time interval from the triangular distribution.
#[inline]
pub fn random_triangular_process(min: f64, median: f64, max: f64) -> impl Process<Item = f64> + Clone {
    random_triangular_parameter(min, median, max)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval from the triangular distribution.
#[inline]
pub fn random_triangular_process_(min: f64, median: f64, max: f64) -> impl Process<Item = ()> + Clone {
    random_triangular_parameter(min, median, max)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for a random time interval distributed normally.
#[inline]
pub fn random_normal_process(mu: f64, nu: f64) -> impl Process<Item = f64> + Clone {
    random_normal_parameter(mu, nu)
        .into_process()
        .flat_map(|dt| {
            let dt = dt.max(0.0);
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval distributed normally.
#[inline]
pub fn random_normal_process_(mu: f64, nu: f64) -> impl Process<Item = ()> + Clone {
    random_normal_parameter(mu, nu)
        .into_process()
        .flat_map(|dt| {
            let dt = dt.max(0.0);
            hold_process(dt)
        })
}

/// Hold the process for a random time interval distributed lognormally.
#[inline]
pub fn random_log_normal_process(mu: f64, nu: f64) -> impl Process<Item = f64> + Clone {
    random_log_normal_parameter(mu, nu)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval distributed lognormally.
#[inline]
pub fn random_log_normal_process_(mu: f64, nu: f64) -> impl Process<Item = ()> + Clone {
    random_log_normal_parameter(mu, nu)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for an exponential random time interval with the specified mean.
#[inline]
pub fn random_exponential_process(mu: f64) -> impl Process<Item = f64> + Clone {
    random_exponential_parameter(mu)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for an exponential random time interval with the specified mean.
#[inline]
pub fn random_exponential_process_(mu: f64) -> impl Process<Item = ()> + Clone {
    random_exponential_parameter(mu)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for an Erlang random time interval with the specified scale
/// (the reciprocal of the rate) and integer shape.
#[inline]
pub fn random_erlang_process(scale: f64, shape: isize) -> impl Process<Item = f64> + Clone {
    random_erlang_parameter(scale, shape)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for an Erlang random time interval with the specified scale
/// (the reciprocal of the rate) and integer shape.
#[inline]
pub fn random_erlang_process_(scale: f64, shape: isize) -> impl Process<Item = ()> + Clone {
    random_erlang_parameter(scale, shape)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for a Poisson random time interval with the specified mean.
#[inline]
pub fn random_poisson_process(mu: f64) -> impl Process<Item = isize> + Clone {
    random_poisson_parameter(mu)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt as f64)
                .map(move |()| dt)
        })
}

/// Hold the process for a Poisson random time interval with the specified mean.
#[inline]
pub fn random_poisson_process_(mu: f64) -> impl Process<Item = ()> + Clone {
    random_poisson_parameter(mu)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt as f64)
        })
}

/// Hold the process for a binomial random time interval with the specified
/// probability and trials.
#[inline]
pub fn random_binomial_process(prob: f64, trials: isize) -> impl Process<Item = isize> + Clone {
    random_binomial_parameter(prob, trials)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt as f64)
                .map(move |()| dt)
        })
}

/// Hold the process for a binomial random time interval with the specified
/// probability and trials.
#[inline]
pub fn random_binomial_process_(prob: f64, trials: isize) -> impl Process<Item = ()> + Clone {
    random_binomial_parameter(prob, trials)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt as f64)
        })
}

/// Hold the process for a random time interval from the Gamma distribution.
#[inline]
pub fn random_gamma_process(kappa: f64, theta: f64) -> impl Process<Item = f64> + Clone {
    random_gamma_parameter(kappa, theta)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval from the Gamma distribution.
#[inline]
pub fn random_gamma_process_(kappa: f64, theta: f64) -> impl Process<Item = ()> + Clone {
    random_gamma_parameter(kappa, theta)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for a random time interval from the Beta distribution.
#[inline]
pub fn random_beta_process(alpha: f64, beta: f64) -> impl Process<Item = f64> + Clone {
    random_beta_parameter(alpha, beta)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval from the Beta distribution.
#[inline]
pub fn random_beta_process_(alpha: f64, beta: f64) -> impl Process<Item = ()> + Clone {
    random_beta_parameter(alpha, beta)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for a random time interval from the Weibull distribution.
#[inline]
pub fn random_weibull_process(alpha: f64, beta: f64) -> impl Process<Item = f64> + Clone {
    random_weibull_parameter(alpha, beta)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval from the Weibull distribution.
#[inline]
pub fn random_weibull_process_(alpha: f64, beta: f64) -> impl Process<Item = ()> + Clone {
    random_weibull_parameter(alpha, beta)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}

/// Hold the process for a random time interval from the specified discrete distribution.
#[inline]
pub fn random_discrete_process(dpdf: Rc<Vec<(f64, f64)>>) -> impl Process<Item = f64> + Clone {
    random_discrete_parameter(dpdf)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
                .map(move |()| dt)
        })
}

/// Hold the process for a random time interval from the specified discrete distribution.
#[inline]
pub fn random_discrete_process_(dpdf: Rc<Vec<(f64, f64)>>) -> impl Process<Item = ()> + Clone {
    random_discrete_parameter(dpdf)
        .into_process()
        .flat_map(|dt| {
            hold_process(dt)
        })
}
