// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use crate::simulation::Run;

/// The modeling point.
#[cfg(feature="dist_mode")]
#[repr(C)]
#[derive(Clone)]
pub struct Point<'a> {

    /// The current simulation run.
    pub run: &'a Run<'a>,

    /// The current simulation time.
    pub time: f64,

    /// The current simulation time priority.
    pub priority: isize,

    /// The current integration iteration.
    pub iteration: usize,

    /// The integration iteration phase.
    pub phase: i32
}

#[cfg(feature="dist_mode")]
impl<'a> Point<'a> {

    /// Return the integration time point by the specified iteration number.
    #[inline]
    pub fn with_iteration(&self, n: usize) -> Self {
        let run = &self.run;
        let specs = &run.specs;
        let t0 = specs.start_time;
        let t2 = specs.stop_time;
        let dt = specs.dt;
        let n2 = ((t2 - t0) / dt).floor() as usize;
        let t  = if n == n2 { t2 } else { t0 + (n as f64) * dt };
        Point {
            run: run,
            time: t,
            priority: self.priority,
            iteration: n,
            phase: -1
        }
    }

    /// Trace by displaying the specified message.
    pub fn trace(&self, msg: &str) {
        println!("t={}: {}", self.time, msg);
    }
}


/// The modeling point.
#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
#[repr(C)]
#[derive(Clone)]
pub struct Point<'a> {

    /// The current simulation run.
    pub run: &'a Run,

    /// The current simulation time.
    pub time: f64,

    /// The current simulation time priority.
    pub priority: isize,

    /// The current integration iteration.
    pub iteration: usize,

    /// The integration iteration phase.
    pub phase: i32
}

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
impl<'a> Point<'a> {

    /// Return the integration time point by the specified iteration number.
    #[inline]
    pub fn with_iteration(&self, n: usize) -> Self {
        let run = &self.run;
        let specs = &run.specs;
        let t0 = specs.start_time;
        let t2 = specs.stop_time;
        let dt = specs.dt;
        let n2 = ((t2 - t0) / dt).floor() as usize;
        let t  = if n == n2 { t2 } else { t0 + (n as f64) * dt };
        Point {
            run: run,
            time: t,
            priority: self.priority,
            iteration: n,
            phase: -1
        }
    }

    /// Trace by displaying the specified message.
    pub fn trace(&self, msg: &str) {
        println!("t={}: {}", self.time, msg);
    }
}
