// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use std::cell::UnsafeCell;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use std::ops::Deref;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use std::rc::Rc;

#[cfg(feature="dist_mode")]
use libc::*;

use crate::simulation::specs::SpecsRepr;
use crate::simulation::point::Point;
use crate::simulation::event::EventRepr;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use crate::simulation;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use crate::simulation::error::*;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use crate::simulation::event::*;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use crate::simulation::utils::priority_queue::PriorityQueue;

/// The event queue for the optimistic Time Warp method.
#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
pub struct EventQueue {

    /// The current event time.
    time: UnsafeCell<f64>,

    /// The priority queue.
    pq: UnsafeCell<PriorityQueue<(f64, isize), Rc<EventRepr>>>
}

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
impl EventQueue {

    /// Create a new event queue for the sequential simulation.
    pub fn new(specs: &SpecsRepr) -> EventQueue {
        EventQueue {
            time: UnsafeCell::new(specs.start_time),
            pq: UnsafeCell::new(PriorityQueue::new())
        }
    }

    /// Clone the event queue at the specified modeling time point.
    pub fn clone_at(&self, _p: &Point) -> EventQueue {
        let t0 = unsafe {
            *(self.time.get())
        };
        let pq = self.pq.get();
        let pq = unsafe {
            (*pq).clone()
        };
        EventQueue {
            time: UnsafeCell::new(t0),
            pq: UnsafeCell::new(pq)
        }
    }

    /// Enqueue the event by the specified activation time and handler generator.
    #[inline]
    pub fn enqueue_event(&self, event_time: f64, priority: isize, comp: EventRepr, p: &Point) {
        assert!(event_time >= p.time, "The event time cannot be less than the current modeling time");
        let pq = self.pq.get();
        unsafe {
            *pq = (*pq).enqueue((event_time, - priority), Rc::new(comp));
        }
    }

    /// Enqueue the IO-based event by the specified activation time and handler generator.
    #[inline]
    pub fn enqueue_io_event(&self, event_time: f64, priority: isize, comp: EventRepr, p: &Point) {
        self.enqueue_event(event_time, priority, comp, p)
    }

    /// Run the pending events.
    pub fn run_events(&self, including_current: bool, p: &Point) -> simulation::Result<()> {
        loop {
            let t0 = self.time.get();
            let pq = self.pq.get();
            unsafe {
                match (*pq).front_key() {
                    Some((t2, _)) if *t2 < *t0 => {
                        panic!("t={}: The time value ({}) is too small. The event queue is desynchronized", *t0, *t2);
                    }
                    Some((t2, _)) if *t2 < p.time || (including_current && *t2 == p.time) => {
                        *t0 = *t2;
                    }
                    _ => {
                        break;
                    }
                }
            }
            let ((t2, pri2), c2) = unsafe {
                (*pq).front().unwrap()
            };
            let t2 = *t2;
            let c2 = c2.deref().clone();
            let p2 = p.run.point_at(t2, -(*pri2));
            unsafe {
                *pq = (*pq).dequeue()
            }
            match c2.call_event(&p2) {
                Result::Ok(()) => (),
                Result::Err(Error::Cancel) => (),
                Result::Err(Error::Other(x)) => {
                    match x.deref() {
                        &OtherError::Retry(_) => {
                            return Result::Err(Error::Other(x.clone()))
                        },
                        &OtherError::Panic(_) => {
                            return Result::Err(Error::Other(x.clone()))
                        },
                        &OtherError::IO(_) => {
                            return Result::Err(Error::Other(x.clone()))
                        }
                    }
                }
            }
        }
        Result::Ok(())
    }
}

/// Represents the event queue.
#[cfg(feature="dist_mode")]
pub type EventQueue = c_void;

#[cfg(feature="dist_mode")]
#[cfg_attr(windows, link(name = "dvcompute_core_dist.dll"))]
#[cfg_attr(not(windows), link(name = "dvcompute_core_dist"))]
extern {

    /// Create a new event queue.
    #[doc(hidden)]
    pub fn create_extern_event_queue(specs: *const SpecsRepr) -> *mut EventQueue;

    /// Delete the event queue.
    #[doc(hidden)]
    pub fn delete_extern_event_queue(queue: *mut EventQueue);

    /// Enqueue a new event.
    #[doc(hidden)]
    pub fn enqueue_extern_event(queue: *mut EventQueue, event_time: f64, priority: isize, comp: EventRepr, p: *const Point);

    /// Enqueue a new IO event.
    #[doc(hidden)]
    pub fn enqueue_extern_io_event(queue: *mut EventQueue, event_time: f64, priority: isize, comp: EventRepr, p: *const Point);

    /// Synchronize the events.
    #[doc(hidden)]
    pub fn sync_extern_events(including_current: isize, p: *const Point);

    /// Leave the simulation.
    #[doc(hidden)]
    pub fn leave_extern_simulation(queue: *mut EventQueue, p: *const Point);
}
