// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::marker::PhantomData;

#[cfg(feature="dist_mode")]
use std::mem;

#[cfg(feature="dist_mode")]
use std::ptr;

#[cfg(feature="dist_mode")]
use libc::*;

use crate::simulation;
use crate::simulation::Point;
use crate::simulation::Run;
use crate::simulation::simulation::Simulation;
use crate::simulation::process::is_process_cancelled;
use crate::simulation::process::revoke_process;
use crate::simulation::process::revoke_process_boxed;
use crate::simulation::process::Process;
use crate::simulation::process::ProcessId;
use crate::simulation::process::ProcessBoxCont;
use crate::simulation::observable::disposable::*;
use crate::simulation::composite::*;

#[cfg(feature="dist_mode")]
use crate::simulation::error::*;

#[cfg(feature="dist_mode")]
use crate::simulation::internal::event_queue::*;

/// Additional operations.
pub mod ops;

/// Return a new `Event` computation by the specified pure value.
#[inline]
pub fn return_event<T>(val: T) -> Return<T> {
    Return { val: val }
}

/// Delay the `Event` computation.
#[inline]
pub fn delay_event<F, M>(f: F) -> Delay<F, M>
    where F: FnOnce() -> M,
          M: Event
{
    Delay { f: f, _phantom: PhantomData }
}

/// Construct a new `Event` computation by the specified function.
#[inline]
pub fn cons_event<F, T>(f: F) -> Cons<F, T>
    where F: FnOnce(&Point) -> simulation::Result<T>
{
     Cons { f: f, _phantom: PhantomData }
}

/// Enqueue an event which handler should be activated at the specified time.
#[inline]
pub fn enqueue_event(time: f64, comp: EventBox<()>) -> Enqueue {
    Enqueue { time: time, comp: comp }
}

/// Enqueue an event which handler should be activated at the specified time with the given priority.
#[inline]
pub fn enqueue_event_with_priority(time: f64, priority: isize, comp: EventBox<()>) -> EnqueueWithPriority {
    EnqueueWithPriority { time: time, priority: priority, comp: comp }
}

/// Enqueue events which handlers should be activated in the integration time points.
#[inline]
pub fn enqueue_events_with_integ_times<F>(f: F) -> EnqueueWithIntegTimes<F>
    where F: Fn() -> EventBox<()> + Clone + 'static
{
    EnqueueWithIntegTimes { f: f }
}

/// Enqueue an IO-based event which handler should be activated at the specified time.
#[inline]
pub fn enqueue_io_event(time: f64, comp: EventBox<()>) -> EnqueueIO {
    EnqueueIO { time: time, comp: comp }
}

/// Enqueue an IO-based event which handler should be activated at the specified time with the given priority.
#[inline]
pub fn enqueue_io_event_with_priority(time: f64, priority: isize, comp: EventBox<()>) -> EnqueueIOWithPriority {
    EnqueueIOWithPriority { time: time, priority: priority, comp: comp }
}

/// Enqueue IO-based events which handlers should be activated in the integration time points.
#[inline]
pub fn enqueue_io_events_with_integ_times<F>(f: F) -> EnqueueIOWithIntegTimes<F>
    where F: Fn() -> EventBox<()> + Clone + 'static
{
    EnqueueIOWithIntegTimes { f: f }
}

/// Enqueue the event which must be actuated with the current modeling time but later.
#[inline]
pub fn yield_event<M>(comp: M) -> Yield<M>
    where M: Event<Item = ()> + Clone + 'static
{
    Yield { comp: comp }
}

/// Return the current modeling time.
#[inline]
pub fn time_event() -> Time {
    Time {}
}

/// Return the current modeling time priority.
#[inline]
pub fn priority_event() -> Priority {
    Priority {}
}

/// Create a sequence of computations.
#[inline]
pub fn event_sequence<I, M>(comps: I) -> Sequence<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Event
{
    Sequence { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Create a sequence of computations, where the result is ignored.
#[inline]
pub fn event_sequence_<I, M>(comps: I) -> Sequence_<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Event
{
    Sequence_ { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Trace the computation.
#[inline]
pub fn trace_event<M>(msg: String, comp: M) -> Trace<M>
    where M: Event
{
    Trace { comp: comp, msg: msg}
}

/// Try to leave the simulation.
///
/// It enqueues a new event with obligatory synchronization with the global virtual time.
/// If that event will not be reverted during rollback, then the logical process will
/// leave the simulation immediately after approaching the final modeling time without
/// synchronization with the global virtual time anymore if it has no unacknowledged
/// sent messages.
///
/// It makes sense to use this action if logical processes can enter and leave your long-running
/// distributed simulation by demand.
#[cfg(feature="dist_mode")]
#[inline]
pub fn leave_simulation() -> LeaveSimulation {
    LeaveSimulation {}
}

/// The computation synchronized with the event queue.
pub trait Event {

    /// The type of the item that is returned in the current modeling time.
    type Item;

    /// Call the `Event` computation in the current modeling time.
    #[doc(hidden)]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item>;

    /// Convert to the `Process` computation.
    fn into_process(self) -> EventIntoProcess<Self>
        where Self: Sized
    {
        EventIntoProcess { comp: self }
    }

    /// Convert to the `Composite` computation.
    fn into_composite(self) -> EventIntoComposite<Self>
        where Self: Sized
    {
        EventIntoComposite { comp: self }
    }

    /// Bind the current computation with its continuation within the resulting computation.
    #[inline]
    fn flat_map<U, F>(self, f: F) -> FlatMap<Self, U, F>
        where Self: Sized,
              U: Event,
              F: FnOnce(Self::Item) -> U,
    {
        FlatMap { comp: self, f: f, _phantom: PhantomData }
    }

    /// Map the current computation using the specified transform.
    #[inline]
    fn map<B, F>(self, f: F) -> Map<Self, B, F>
        where Self: Sized,
              F: FnOnce(Self::Item) -> B,
    {
        Map { comp: self, f: f, _phantom: PhantomData }
    }

    /// Zip the current computation with another one within the resulting computation.
    #[inline]
    fn zip<U>(self, other: U) -> Zip<Self, U>
        where Self: Sized,
              U: Event
    {
        Zip { comp: self, other: other }
    }

    /// The function application.
    #[inline]
    fn ap<U, B>(self, other: U) -> Ap<Self, U, B>
        where Self: Sized,
              Self::Item: FnOnce(U::Item) -> B,
              U: Event
    {
        Ap { comp: self, other: other, _phantom: PhantomData }
    }

    /// Run the computation in the start time.
    #[inline]
    fn run_in_start_time(self) -> RunInStartTime<Self>
        where Self: Sized
    {
        RunInStartTime { comp: self, including_current_events: true }
    }

    /// Run the computation in the start time by processing the current events or not.
    #[inline]
    fn run_in_start_time_by(self, including_current_events: bool) -> RunInStartTime<Self>
        where Self: Sized
    {
        RunInStartTime { comp: self, including_current_events: including_current_events }
    }

    /// Run the computation in the stop time.
    #[inline]
    fn run_in_stop_time(self) -> RunInStopTime<Self>
        where Self: Sized
    {
        RunInStopTime { comp: self, including_current_events: true }
    }

    /// Run the computation in the stop time by processing the current events or not.
    #[inline]
    fn run_in_stop_time_by(self, including_current_events: bool) -> RunInStopTime<Self>
        where Self: Sized
    {
        RunInStopTime { comp: self, including_current_events: including_current_events }
    }

    /// Convert into a boxed value.
    #[inline]
    fn into_boxed(self) -> EventBox<Self::Item>
        where Self: Sized + Clone + 'static
    {
        EventBox::new(move |p: &Point| { self.call_event(p) })
    }
}

/// Allows converting to `Event` computations.
pub trait IntoEvent {

    /// The target computation.
    type Event: Event<Item = Self::Item>;

    /// The type of item that is returned by the computation.
    type Item;

    /// Convert to the `Event` computation.
    fn into_event(self) -> Self::Event;
}

impl<M: Event> IntoEvent for M {

    type Event = M;

    type Item = M::Item;

    #[inline]
    fn into_event(self) -> Self::Event {
        self
    }
}

/// It represents the boxed `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct EventBox<T> {
    f: Box<dyn EventFnBoxClone<T>>
}

impl<T> EventBox<T> {

    /// Create a new boxed computation.
    #[doc(hidden)]
    #[inline]
    fn new<F>(f: F) -> Self
        where F: FnOnce(&Point) -> simulation::Result<T> + Clone + 'static
    {
        EventBox {
            f: Box::new(f)
        }
    }

    /// Call the boxed function.
    #[doc(hidden)]
    #[inline]
    pub fn call_box(self, arg: (&Point,)) -> simulation::Result<T> {
        let EventBox { f } = self;
        f.call_box(arg)
    }
}

impl<T> Clone for EventBox<T> {

    fn clone(&self) -> Self {
        EventBox {
            f: self.f.call_clone()
        }
    }
}

impl<T> Event for EventBox<T> {

    type Item = T;

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        self.call_box((p,))
    }

    #[inline]
    fn into_boxed(self) -> EventBox<Self::Item>
        where Self: Sized + Clone + 'static
    {
        self
    }
}

/// A trait to support the stable version of Rust, where there is no `FnBox`.
trait EventFnBox<T> {

    /// Call the corresponding function.
    fn call_box(self: Box<Self>, args: (&Point,)) -> simulation::Result<T>;
}

impl<T, F> EventFnBox<T> for F
    where F: for<'a> FnOnce(&'a Point) -> simulation::Result<T>
{
    fn call_box(self: Box<Self>, args: (&Point,)) -> simulation::Result<T> {
        let this: Self = *self;
        this(args.0)
    }
}

/// A trait to implement a cloneable `FnBox`.
trait EventFnBoxClone<T>: EventFnBox<T> {

    /// Clone the function.
    fn call_clone(&self) -> Box<dyn EventFnBoxClone<T>>;
}

impl<T, F> EventFnBoxClone<T> for F
    where F: for<'a> FnOnce(&'a Point) -> simulation::Result<T> + Clone + 'static
{
    fn call_clone(&self) -> Box<dyn EventFnBoxClone<T>> {
        Box::new(self.clone())
    }
}

/// It represents a raw trait object.
#[cfg(feature="dist_mode")]
#[repr(C)]
#[derive(Copy, Clone)]
struct EventTraitObject {

    field1: *mut c_void,
    field2: *mut c_void
}

/// A C-friendly representaton of the `Event` action.
#[cfg(feature="dist_mode")]
#[repr(C)]
pub struct EventRepr {

    /// Delete the object.
    delete: unsafe extern "C" fn(obj: *mut EventTraitObject),

    /// Clone the action.
    clone: unsafe extern "C" fn(obj: *const EventTraitObject) -> EventRepr,

    /// The callback.
    callback: unsafe extern "C" fn(obj: *mut EventTraitObject, p: *const Point) -> *mut ErrorRepr,

    /// The trait object.
    trait_object: EventTraitObject
}

#[cfg(feature="dist_mode")]
impl Drop for EventRepr {

    fn drop(&mut self) {
        unsafe {
            (self.delete)(&mut self.trait_object);
        }
    }
}

#[cfg(feature="dist_mode")]
impl Clone for EventRepr {

    fn clone(&self) -> Self {
        unsafe {
            (self.clone)(&self.trait_object)
        }
    }
}

#[cfg(feature="dist_mode")]
impl EventRepr {

    /// Convert to a C-friendly representation.
    #[inline]
    pub fn into_repr(comp: EventBox<()>) -> EventRepr {
        unsafe {
            EventRepr {
                delete: delete_event_repr,
                clone: clone_event_repr,
                callback: call_event_repr,
                trait_object: mem::transmute(comp)
            }
        }
    }

    /// Call the representation.
    #[inline]
    fn call_repr(mut self, p: &Point) -> *mut ErrorRepr {
        unsafe {
            let x = (self.callback)(&mut self.trait_object, p);
            mem::forget(self);
            x
        }
    }
}

/// Call the `EventBox` representation.
#[cfg(feature="dist_mode")]
unsafe extern "C" fn call_event_repr(comp: *mut EventTraitObject, p: *const Point) -> *mut ErrorRepr {
    let comp: EventBox<()> = mem::transmute(*comp);
    match comp.call_box((&*p,)) {
        Result::Ok(()) => ptr::null_mut(),
        Result::Err(e) => {
            let e = ErrorRepr::new(e);
            Box::into_raw(Box::new(e))
        }
    }
}

/// Clone the `EventBox` representation.
#[cfg(feature="dist_mode")]
unsafe extern "C" fn clone_event_repr(comp: *const EventTraitObject) -> EventRepr {
    let comp: EventBox<()> = mem::transmute(*comp);
    let x = EventRepr {
        delete: delete_event_repr,
        clone: clone_event_repr,
        callback: call_event_repr,
        trait_object: mem::transmute(comp.clone())
    };
    mem::forget(comp);
    x
}

/// Delete the `EventBox` representation.
#[cfg(feature="dist_mode")]
unsafe extern "C" fn delete_event_repr(comp: *mut EventTraitObject) {
    let _: EventBox<()> = mem::transmute(*comp);
}

#[cfg(feature="dist_mode")]
impl Event for EventRepr {

    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        unsafe {
            let e = self.call_repr(p);
            if e == ptr::null_mut() {
                Result::Ok(())
            } else {
                let e = ffi_error_repr_into_error(e);
                Result::Err(e)
            }
        }
    }
}

/// A thin wrapper representaton of the `Event` action.
#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
#[derive(Clone)]
pub struct EventRepr {

    /// The computation itself.
    comp: EventBox<()>
}

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
impl EventRepr {

    /// Convert to the representation.
    #[inline]
    pub fn into_repr(comp: EventBox<()>) -> EventRepr {
        EventRepr { comp: comp }
    }

    /// Call the representation.
    #[inline]
    fn call_repr(self, p: &Point) -> simulation::Result<()> {
        self.comp.call_event(p)
    }
}

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
impl Event for EventRepr {

    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        self.call_repr(p)
    }
}

/// It represents a raw trait object.
#[cfg(feature="dist_mode")]
#[repr(C)]
#[derive(Copy, Clone)]
struct DisposableEventTraitObject {

    field1: *mut c_void,
    field2: *mut c_void
}

/// A C-friendly representaton of the `Event<DisposableBox>` computation.
#[cfg(feature="dist_mode")]
#[repr(C)]
pub struct DisposableEventRepr {

    /// Delete the object.
    delete: unsafe extern "C" fn(obj: *mut DisposableEventTraitObject),

    /// Clone the action.
    clone: unsafe extern "C" fn(obj: *const DisposableEventTraitObject) -> DisposableEventRepr,

    /// The callback.
    callback: unsafe extern "C" fn(obj: *mut DisposableEventTraitObject, p: *const Point, e: *mut *mut ErrorRepr) -> DisposableRepr,

    /// The trait object.
    trait_object: DisposableEventTraitObject
}

#[cfg(feature="dist_mode")]
impl Drop for DisposableEventRepr {

    fn drop(&mut self) {
        unsafe {
            (self.delete)(&mut self.trait_object);
        }
    }
}

#[cfg(feature="dist_mode")]
impl Clone for DisposableEventRepr {

    fn clone(&self) -> Self {
        unsafe {
            (self.clone)(&self.trait_object)
        }
    }
}

#[cfg(feature="dist_mode")]
impl DisposableEventRepr {

    /// Convert to a C-friendly representation.
    #[inline]
    pub fn into_repr(comp: EventBox<DisposableRepr>) -> DisposableEventRepr {
        unsafe {
            DisposableEventRepr {
                delete: delete_disposable_event_repr,
                clone: clone_disposable_event_repr,
                callback: call_disposable_event_repr,
                trait_object: mem::transmute(comp)
            }
        }
    }

    /// Call the representation.
    #[inline]
    fn call_repr(mut self, p: &Point, e: *mut *mut ErrorRepr) -> DisposableRepr {
        unsafe {
            let x = (self.callback)(&mut self.trait_object, p, e);
            mem::forget(self);
            x
        }
    }
}

/// Call the `EventBox<DisposableRepr>` representation.
#[cfg(feature="dist_mode")]
unsafe extern "C" fn call_disposable_event_repr(comp: *mut DisposableEventTraitObject, p: *const Point, e: *mut *mut ErrorRepr) -> DisposableRepr {
    let comp: EventBox<DisposableRepr> = mem::transmute(*comp);
    match comp.call_box((&*p,)) {
        Result::Ok(x) => {
            *e = ptr::null_mut();
            x
        },
        Result::Err(e0) => {
            let e0 = ErrorRepr::new(e0);
            let e0 = Box::new(e0);
            let e0 = Box::into_raw(e0);
            *e = e0;

            let x = empty_disposable();
            let x = x.into_boxed();
            let x = DisposableRepr::into_repr(x);
            x
        }
    }
}

/// Clone the `EventBox<DisposableRepr>` representation.
#[cfg(feature="dist_mode")]
unsafe extern "C" fn clone_disposable_event_repr(comp: *const DisposableEventTraitObject) -> DisposableEventRepr {
    let comp: EventBox<DisposableRepr> = mem::transmute(*comp);
    let x = DisposableEventRepr {
        delete: delete_disposable_event_repr,
        clone: clone_disposable_event_repr,
        callback: call_disposable_event_repr,
        trait_object: mem::transmute(comp.clone())
    };
    mem::forget(comp);
    x
}

/// Delete the `EventBox<DisposableRepr>` representation.
#[cfg(feature="dist_mode")]
unsafe extern "C" fn delete_disposable_event_repr(comp: *mut DisposableEventTraitObject) {
    let _: EventBox<DisposableRepr> = mem::transmute(*comp);
}

#[cfg(feature="dist_mode")]
impl Event for DisposableEventRepr {

    type Item = DisposableRepr;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        unsafe {
            let mut e: *mut ErrorRepr = ptr::null_mut();
            let x = self.call_repr(p, &mut e);
            if e == ptr::null_mut() {
                Result::Ok(x)
            } else {
                let e = ffi_error_repr_into_error(e);
                Result::Err(e)
            }
        }
    }
}

/// Allows creating the `Event` computation from a pure value.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Return<T> {

    /// Return a pure value, which is then transformed to the computation.
    val: T
}

impl<T> Event for Return<T> {

    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, _: &Point) -> simulation::Result<T> {
        let Return { val } = self;
        Result::Ok(val)
    }
}

/// Allows delaying the `Event` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Delay<F, M> {

    /// Return the computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<F, M> Event for Delay<F, M>
    where F: FnOnce() -> M,
          M: Event
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<M::Item> {
        let Delay { f, _phantom } = self;
        f().call_event(p)
    }
}

/// Allows constructing the `Event` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Cons<F, T> {

    /// The function of time point.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<T>
}

impl<F, T> Event for Cons<F, T>
    where F: FnOnce(&Point) -> simulation::Result<T>
{
    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<T> {
        let Cons { f, _phantom } = self;
        f(p)
    }
}

/// Enqueues the event which handler should be actuated at the specified time.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Enqueue {

    /// The time of activating the event.
    time: f64,

    /// The event handler.
    comp: EventBox<()>
}

impl Event for Enqueue {

    type Item = ();

    #[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let Enqueue { time, comp } = self;
        let run   = p.run;
        let queue = &run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        queue.enqueue_event(time, p.priority, comp, p);
        Result::Ok(())
    }

    #[cfg(feature="dist_mode")]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let Enqueue { time, comp } = self;
        let run   = p.run;
        let queue = run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        unsafe {
            enqueue_extern_event(queue, time, p.priority, comp, p);
        }
        Result::Ok(())
    }
}

/// Enqueues the event which handler should be actuated at the specified time with the given priority.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct EnqueueWithPriority {

    /// The time of activating the event.
    time: f64,

    /// The time priority.
    priority: isize,

    /// The event handler.
    comp: EventBox<()>
}

impl Event for EnqueueWithPriority {

    type Item = ();

    #[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueWithPriority { time, priority, comp } = self;
        let run   = p.run;
        let queue = &run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        queue.enqueue_event(time, priority, comp, p);
        Result::Ok(())
    }

    #[cfg(feature="dist_mode")]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueWithPriority { time, priority, comp } = self;
        let run   = p.run;
        let queue = run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        unsafe {
            enqueue_extern_event(queue, time, priority, comp, p);
        }
        Result::Ok(())
    }
}

/// Enqueues the events which handlers should be actuated in the integration time points.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct EnqueueWithIntegTimes<F> {

    /// The generator of event handlers.
    f: F
}

impl<F> Event for EnqueueWithIntegTimes<F>
    where F: Fn() -> EventBox<()> + Clone + 'static
{
    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueWithIntegTimes { f } = self;
        let comp = EnqueueWithIntegTimesLoop { f, iteration: p.iteration };
        enqueue_event(p.time, comp.into_boxed())
            .call_event(p)
    }
}

/// Enqueues the events with the integration time points in a loop.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
struct EnqueueWithIntegTimesLoop<F> {

    /// The generator of event handlers.
    f: F,

    /// The iteration number.
    iteration: usize
}

impl<F> Event for EnqueueWithIntegTimesLoop<F>
    where F: Fn() -> EventBox<()> + Clone + 'static
{
    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueWithIntegTimesLoop { f, iteration } = self;
        let comp = f();
        let p = p.with_iteration(iteration); // correction
        comp.call_event(&p)?;
        let p2 = p.with_iteration(1 + iteration);
        let comp = EnqueueWithIntegTimesLoop { f, iteration: p2.iteration };
        enqueue_event(p2.time, comp.into_boxed())
            .call_event(&p)
    }
}

/// Enqueues the IO-based event which handler should be actuated at the specified time.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct EnqueueIO {

    /// The time of activating the event.
    time: f64,

    /// The event handler.
    comp: EventBox<()>
}

impl Event for EnqueueIO {

    type Item = ();

    #[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueIO { time, comp } = self;
        let run   = p.run;
        let queue = &run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        queue.enqueue_io_event(time, p.priority, comp, p);
        Result::Ok(())
    }

    #[cfg(feature="dist_mode")]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueIO { time, comp } = self;
        let run   = p.run;
        let queue = run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        unsafe {
            enqueue_extern_io_event(queue, time, p.priority, comp, p);
        }
        Result::Ok(())
    }
}

/// Enqueues the IO-based event which handler should be actuated at the specified time with the given priority.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct EnqueueIOWithPriority {

    /// The time of activating the event.
    time: f64,

    /// The time priority.
    priority: isize,

    /// The event handler.
    comp: EventBox<()>
}

impl Event for EnqueueIOWithPriority {

    type Item = ();

    #[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueIOWithPriority { time, priority, comp } = self;
        let run   = p.run;
        let queue = &run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        queue.enqueue_io_event(time, priority, comp, p);
        Result::Ok(())
    }

    #[cfg(feature="dist_mode")]
    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueIOWithPriority { time, priority, comp } = self;
        let run   = p.run;
        let queue = run.event_queue;
        let comp  = EventRepr::into_repr(comp);
        unsafe {
            enqueue_extern_io_event(queue, time, priority, comp, p);
        }
        Result::Ok(())
    }
}

/// Enqueues the IO-based events which handlers should be actuated in the integration time points.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct EnqueueIOWithIntegTimes<F> {

    /// The generator of event handlers.
    f: F
}

impl<F> Event for EnqueueIOWithIntegTimes<F>
    where F: Fn() -> EventBox<()> + Clone + 'static
{
    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueIOWithIntegTimes { f } = self;
        let comp = EnqueueIOWithIntegTimesLoop { f, iteration: p.iteration };
        enqueue_io_event(p.time, comp.into_boxed())
            .call_event(p)
    }
}

/// Enqueues the IO-based events with the integration time points in a loop.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
struct EnqueueIOWithIntegTimesLoop<F> {

    /// The generator of event handlers.
    f: F,

    /// The iteration number.
    iteration: usize
}

impl<F> Event for EnqueueIOWithIntegTimesLoop<F>
    where F: Fn() -> EventBox<()> + Clone + 'static
{
    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<()> {
        let EnqueueIOWithIntegTimesLoop { f, iteration } = self;
        let comp = f();
        let p = p.with_iteration(iteration); // correction
        comp.call_event(&p)?;
        let p2 = p.with_iteration(1 + iteration);
        let comp = EnqueueIOWithIntegTimesLoop { f, iteration: p2.iteration };
        enqueue_io_event(p2.time, comp.into_boxed())
            .call_event(&p)
    }
}

/// The monadic bind for the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct FlatMap<M, U, F> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<U>
}

impl<M, U, F> Event for FlatMap<M, U, F>
    where M: Event,
          U: Event,
          F: FnOnce(M::Item) -> U,
{
    type Item = U::Item;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<U::Item> {
        let FlatMap { comp, f, _phantom } = self;
        match comp.call_event(p) {
            Result::Ok(a) => {
                let m = f(a);
                m.call_event(p)
            },
            Result::Err(e) => {
                Result::Err(e)
            }
        }
    }
}

/// The functor for the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Map<M, B, F> {

    /// The current computation.
    comp: M,

    /// The transform.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, B, F> Event for Map<M, B, F>
    where M: Event,
          F: FnOnce(M::Item) -> B,
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<B> {
        let Map { comp, f, _phantom } = self;
        match comp.call_event(p) {
            Result::Ok(a) => Result::Ok(f(a)),
            Result::Err(e) => Result::Err(e)
        }
    }
}

/// The zip of two `Event` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Zip<M, U> {

    /// The current computation.
    comp: M,

    /// Another computation.
    other: U,
}

impl<M, U> Event for Zip<M, U>
    where M: Event,
          U: Event
{
    type Item = (M::Item, U::Item);

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<(M::Item, U::Item)> {
        let Zip { comp, other } = self;
        comp.flat_map(move |a| {
            other.map(move |b| (a, b))
        }).call_event(p)
    }
}

/// The function application for the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Ap<M, U, B> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    other: U,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, U, B> Event for Ap<M, U, B>
    where M: Event,
          U: Event,
          M::Item: FnOnce(U::Item) -> B,
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<B> {
        let Ap { comp, other, _phantom } = self;
        comp.flat_map(move |f| {
            other.map(move |a| { f(a) })
        }).call_event(p)
    }
}

/// Run the computation in the start time.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct RunInStartTime<M> {

    /// The current computation.
    comp: M,

    /// Whether to include the current events.
    including_current_events: bool
}

impl<M> Simulation for RunInStartTime<M>
    where M: Event
{
    type Item = M::Item;

    #[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<M::Item> {
        let RunInStartTime { comp, including_current_events } = self;
        let p = r.point_in_start_time();
        let queue = &r.event_queue;
        queue.run_events(including_current_events, &p)?;
        comp.call_event(&p)
    }

    #[cfg(feature="dist_mode")]
    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<M::Item> {
        let RunInStartTime { comp, including_current_events } = self;
        let p = r.point_in_start_time();
        let including_current_events = if including_current_events { 1 } else { 0 };
        unsafe {
            sync_extern_events(including_current_events, &p);
        }
        comp.call_event(&p)
    }
}

/// Run the computation in the stop time.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct RunInStopTime<M> {

    /// The current computation.
    comp: M,

    /// Whether to include the current events.
    including_current_events: bool
}

impl<M> Simulation for RunInStopTime<M>
    where M: Event
{
    type Item = M::Item;

    #[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<M::Item> {
        let RunInStopTime { comp, including_current_events } = self;
        let p = r.point_in_stop_time();
        let queue = &r.event_queue;
        queue.run_events(including_current_events, &p)?;
        comp.call_event(&p)
    }

    #[cfg(feature="dist_mode")]
    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<M::Item> {
        let RunInStopTime { comp, including_current_events } = self;
        let p = r.point_in_stop_time();
        let including_current_events = if including_current_events { 1 } else { 0 };
        unsafe {
            sync_extern_events(including_current_events, &p);
        }
        comp.call_event(&p)
    }
}

/// Returns the current simulation time.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Time {}

impl Event for Time {

    type Item = f64;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<f64> {
        Result::Ok(p.time)
    }
}

/// Returns the current simulation time priority.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Priority {}

impl Event for Priority {

    type Item = isize;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<isize> {
        Result::Ok(p.priority)
    }
}

/// Allows converting to the `Process` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct EventIntoProcess<M> {

    /// The current computation.
    comp: M
}

impl<M> Process for EventIntoProcess<M>
    where M: Event
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_process<C>(self, cont: C, pid: Rc<ProcessId>, p: &Point) -> simulation::Result<()>
        where C: FnOnce(simulation::Result<Self::Item>, Rc<ProcessId>, &Point) -> simulation::Result<()> + Clone + 'static
    {
        if is_process_cancelled(&pid, p) {
            revoke_process(cont, pid, p)
        } else {
            let EventIntoProcess { comp } = self;
            let t = comp.call_event(p);
            cont(t, pid, p)
        }
    }

    #[doc(hidden)]
    #[inline]
    fn call_process_boxed(self, cont: ProcessBoxCont<Self::Item>, pid: Rc<ProcessId>, p: &Point) -> simulation::Result<()> {
        if is_process_cancelled(&pid, p) {
            revoke_process_boxed(cont, pid, p)
        } else {
            let EventIntoProcess { comp } = self;
            let t = comp.call_event(p);
            cont.call_box((t, pid, p))
        }
    }
}

/// A conversion into the `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct EventIntoComposite<M> {

    /// The current computation.
    comp: M
}

impl<M> Composite for EventIntoComposite<M>
    where M: Event
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(M::Item, DisposableBox)> {
        let EventIntoComposite { comp } = self;
        let a = comp.call_event(p)?;
        Result::Ok((a, disposable))
    }
}

/// The sequence of computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Event for Sequence<I, M>
    where I: Iterator<Item = M>,
          M: Event
{
    type Item = Vec<M::Item>;

    #[doc(hidden)]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Sequence { comps, _phantom } = self;
        let mut v = {
            match comps.size_hint() {
                (_, Some(n)) => Vec::with_capacity(n),
                (_, None) => Vec::new()
            }
        };
        for m in comps {
            match m.call_event(p) {
                Result::Ok(a) => {
                    v.push(a)
                },
                Result::Err(e) => {
                    return Result::Err(e)
                }
            }
        }
        Result::Ok(v)
    }
}

/// The sequence of computations with ignored result.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence_<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Event for Sequence_<I, M>
    where I: Iterator<Item = M>,
          M: Event
{
    type Item = ();

    #[doc(hidden)]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Sequence_ { comps, _phantom } = self;
        for m in comps {
            match m.call_event(p) {
                Result::Ok(_) => (),
                Result::Err(e) => return Result::Err(e)
            }
        }
        Result::Ok(())
    }
}

/// Enqueue the event which must be actuated with the current modeling time but later.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Yield<M> {

    /// The computation.
    comp: M
}

impl<M> Event for Yield<M>
    where M: Event<Item = ()> + Clone + 'static
{
    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Yield { comp } = self;
        enqueue_event(p.time, comp.into_boxed()).call_event(p)
    }
}

/// A computation that tries to leave the simulation.
#[cfg(feature="dist_mode")]
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct LeaveSimulation {}

#[cfg(feature="dist_mode")]
impl Event for LeaveSimulation {

    type Item = ();

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let LeaveSimulation {} = self;
        let r = p.run;
        let q = r.event_queue;
        unsafe {
            leave_extern_simulation(q, p);
        }
        Result::Ok(())
    }
}

/// Trace the computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Trace<M> {

    /// The computation.
    comp: M,

    /// The message to print.
    msg: String
}

impl<M> Event for Trace<M>
    where M: Event
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Trace { comp, msg } = self;
        p.trace(&msg);
        comp.call_event(p)
    }
}
