// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::time::*;
use std::ptr;
use std::ffi::CString;
use std::ffi::CStr;

use libc::*;

use dvcompute_network::network::*;

/// The Time Server parameters.
pub struct TimeServerParameters {

    /// The logical process name.
    pub name: String,

    /// The timeout used for the time synchronization sessions.
    pub sync_timeout: Duration,

    /// The delay between the time synchronization sessions.
    pub sync_delay: Duration,

    /// The idle loop interval when processing the messages.
    pub idle_loop_interval: Duration,

    /// The test interval for network messages.
    pub network_test_interval: Duration,

    /// The threshold of network requests to test.
    pub network_test_threshold: usize,

    /// The threshold of network requests to wait.
    pub network_wait_threshold: usize
}

impl TimeServerParameters {

    /// Create new Time Server parameters.
    pub fn new() -> TimeServerParameters {
        TimeServerParameters {
            name: String::from("TimeServer"),
            sync_timeout: Duration::from_secs(60),
            sync_delay: Duration::from_millis(100),
            idle_loop_interval: Duration::from_millis(10),
            network_test_interval: Duration::from_millis(10),
            network_test_threshold: 256,
            network_wait_threshold: 8192
        }
    }
}

/// A C-friendly representation of the time server parameters.
#[repr(C)]
pub struct TimeServerParametersRepr {

    /// Delete the representation.
    delete_repr: unsafe extern "C" fn(*mut TimeServerParametersRepr),

    /// The time server name.
    name: *mut c_char,

    /// Delete the time server name.
    delete_name: unsafe extern "C" fn(*mut c_char),

    /// The timeout used for the time synchronization sessions.
    sync_timeout: [u8; 16],

    /// The delay between the time synchronization sessions.
    sync_delay: [u8; 16],

    /// The idle loop interval when processing the messages.
    idle_loop_interval: [u8; 16],

    /// The test interval for network messages.
    network_test_interval: [u8; 16],

    /// The threshold of network requests to test.
    network_test_threshold: usize,

    /// The threshold of network requests to wait.
    network_wait_threshold: usize
}

impl TimeServerParametersRepr {

    /// Convert to a C-friendly representation.
    pub fn new(ps: TimeServerParameters) -> Self {
        let name = CString::new(ps.name.clone()).expect("NulError");
        let name = CString::into_raw(name);
        TimeServerParametersRepr {
            delete_repr: delete_ts_repr,
            name: name,
            delete_name: delete_ts_name,
            sync_timeout: u128::to_ne_bytes(ps.sync_timeout.as_nanos()),
            sync_delay: u128::to_ne_bytes(ps.sync_delay.as_nanos()),
            idle_loop_interval: u128::to_ne_bytes(ps.idle_loop_interval.as_nanos()),
            network_test_interval: u128::to_ne_bytes(ps.network_test_interval.as_nanos()),
            network_test_threshold: ps.network_test_threshold,
            network_wait_threshold: ps.network_wait_threshold
        }
    }
}

impl Drop for TimeServerParametersRepr {

    fn drop(&mut self) {
        unsafe {
            (self.delete_name)(self.name);
        }
    }
}

unsafe extern "C" fn delete_ts_name(name: *mut c_char) {
    if name != ptr::null_mut() {
        let _ = CString::from_raw(name);
    }
}

unsafe extern "C" fn delete_ts_repr(e: *mut TimeServerParametersRepr) {
    if e != ptr::null_mut() {
        let _ = Box::from_raw(e);
    }
}

/// Convert the `TimeServerParametersRepr` referenced to by the FFI pointer to a `TimeServerParameters`.
pub unsafe fn ffi_time_server_parameters_repr_into(ps: *mut TimeServerParametersRepr) -> TimeServerParameters {
    let name = CStr::from_ptr((*ps).name);
    let name = name.to_bytes().to_vec();
    let name = String::from_utf8(name).expect("FromUtf8Error");
    let xs   = TimeServerParameters {
        name: name,
        sync_timeout: Duration::from_nanos(u128::from_ne_bytes((*ps).sync_timeout) as u64),
        sync_delay: Duration::from_nanos(u128::from_ne_bytes((*ps).sync_delay) as u64),
        idle_loop_interval: Duration::from_nanos(u128::from_ne_bytes((*ps).idle_loop_interval) as u64),
        network_test_interval: Duration::from_nanos(u128::from_ne_bytes((*ps).network_test_interval) as u64),
        network_test_threshold: (*ps).network_test_threshold,
        network_wait_threshold: (*ps).network_wait_threshold
    };
    ((*ps).delete_repr)(ps);
    xs
}

/// Represents the Time Server.
pub struct TimeServer {}

impl TimeServer {

    /// Run the Time Server.
    pub fn run(network: NetworkSupport, init_quorum: usize, ps: TimeServerParameters) {
        unsafe {
            let ps = TimeServerParametersRepr::new(ps);
            let ps = Box::new(ps);
            let ps = Box::into_raw(ps);

            run_extern_time_server(network, init_quorum, ps)
        }
    }
}

#[cfg_attr(windows, link(name = "dvcompute_core_dist.dll"))]
#[cfg_attr(not(windows), link(name = "dvcompute_core_dist"))]
extern {

    /// Run the time server.
    #[doc(hidden)]
    pub fn run_extern_time_server(network: NetworkSupport, init_quorum: usize, ps: *mut TimeServerParametersRepr);
}
