// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;

use crate::simulation::stream::*;

/// It represents the source of `Stream` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct StreamFn<T> {
    gen: Rc<dyn Fn() -> Stream<T>>
}

impl<T> StreamFn<T> {

    /// Create a new source of computations.
    #[inline]
    pub fn new<F>(f: F) -> Self
        where F: Fn() -> Stream<T> + 'static
    {
        StreamFn {
            gen: Rc::new(move || { f() })
        }
    }

    /// Get the next computation.
    #[inline]
    pub fn next(&self) -> Stream<T> {
        (self.gen)()
    }

    /// Return the prefix of the stream of the specified length.
    pub fn take(&self, n: isize) -> Self
        where T: 'static
    {
        let gen = self.gen.clone();
        StreamFn {
            gen: Rc::new(move || { gen().take(n) })
        }
    }

    /// Return the suffix of the stream after the specified first elements.
    pub fn drop(&self, n: isize) -> Self
        where T: 'static
    {
        let gen = self.gen.clone();
        StreamFn {
            gen: Rc::new(move || { gen().take(n) })
        }
    }
}
