// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;

use crate::simulation::simulation::*;

/// It represents the source of `Simulation` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct SimulationFn<T> {
    gen: Rc<dyn Fn() -> SimulationBox<T>>
}

impl<T> SimulationFn<T> {

    /// Create a new source of computations.
    #[inline]
    pub fn new<F, M>(f: F) -> Self
        where F: Fn() -> M + 'static,
              M: Simulation<Item = T> + 'static
    {
        SimulationFn {
            gen: Rc::new(move || { f().into_boxed() })
        }
    }

    /// Get the next computation.
    #[inline]
    pub fn next(&self) -> SimulationBox<T> {
        (self.gen)()
    }
}
