// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::marker::PhantomData;

use crate::simulation;
use crate::simulation::*;
use crate::simulation::Run;
use crate::simulation::Point;
use crate::simulation::event::Event;
use crate::simulation::process::*;
use crate::simulation::observable::disposable::*;
use crate::simulation::composite::*;

#[cfg(feature="cons_mode")]
use crate::simulation::comm::context::LogicalProcessContext;

/// Additional operations.
pub mod ops;

/// Return a new `Simulation` computation by the specified pure value.
#[inline]
pub fn return_simulation<T>(val: T) -> Return<T> {
    Return { val: val }
}

/// Delay the `Simulation` computation.
#[inline]
pub fn delay_simulation<F, M>(f: F) -> Delay<F, M>
    where F: FnOnce() -> M,
          M: Simulation
{
    Delay { f: f, _phantom: PhantomData }
}

/// Construct a new `Simulation` computation by the specified function.
#[inline]
pub fn cons_simulation<F, T>(f: F) -> Cons<F, T>
    where F: FnOnce(&Run) -> simulation::Result<T>
{
    Cons { f: f, _phantom: PhantomData }
}

/// Create a sequence of computations.
#[inline]
pub fn simulation_sequence<I, M>(comps: I) -> Sequence<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Simulation
{
    Sequence { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Create a sequence of computations, where the result is ignored.
#[inline]
pub fn simulation_sequence_<I, M>(comps: I) -> Sequence_<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Simulation
{
    Sequence_ { comps: comps.into_iter(), _phantom: PhantomData }
}

/// The computation of arbitrary function of simulation run.
pub trait Simulation {

    /// The type of the item that is returned in within the simulation run.
    type Item;

    /// Call the `Simulation` computation within the current simulation run.
    #[doc(hidden)]
    fn call_simulation(self, r: &Run) -> simulation::Result<Self::Item>;

    /// Convert into the `Event` computation.
    #[inline]
    fn into_event(self) -> SimulationIntoEvent<Self>
        where Self: Sized
    {
        SimulationIntoEvent { comp: self }
    }

    /// Convert to the `Process` computation.
    #[inline]
    fn into_process(self) -> SimulationIntoProcess<Self>
        where Self: Sized
    {
        SimulationIntoProcess { comp: self }
    }

    /// Convert into the `Composite` computation.
    #[inline]
    fn into_composite(self) -> SimulationIntoComposite<Self>
        where Self: Sized
    {
        SimulationIntoComposite { comp: self }
    }

    /// Bind the current computation with its continuation within the resulting computation.
    #[inline]
    fn flat_map<U, F>(self, f: F) -> FlatMap<Self, U, F>
        where Self: Sized,
              U: Simulation,
              F: FnOnce(Self::Item) -> U,
    {
        FlatMap { comp: self, f: f, _phantom: PhantomData }
    }

    /// Map the current computation using the specified transform.
    #[inline]
    fn map<B, F>(self, f: F) -> Map<Self, B, F>
        where Self: Sized,
              F: FnOnce(Self::Item) -> B,
    {
        Map { comp: self, f: f, _phantom: PhantomData }
    }

    /// Zip the current computation with another one within the resulting computation.
    #[inline]
    fn zip<U>(self, other: U) -> Zip<Self, U>
        where Self: Sized,
              U: Simulation
    {
        Zip { comp: self, other: other }
    }

    /// The function application.
    #[inline]
    fn ap<U, B>(self, other: U) -> Ap<Self, U, B>
        where Self: Sized,
              Self::Item: FnOnce(U::Item) -> B,
              U: Simulation
    {
        Ap { comp: self, other: other, _phantom: PhantomData }
    }

    /// Finalize the current computation regardless of canceling it or not.
    #[inline]
    fn finally<U>(self, finalization: U) -> Finally<Self, U>
        where Self: Sized,
              U: Simulation<Item = ()>
    {
        Finally { comp: self, finalization: finalization }
    }

    /// Run the simulation.
    #[cfg(any(feature="seq_mode", feature="wasm_mode"))]
    #[inline]
    fn run(self, specs: Specs) -> simulation::Result<Self::Item>
        where Self: Sized
    {
        let run = Run::new(specs);
        self.call_simulation(&run)
    }

    /// Run the simulation.
    #[cfg(feature="cons_mode")]
    #[inline]
    fn run<'a>(self, specs: Specs, ctx: &'a LogicalProcessContext) -> simulation::Result<Self::Item>
        where Self: Sized
    {
        let run = Run::new(specs, ctx);
        self.call_simulation(&run)
    }

    /// Run the simulation by index.
    #[cfg(any(feature="seq_mode", feature="wasm_mode"))]
    #[inline]
    fn run_by_index(self, specs: Specs, run_index: usize, run_count: usize) -> simulation::Result<Self::Item>
        where Self: Sized
    {
        let run = Run::by_index(specs, run_index, run_count);
        self.call_simulation(&run)
    }

    /// Run the simulation by index.
    #[cfg(feature="cons_mode")]
    #[inline]
    fn run_by_index<'a>(self, specs: Specs, ctx: &'a LogicalProcessContext, run_index: usize, run_count: usize) -> simulation::Result<Self::Item>
        where Self: Sized
    {
        let run = Run::by_index(specs, ctx, run_index, run_count);
        self.call_simulation(&run)
    }

    /// Convert into a boxed value.
    #[inline]
    fn into_boxed(self) -> SimulationBox<Self::Item>
        where Self: Sized + 'static
    {
        SimulationBox::new(move |r: &Run| { self.call_simulation(r) })
    }
}

/// Allows converting to `Simulation` computations.
pub trait IntoSimulation {

    /// The target computation.
    type Simulation: Simulation<Item = Self::Item>;

    /// The type of item that is returned by the computation.
    type Item;

    /// Convert to the `Simulation` computation.
    fn into_simulation(self) -> Self::Simulation;
}

impl<M: Simulation> IntoSimulation for M {

    type Simulation = M;

    type Item = M::Item;

    #[inline]
    fn into_simulation(self) -> Self::Simulation {
        self
    }
}

/// It represents the boxed `Simulation` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct SimulationBox<T> {
    f: Box<dyn SimulationFnBox<T>>
}

impl<T> SimulationBox<T> {

    /// Create a new boxed computation.
    #[doc(hidden)]
    #[inline]
    fn new<F>(f: F) -> Self
        where F: FnOnce(&Run) -> simulation::Result<T> + 'static
    {
        SimulationBox {
            f: Box::new(f)
        }
    }

    /// Call the boxed function.
    #[doc(hidden)]
    #[inline]
    pub fn call_box(self, arg: (&Run,)) -> simulation::Result<T> {
        let SimulationBox { f } = self;
        f.call_box(arg)
    }
}

impl<T> Simulation for SimulationBox<T> {

    type Item = T;

    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<Self::Item> {
        self.call_box((r,))
    }

    #[inline]
    fn into_boxed(self) -> SimulationBox<Self::Item>
        where Self: Sized + 'static
    {
        self
    }
}

/// A trait to support the stable version of Rust, where there is no `FnBox`.
trait SimulationFnBox<T> {

    /// Call the corresponding function.
    fn call_box(self: Box<Self>, args: (&Run,)) -> simulation::Result<T>;
}

impl<T, F> SimulationFnBox<T> for F
    where F: for<'a> FnOnce(&'a Run) -> simulation::Result<T>
{
    fn call_box(self: Box<Self>, args: (&Run,)) -> simulation::Result<T> {
        let this: Self = *self;
        this(args.0)
    }
}

/// Allows creating the `Simulation` computation from a pure value.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Return<T> {

    /// Return a pure value, which is then transformed to the computation.
    val: T
}

impl<T> Simulation for Return<T> {

    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, _: &Run) -> simulation::Result<T> {
        let Return { val } = self;
        Result::Ok(val)
    }
}

/// Allows delaying the `Simulation` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Delay<F, M> {

    /// Return the computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<F, M> Simulation for Delay<F, M>
    where F: FnOnce() -> M,
          M: Simulation
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<M::Item> {
        let Delay { f, _phantom } = self;
        f().call_simulation(r)
    }
}

/// Allows constructing the `Simulation` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Cons<F, T> {

    /// The function of simulation run.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<T>
}

impl<F, T> Simulation for Cons<F, T>
    where F: FnOnce(&Run) -> simulation::Result<T>
{
    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<T> {
        let Cons { f, _phantom } = self;
        f(r)
    }
}

/// A conversion into the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct SimulationIntoEvent<M> {

    /// The current computation.
    comp: M
}

impl<M> Event for SimulationIntoEvent<M>
    where M: Simulation
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<M::Item> {
        let SimulationIntoEvent { comp } = self;
        comp.call_simulation(p.run)
    }
}

/// A conversion into the `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct SimulationIntoComposite<M> {

    /// The current computation.
    comp: M
}

impl<M> Composite for SimulationIntoComposite<M>
    where M: Simulation
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(M::Item, DisposableBox)> {
        let SimulationIntoComposite { comp } = self;
        let a = comp.call_simulation(p.run)?;
        Result::Ok((a, disposable))
    }
}

/// The monadic bind for the `Simulation` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct FlatMap<M, U, F> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<U>
}

impl<M, U, F> Simulation for FlatMap<M, U, F>
    where M: Simulation,
          U: Simulation,
          F: FnOnce(M::Item) -> U
{
    type Item = U::Item;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<U::Item> {
        let FlatMap { comp, f, _phantom } = self;
        match comp.call_simulation(r) {
            Result::Ok(a) => {
                let m = f(a);
                m.call_simulation(r)
            },
            Result::Err(e) => {
                Result::Err(e)
            }
        }
    }
}

/// The functor for the `Simulation` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Map<M, B, F> {

    /// The current computation.
    comp: M,

    /// The transform.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, B, F> Simulation for Map<M, B, F>
    where M: Simulation,
          F: FnOnce(M::Item) -> B
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<B> {
        let Map { comp, f, _phantom } = self;
        match comp.call_simulation(r) {
            Result::Ok(a) => Result::Ok(f(a)),
            Result::Err(e) => Result::Err(e)
        }
    }
}

/// The zip of two `Simulation` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Zip<M, U> {

    /// The current computation.
    comp: M,

    /// Another computation.
    other: U,
}

impl<M, U> Simulation for Zip<M, U>
    where M: Simulation,
          U: Simulation
{
    type Item = (M::Item, U::Item);

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<(M::Item, U::Item)> {
        let Zip { comp, other } = self;
        comp.flat_map(move |a| {
            other.map(move |b| (a, b))
        }).call_simulation(r)
    }
}

/// The function application for the `Simulation` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Ap<M, U, B> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    other: U,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, U, B> Simulation for Ap<M, U, B>
    where M: Simulation,
          U: Simulation,
          M::Item: FnOnce(U::Item) -> B
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<B> {
        let Ap { comp, other, _phantom } = self;
        comp.flat_map(move |f| {
            other.map(move |a| { f(a) })
        }).call_simulation(r)
    }
}

/// The finally block for the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Finally<M, U> {

    /// The current computation.
    comp: M,

    /// The finalization computation.
    finalization: U
}

impl<M, U> Simulation for Finally<M, U>
    where M: Simulation,
          U: Simulation<Item = ()>
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<Self::Item> {
        let Finally { comp, finalization } = self;
        let x = comp.call_simulation(r);
        match finalization.call_simulation(r) {
            Result::Ok(_) => x,
            Result::Err(e0) => {
                match x {
                    Result::Ok(_) => Result::Err(e0),
                    Result::Err(e) => Result::Err(e.merge(&e0))
                }
            }
        }
    }
}

/// Allows converting to the `Process` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct SimulationIntoProcess<M> {

    /// The current computation.
    comp: M
}

impl<M> Process for SimulationIntoProcess<M>
    where M: Simulation
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_process<C>(self, cont: C, pid: Rc<ProcessId>, p: &Point) -> simulation::Result<()>
        where C: FnOnce(simulation::Result<Self::Item>, Rc<ProcessId>, &Point) -> simulation::Result<()> + 'static
    {
        if is_process_cancelled(&pid, p) {
            revoke_process(cont, pid, p)
        } else {
            let SimulationIntoProcess { comp } = self;
            let t = comp.call_simulation(p.run);
            cont(t, pid, p)
        }
    }

    #[doc(hidden)]
    #[inline]
    fn call_process_boxed(self, cont: ProcessBoxCont<Self::Item>, pid: Rc<ProcessId>, p: &Point) -> simulation::Result<()> {
        if is_process_cancelled(&pid, p) {
            revoke_process_boxed(cont, pid, p)
        } else {
            let SimulationIntoProcess { comp } = self;
            let t = comp.call_simulation(p.run);
            cont.call_box((t, pid, p))
        }
    }
}

/// The sequence of computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Simulation for Sequence<I, M>
    where I: Iterator<Item = M>,
          M: Simulation
{
    type Item = Vec<M::Item>;

    #[doc(hidden)]
    fn call_simulation(self, r: &Run) -> simulation::Result<Self::Item> {
        let Sequence { comps, _phantom } = self;
        let mut v = {
            match comps.size_hint() {
                (_, Some(n)) => Vec::with_capacity(n),
                (_, None) => Vec::new()
            }
        };
        for m in comps {
            match m.call_simulation(r) {
                Result::Ok(a) => {
                    v.push(a)
                },
                Result::Err(e) => {
                    return Result::Err(e)
                }
            }
        }
        Result::Ok(v)
    }
}

/// The sequence of computations with ignored result.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence_<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Simulation for Sequence_<I, M>
    where I: Iterator<Item = M>,
          M: Simulation
{
    type Item = ();

    #[doc(hidden)]
    fn call_simulation(self, r: &Run) -> simulation::Result<Self::Item> {
        let Sequence_ { comps, _phantom } = self;
        for m in comps {
            match m.call_simulation(r) {
                Result::Ok(_) => (),
                Result::Err(e) => return Result::Err(e)
            }
        }
        Result::Ok(())
    }
}
