// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;

use crate::simulation::event::*;

/// It represents the source of `Event` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct EventFn<T> {
    gen: Rc<dyn Fn() -> EventBox<T>>
}

impl<T> EventFn<T> {

    /// Create a new source of computations.
    #[inline]
    pub fn new<F, M>(f: F) -> Self
        where F: Fn() -> M + 'static,
              M: Event<Item = T> + 'static
    {
        EventFn {
            gen: Rc::new(move || { f().into_boxed() })
        }
    }

    /// Get the next computation.
    #[inline]
    pub fn next(&self) -> EventBox<T> {
        (self.gen)()
    }
}
