// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

//! This crate is a part of discrete event simulation framework DVCompute Simulator (registration
//! number 2021660590 of Rospatent). The `dvcompute` crate is destined for sequential simulation,
//! but the same code base is shared by the `dvcompute_cons` crate destined for conservative
//! distributed simulation.
//!
//! There are the following main crates: `dvcompute` (for sequential simulation),
//! `dvcompute_dist` (for optimistic distributed simulation - from multicore computers till supercomputers),
//! `dvcompute_cons` (for conservative distributed simulation - from multicore computers till supercomputers) and
//! `dvcompute_branch` (for nested simulation - theory of games, estimation of the optimal strategy). All four crates are
//! very close. They are based on the same method.
//!
//! In the case of conservative distributed simulation, it is expected that the `dvcompute_mpi`
//! and `dvcompute_core_cons` dynamic (shared) libraries can be found by the operating system, when
//! launching the executable file of the simulation model. You can build the `dvcompute_mpi`
//! library yourself from <https://github.com/dsorokin/dvcompute/tree/main/src/dvcompute_mpi_cdylib> sources
//! that require CMake, C++ compiler and some MPI implementation that you are going to use.
//! Because of specific MPI implementation, this library cannot be unified. But
//! the `dvcompute_core_cons` dynamic library must satisfy the predefined binary
//! interface (it must implement the event queue). You can download the trial version that
//! implements this interface from the author's website <https://aivikasoft.com>.
//! This prebuilt version is a part of "Redistributable Code" portions of DVCompute Simulator.
//!
//! You can find examples in the author's repository: <https://github.com/dsorokin/dvcompute>.
//!
//! The simulation method is described in the author's article:
//! Sorokin David. DVCompute Simulator for discrete event simulation.
//! Prikladnaya informatika=Journal of Applied Informatics, 2021, vol.16, no.3, pp.93-108 (in Russian).
//! DOI: 10.37791/2687-0649-2021-16-3-93-108
//!
//! Also you can use the author's tool DVCompute Modeler for fast prototyping of your future models.
//! The tool is available on the <https://aivikasoft.com> web site. It allows creating
//! discrete event models and queue network models with help of the Python programming language.
//! The tool is available for Windows, Linux and macOS from the author's web site. Then such
//! models can be directly translated into Rust with help of DVCompute Simulator.

#[cfg(feature="cons_mode")]
#[macro_use]
extern crate serde_derive;

#[cfg(feature="cons_mode")]
extern crate serde;

#[cfg(feature="cons_mode")]
extern crate bincode;

#[cfg(feature="cons_mode")]
extern crate log;

extern crate dvcompute_rand;
extern crate dvcompute_utils;

#[cfg(feature="cons_mode")]
extern crate dvcompute_network;

#[cfg(feature="cons_mode")]
use std::ffi::CString;

#[cfg(feature="cons_mode")]
use std::ptr;

#[cfg(feature="cons_mode")]
use libc::*;

/// The main simulation module.
pub mod simulation;

/// The prelude module.
pub mod prelude;

#[cfg(feature="cons_mode")]
#[cfg_attr(windows, link(name = "dvcompute_core_cons.dll"))]
#[cfg_attr(not(windows), link(name = "dvcompute_core_cons"))]
extern {

    /// Setup the simulator logger.
    #[doc(hidden)]
    fn extern_setup_simulator_logger(filter: log::LevelFilter,
        track_id: c_int,
        log_to_console: c_int,
        log_filename: *const c_char);
}

/// Setup the simulator logger.
#[cfg(feature="cons_mode")]
pub fn setup_simulator_logger(filter: log::LevelFilter,
    track_id: c_int,
    log_to_console: bool,
    log_filename: Option<String>)
{
    unsafe {
        let log_to_console = if log_to_console { 1 } else { 0 };
        match log_filename {
            None => {
                extern_setup_simulator_logger(filter, track_id, log_to_console, ptr::null_mut())
            },
            Some(log_filename) => {
                let log_filename = CString::new(log_filename).expect("NulError");
                let log_filename = CString::into_raw(log_filename);

                extern_setup_simulator_logger(filter, track_id, log_to_console, log_filename)
            }
        }
    }
}

#[cfg(test)]
mod tests {
    #[test]
    fn it_works() {
    }
}
