// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::marker::PhantomData;

use crate::simulation;
use crate::simulation::Run;
use crate::simulation::Point;
use crate::simulation::error::*;
use crate::simulation::simulation::*;
use crate::simulation::event::Event;
use crate::simulation::process::*;
use crate::simulation::observable::disposable::*;
use crate::simulation::composite::*;

/// The random parameters.
pub mod random;

/// Additional operations.
pub mod ops;

/// Return a new `Parameter` computation by the specified pure value.
#[inline]
pub fn return_parameter<T>(val: T) -> Return<T> {
    Return { val: val }
}

/// Return a `Parameter` computation that panics with the specified error message.
#[inline]
pub fn panic_parameter<T>(msg: String) -> Panic<T> {
    Panic { msg: msg, _phantom: PhantomData }
}

/// Delay the `Parameter` computation.
#[inline]
pub fn delay_parameter<F, M>(f: F) -> Delay<F, M>
    where F: FnOnce() -> M,
          M: Parameter
{
    Delay { f: f, _phantom: PhantomData }
}

/// Construct a new `Parameter` computation by the specified function.
#[inline]
pub fn cons_parameter<F, T>(f: F) -> Cons<F, T>
    where F: FnOnce(&Run) -> simulation::Result<T>
{
    Cons { f: f, _phantom: PhantomData }
}

/// Create a sequence of computations.
#[inline]
pub fn parameter_sequence<I, M>(comps: I) -> Sequence<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Parameter
{
    Sequence { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Create a sequence of computations, where the result is ignored.
#[inline]
pub fn parameter_sequence_<I, M>(comps: I) -> Sequence_<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Parameter
{
    Sequence_ { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Return the simulation run index.
#[inline]
pub fn simulation_index() -> SimulationIndex {
    SimulationIndex {}
}

/// The computation to define external parameters.
pub trait Parameter {

    /// The type of the item that is returned by the parameter.
    type Item;

    /// Call the `Parameter` computation.
    #[doc(hidden)]
    fn call_parameter(self, r: &Run) -> simulation::Result<Self::Item>;

    /// Convert into the `Simulation` computation.
    #[inline]
    fn into_simulation(self) -> ParameterIntoSimulation<Self>
        where Self: Sized
    {
        ParameterIntoSimulation { comp: self }
    }

    /// Convert into the `Event` computation.
    #[inline]
    fn into_event(self) -> ParameterIntoEvent<Self>
        where Self: Sized
    {
        ParameterIntoEvent { comp: self }
    }

    /// Convert to the `Process` computation.
    #[inline]
    fn into_process(self) -> ParameterIntoProcess<Self>
        where Self: Sized
    {
        ParameterIntoProcess { comp: self }
    }

    /// Convert into the `Composite` computation.
    #[inline]
    fn into_composite(self) -> ParameterIntoComposite<Self>
        where Self: Sized
    {
        ParameterIntoComposite { comp: self }
    }

    /// Bind the current computation with its continuation within the resulting computation.
    #[inline]
    fn flat_map<U, F>(self, f: F) -> FlatMap<Self, U, F>
        where Self: Sized,
              U: Parameter,
              F: FnOnce(Self::Item) -> U,
    {
        FlatMap { comp: self, f: f, _phantom: PhantomData }
    }

    /// Map the current computation using the specified transform.
    #[inline]
    fn map<B, F>(self, f: F) -> Map<Self, B, F>
        where Self: Sized,
              F: FnOnce(Self::Item) -> B,
    {
        Map { comp: self, f: f, _phantom: PhantomData }
    }

    /// Zip the current computation with another one within the resulting computation.
    #[inline]
    fn zip<U>(self, other: U) -> Zip<Self, U>
        where Self: Sized,
              U: Parameter
    {
        Zip { comp: self, other: other }
    }

    /// The function application.
    #[inline]
    fn ap<U, B>(self, other: U) -> Ap<Self, U, B>
        where Self: Sized,
              Self::Item: FnOnce(U::Item) -> B,
              U: Parameter
    {
        Ap { comp: self, other: other, _phantom: PhantomData }
    }

    /// Convert into a boxed value.
    #[inline]
    fn into_boxed(self) -> ParameterBox<Self::Item>
        where Self: Sized + Clone + 'static
    {
        ParameterBox::new(move |r: &Run| { self.call_parameter(r) })
    }
}
/// Allows converting to `Parameter` computations.
pub trait IntoParameter {

    /// The target computation.
    type Parameter: Parameter<Item = Self::Item>;

    /// The type of item that is returned by the computation.
    type Item;

    /// Convert to the `Parameter` computation.
    fn into_parameter(self) -> Self::Parameter;
}

impl<M: Parameter> IntoParameter for M {

    type Parameter = M;

    type Item = M::Item;

    #[inline]
    fn into_parameter(self) -> Self::Parameter {
        self
    }
}

/// It represents the boxed `Parameter` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct ParameterBox<T> {
    f: Box<dyn ParameterFnBoxClone<T>>
}

impl<T> ParameterBox<T> {

    /// Create a new boxed computation.
    #[doc(hidden)]
    #[inline]
    fn new<F>(f: F) -> Self
        where F: FnOnce(&Run) -> simulation::Result<T> + Clone + 'static
    {
        ParameterBox {
            f: Box::new(f)
        }
    }

    /// Call the boxed function.
    #[doc(hidden)]
    #[inline]
    pub fn call_box(self, arg: (&Run,)) -> simulation::Result<T> {
        let ParameterBox { f } = self;
        f.call_box(arg)
    }
}

impl<T> Clone for ParameterBox<T> {

    fn clone(&self) -> Self {
        ParameterBox {
            f: self.f.call_clone()
        }
    }
}

impl<T> Parameter for ParameterBox<T> {

    type Item = T;

    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<Self::Item> {
        self.call_box((r,))
    }

    #[inline]
    fn into_boxed(self) -> ParameterBox<Self::Item>
        where Self: Sized + Clone + 'static
    {
        self
    }
}

/// A trait to support the stable version of Rust, where there is no `FnBox`.
trait ParameterFnBox<T> {

    /// Call the corresponding function.
    fn call_box(self: Box<Self>, args: (&Run,)) -> simulation::Result<T>;
}

impl<T, F> ParameterFnBox<T> for F
    where F: for<'a> FnOnce(&'a Run) -> simulation::Result<T>
{
    fn call_box(self: Box<Self>, args: (&Run,)) -> simulation::Result<T> {
        let this: Self = *self;
        this(args.0)
    }
}

/// A trait to implement a cloneable `FnBox`.
trait ParameterFnBoxClone<T>: ParameterFnBox<T> {

    /// Clone the function.
    fn call_clone(&self) -> Box<dyn ParameterFnBoxClone<T>>;
}

impl<T, F> ParameterFnBoxClone<T> for F
    where F: for<'a> FnOnce(&'a Run) -> simulation::Result<T> + Clone + 'static
{
    fn call_clone(&self) -> Box<dyn ParameterFnBoxClone<T>> {
        Box::new(self.clone())
    }
}

/// Allows creating the `Parameter` computation from a pure value.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Return<T> {

    /// Return a pure value, which is then transformed to the computation.
    val: T
}

impl<T> Parameter for Return<T> {

    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, _: &Run) -> simulation::Result<T> {
        let Return { val } = self;
        Result::Ok(val)
    }
}

/// Allows delaying the `Parameter` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Delay<F, M> {

    /// Return the computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<F, M> Parameter for Delay<F, M>
    where F: FnOnce() -> M,
          M: Parameter
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<M::Item> {
        let Delay { f, _phantom } = self;
        f().call_parameter(r)
    }
}

/// Allows constructing the `Parameter` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Cons<F, T> {

    /// The function of simulation run.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<T>
}

impl<F, T> Parameter for Cons<F, T>
    where F: FnOnce(&Run) -> simulation::Result<T>
{
    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<T> {
        let Cons { f, _phantom } = self;
        f(r)
    }
}

/// A conversion into the `Simulation` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct ParameterIntoSimulation<M> {

    /// The current computation.
    comp: M
}

impl<M> Simulation for ParameterIntoSimulation<M>
    where M: Parameter
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_simulation(self, r: &Run) -> simulation::Result<M::Item> {
        let ParameterIntoSimulation { comp } = self;
        comp.call_parameter(r)
    }
}

/// A conversion into the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct ParameterIntoEvent<M> {

    /// The current computation.
    comp: M
}

impl<M> Event for ParameterIntoEvent<M>
    where M: Parameter
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<M::Item> {
        let ParameterIntoEvent { comp } = self;
        comp.call_parameter(p.run)
    }
}

/// A conversion into the `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct ParameterIntoComposite<M> {

    /// The current computation.
    comp: M
}

impl<M> Composite for ParameterIntoComposite<M>
    where M: Parameter
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(M::Item, DisposableBox)> {
        let ParameterIntoComposite { comp } = self;
        let a = comp.call_parameter(p.run)?;
        Result::Ok((a, disposable))
    }
}

/// The monadic bind for the `Parameter` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct FlatMap<M, U, F> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<U>
}

impl<M, U, F> Parameter for FlatMap<M, U, F>
    where M: Parameter,
          U: Parameter,
          F: FnOnce(M::Item) -> U
{
    type Item = U::Item;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<U::Item> {
        let FlatMap { comp, f, _phantom } = self;
        match comp.call_parameter(r) {
            Result::Ok(a) => {
                let m = f(a);
                m.call_parameter(r)
            },
            Result::Err(e) => {
                Result::Err(e)
            }
        }
    }
}

/// The functor for the `Parameter` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Map<M, B, F> {

    /// The current computation.
    comp: M,

    /// The transform.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, B, F> Parameter for Map<M, B, F>
    where M: Parameter,
          F: FnOnce(M::Item) -> B
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<B> {
        let Map { comp, f, _phantom } = self;
        match comp.call_parameter(r) {
            Result::Ok(a) => Result::Ok(f(a)),
            Result::Err(e) => Result::Err(e)
        }
    }
}

/// The zip of two `Parameter` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Zip<M, U> {

    /// The current computation.
    comp: M,

    /// Another computation.
    other: U,
}

impl<M, U> Parameter for Zip<M, U>
    where M: Parameter,
          U: Parameter
{
    type Item = (M::Item, U::Item);

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<(M::Item, U::Item)> {
        let Zip { comp, other } = self;
        comp.flat_map(move |a| {
            other.map(move |b| (a, b))
        }).call_parameter(r)
    }
}

/// The function application for the `Parameter` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Ap<M, U, B> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    other: U,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, U, B> Parameter for Ap<M, U, B>
    where M: Parameter,
          U: Parameter,
          M::Item: FnOnce(U::Item) -> B
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<B> {
        let Ap { comp, other, _phantom } = self;
        comp.flat_map(move |f| {
            other.map(move |a| { f(a) })
        }).call_parameter(r)
    }
}

/// Allows converting to the `Process` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct ParameterIntoProcess<M> {

    /// The current computation.
    comp: M
}

impl<M> Process for ParameterIntoProcess<M>
    where M: Parameter
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_process<C>(self, cont: C, pid: Rc<ProcessId>, p: &Point) -> simulation::Result<()>
        where C: FnOnce(simulation::Result<Self::Item>, Rc<ProcessId>, &Point) -> simulation::Result<()> + Clone + 'static
    {
        if is_process_cancelled(&pid, p) {
            revoke_process(cont, pid, p)
        } else {
            let ParameterIntoProcess { comp } = self;
            let t = comp.call_parameter(p.run);
            cont(t, pid, p)
        }
    }

    #[doc(hidden)]
    #[inline]
    fn call_process_boxed(self, cont: ProcessBoxCont<Self::Item>, pid: Rc<ProcessId>, p: &Point) -> simulation::Result<()> {
        if is_process_cancelled(&pid, p) {
            revoke_process_boxed(cont, pid, p)
        } else {
            let ParameterIntoProcess { comp } = self;
            let t = comp.call_parameter(p.run);
            cont.call_box((t, pid, p))
        }
    }
}

/// The sequence of computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Parameter for Sequence<I, M>
    where I: Iterator<Item = M>,
          M: Parameter
{
    type Item = Vec<M::Item>;

    #[doc(hidden)]
    fn call_parameter(self, r: &Run) -> simulation::Result<Self::Item> {
        let Sequence { comps, _phantom } = self;
        let mut v = {
            match comps.size_hint() {
                (_, Some(n)) => Vec::with_capacity(n),
                (_, None) => Vec::new()
            }
        };
        for m in comps {
            match m.call_parameter(r) {
                Result::Ok(a) => {
                    v.push(a)
                },
                Result::Err(e) => {
                    return Result::Err(e)
                }
            }
        }
        Result::Ok(v)
    }
}

/// The sequence of computations with ignored result.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence_<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Parameter for Sequence_<I, M>
    where I: Iterator<Item = M>,
          M: Parameter
{
    type Item = ();

    #[doc(hidden)]
    fn call_parameter(self, r: &Run) -> simulation::Result<Self::Item> {
        let Sequence_ { comps, _phantom } = self;
        for m in comps {
            match m.call_parameter(r) {
                Result::Ok(_) => (),
                Result::Err(e) => return Result::Err(e)
            }
        }
        Result::Ok(())
    }
}

/// It returns the simulation run index.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct SimulationIndex {}

impl Parameter for SimulationIndex {

    type Item = usize;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, r: &Run) -> simulation::Result<Self::Item> {
        Result::Ok(r.run_index)
    }
}

/// Allows creating the `Parameter` computation that panics with the specified message.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Panic<T> {

    /// The panic message.
    msg: String,

    /// Keep the type parameter.
    _phantom: PhantomData<T>
}

impl<T> Parameter for Panic<T> {

    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_parameter(self, _r: &Run) -> simulation::Result<Self::Item> {
        let Panic { msg, _phantom } = self;
        let err = Rc::new(OtherError::Panic(msg));
        let err = Error::Other(err);
        Result::Err(err)
    }
}
