// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::marker::PhantomData;

use crate::simulation;
use crate::simulation::Point;
use crate::simulation::event::*;
use crate::simulation::observable::disposable::*;

/// Return a new `Composite` computation by the specified pure value.
#[inline]
pub fn return_composite<T>(val: T) -> Return<T> {
    Return { val: val }
}

/// Delay the `Composite` computation.
#[inline]
pub fn delay_composite<F, M>(f: F) -> Delay<F, M>
    where F: FnOnce() -> M,
          M: Composite
{
    Delay { f: f, _phantom: PhantomData }
}

/// Construct a new `Composite` computation by the specified function.
#[inline]
pub fn cons_composite<F, T>(f: F) -> Cons<F, T>
    where F: FnOnce(DisposableBox, &Point) -> simulation::Result<(T, DisposableBox)>
{
     Cons { f: f, _phantom: PhantomData }
}

/// Allows embedding arbitrary `Disposable` actions in `Composite` computation.
#[inline]
pub fn disposable_composite<D>(action: D) -> impl Composite<Item = ()>
    where D: Disposable + Clone + 'static
{
    DisposableComposite { action: action }
}

/// Create a sequence of computations.
#[inline]
pub fn composite_sequence<I, M>(comps: I) -> Sequence<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Composite
{
    Sequence { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Create a sequence of computations, where the result is ignored.
#[inline]
pub fn composite_sequence_<I, M>(comps: I) -> Sequence_<I::IntoIter, M>
    where I: IntoIterator<Item = M>,
          M: Composite
{
    Sequence_ { comps: comps.into_iter(), _phantom: PhantomData }
}

/// Trace the computation.
#[inline]
pub fn trace_composite<M>(msg: String, comp: M) -> Trace<M>
    where M: Composite
{
    Trace { comp: comp, msg: msg}
}

/// The computation synchronized with the composite queue.
pub trait Composite {

    /// The type of the item that is returned in the current modeling time.
    type Item;

    /// Call the `Composite` computation in the current modeling time by updating the `Disposable` object.
    #[doc(hidden)]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(Self::Item, DisposableBox)>;

    /// Bind the current computation with its continuation within the resulting computation.
    #[inline]
    fn flat_map<U, F>(self, f: F) -> FlatMap<Self, U, F>
        where Self: Sized,
              U: Composite,
              F: FnOnce(Self::Item) -> U,
    {
        FlatMap { comp: self, f: f, _phantom: PhantomData }
    }

    /// Map the current computation using the specified transform.
    #[inline]
    fn map<B, F>(self, f: F) -> Map<Self, B, F>
        where Self: Sized,
              F: FnOnce(Self::Item) -> B,
    {
        Map { comp: self, f: f, _phantom: PhantomData }
    }

    /// Zip the current computation with another one within the resulting computation.
    #[inline]
    fn zip<U>(self, other: U) -> Zip<Self, U>
        where Self: Sized,
              U: Composite
    {
        Zip { comp: self, other: other }
    }

    /// The function application.
    #[inline]
    fn ap<U, B>(self, other: U) -> Ap<Self, U, B>
        where Self: Sized,
              Self::Item: FnOnce(U::Item) -> B,
              U: Composite
    {
        Ap { comp: self, other: other, _phantom: PhantomData }
    }

    /// Convert into a boxed value.
    #[inline]
    fn into_boxed(self) -> CompositeBox<Self::Item>
        where Self: Sized + Clone + 'static
    {
        CompositeBox::new(move |disposable, p: &Point| { self.call_composite(disposable, p) })
    }

    /// Run the computation by the specified initial `Disposable` object.
    #[inline]
    fn run<D>(self, disposable: D) -> Run<Self, D>
        where Self: Sized,
              D: Disposable + Clone + 'static
    {
        Run { comp: self, disposable: disposable }
    }

    /// Run the computation by ignoring `Disposable` objects.
    #[inline]
    fn run_(self) -> Run_<Self>
        where Self: Sized
    {
        Run_ { comp: self }
    }
}

/// Allows converting to `Composite` computations.
pub trait IntoComposite {

    /// The target computation.
    type Composite: Composite<Item = Self::Item>;

    /// The type of item that is returned by the computation.
    type Item;

    /// Convert to the `Composite` computation.
    fn into_composite(self) -> Self::Composite;
}

impl<M: Composite> IntoComposite for M {

    type Composite = M;

    type Item = M::Item;

    #[inline]
    fn into_composite(self) -> Self::Composite {
        self
    }
}

/// It represents the boxed `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
pub struct CompositeBox<T> {
    f: Box<dyn CompositeFnBoxClone<T>>
}

impl<T> CompositeBox<T> {

    /// Create a new boxed computation.
    #[doc(hidden)]
    #[inline]
    fn new<F>(f: F) -> Self
        where F: FnOnce(DisposableBox, &Point) -> simulation::Result<(T, DisposableBox)> + Clone + 'static
    {
        CompositeBox {
            f: Box::new(f)
        }
    }

    /// Call the boxed function.
    #[doc(hidden)]
    #[inline]
    pub fn call_box(self, arg: (DisposableBox, &Point,)) -> simulation::Result<(T, DisposableBox)> {
        let CompositeBox { f } = self;
        f.call_box(arg)
    }
}

impl<T> Clone for CompositeBox<T> {

    fn clone(&self) -> Self {
        CompositeBox {
            f: self.f.call_clone()
        }
    }
}

impl<T> Composite for CompositeBox<T> {

    type Item = T;

    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(Self::Item, DisposableBox)> {
        self.call_box((disposable, p,))
    }

    #[inline]
    fn into_boxed(self) -> CompositeBox<Self::Item>
        where Self: Sized + 'static
    {
        self
    }
}

/// A trait to support the stable version of Rust, where there is no `FnBox`.
trait CompositeFnBox<T> {

    /// Call the corresponding function.
    fn call_box(self: Box<Self>, args: (DisposableBox, &Point,)) -> simulation::Result<(T, DisposableBox)>;
}

impl<T, F> CompositeFnBox<T> for F
    where F: for<'a> FnOnce(DisposableBox, &'a Point) -> simulation::Result<(T, DisposableBox)>
{
    fn call_box(self: Box<Self>, args: (DisposableBox, &Point,)) -> simulation::Result<(T, DisposableBox)> {
        let this: Self = *self;
        this(args.0, args.1)
    }
}

/// A trait to implement a cloneable `FnBox`.
trait CompositeFnBoxClone<T>: CompositeFnBox<T> {

    /// Clone the function.
    fn call_clone(&self) -> Box<dyn CompositeFnBoxClone<T>>;
}

impl<T, F> CompositeFnBoxClone<T> for F
    where F: for<'a> FnOnce(DisposableBox, &'a Point) -> simulation::Result<(T, DisposableBox)> + Clone + 'static
{
    fn call_clone(&self) -> Box<dyn CompositeFnBoxClone<T>> {
        Box::new(self.clone())
    }
}

/// Allows creating the `Composite` computation from a pure value.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Return<T> {

    /// Return a pure value, which is then transformed to the computation.
    val: T
}

impl<T> Composite for Return<T> {

    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, _: &Point) -> simulation::Result<(T, DisposableBox)> {
        let Return { val } = self;
        Result::Ok((val, disposable))
    }
}

/// Allows delaying the `Composite` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Delay<F, M> {

    /// Return the computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<F, M> Composite for Delay<F, M>
    where F: FnOnce() -> M,
          M: Composite
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(M::Item, DisposableBox)> {
        let Delay { f, _phantom } = self;
        f().call_composite(disposable, p)
    }
}

/// Allows constructing the `Composite` computation by the specified function.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Cons<F, T> {

    /// The function of time point.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<T>
}

impl<F, T> Composite for Cons<F, T>
    where F: FnOnce(DisposableBox, &Point) -> simulation::Result<(T, DisposableBox)>
{
    type Item = T;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(T, DisposableBox)> {
        let Cons { f, _phantom } = self;
        f(disposable, p)
    }
}

/// The monadic bind for the `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct FlatMap<M, U, F> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<U>
}

impl<M, U, F> Composite for FlatMap<M, U, F>
    where M: Composite,
          U: Composite,
          F: FnOnce(M::Item) -> U,
{
    type Item = U::Item;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(U::Item, DisposableBox)> {
        let FlatMap { comp, f, _phantom } = self;
        match comp.call_composite(disposable, p) {
            Result::Ok((a, disposable)) => {
                let m = f(a);
                m.call_composite(disposable, p)
            },
            Result::Err(e) => {
                Result::Err(e)
            }
        }
    }
}

/// The functor for the `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Map<M, B, F> {

    /// The current computation.
    comp: M,

    /// The transform.
    f: F,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, B, F> Composite for Map<M, B, F>
    where M: Composite,
          F: FnOnce(M::Item) -> B,
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(B, DisposableBox)> {
        let Map { comp, f, _phantom } = self;
        match comp.call_composite(disposable, p) {
            Result::Ok((a, disposable)) => Result::Ok((f(a), disposable)),
            Result::Err(e) => Result::Err(e)
        }
    }
}

/// The zip of two `Composite` computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Zip<M, U> {

    /// The current computation.
    comp: M,

    /// Another computation.
    other: U,
}

impl<M, U> Composite for Zip<M, U>
    where M: Composite,
          U: Composite
{
    type Item = (M::Item, U::Item);

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<((M::Item, U::Item), DisposableBox)> {
        let Zip { comp, other } = self;
        comp.flat_map(move |a| {
            other.map(move |b| (a, b))
        }).call_composite(disposable, p)
    }
}

/// The function application for the `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Ap<M, U, B> {

    /// The current computation.
    comp: M,

    /// The continuation of the current computation.
    other: U,

    /// To keep the type parameter.
    _phantom: PhantomData<B>
}

impl<M, U, B> Composite for Ap<M, U, B>
    where M: Composite,
          U: Composite,
          M::Item: FnOnce(U::Item) -> B,
{
    type Item = B;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(B, DisposableBox)> {
        let Ap { comp, other, _phantom } = self;
        comp.flat_map(move |f| {
            other.map(move |a| { f(a) })
        }).call_composite(disposable, p)
    }
}

/// The sequence of computations.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Composite for Sequence<I, M>
    where I: Iterator<Item = M>,
          M: Composite
{
    type Item = Vec<M::Item>;

    #[doc(hidden)]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(Self::Item, DisposableBox)> {
        let Sequence { comps, _phantom } = self;
        let mut v = {
            match comps.size_hint() {
                (_, Some(n)) => Vec::with_capacity(n),
                (_, None) => Vec::new()
            }
        };
        let mut disposable = disposable;
        for m in comps {
            match m.call_composite(disposable, p) {
                Result::Ok((a, x)) => {
                    v.push(a);
                    disposable = x;
                },
                Result::Err(e) => {
                    return Result::Err(e)
                }
            }
        }
        Result::Ok((v, disposable))
    }
}

/// The sequence of computations with ignored result.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Sequence_<I, M> {

    /// The computations.
    comps: I,

    /// To keep the type parameter.
    _phantom: PhantomData<M>
}

impl<I, M> Composite for Sequence_<I, M>
    where I: Iterator<Item = M>,
          M: Composite
{
    type Item = ();

    #[doc(hidden)]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(Self::Item, DisposableBox)> {
        let Sequence_ { comps, _phantom } = self;
        let mut disposable = disposable;
        for m in comps {
            match m.call_composite(disposable, p) {
                Result::Ok((_, x)) => { disposable = x; },
                Result::Err(e) => return Result::Err(e)
            }
        }
        Result::Ok(((), disposable))
    }
}

/// The run function for `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Run<M, D> {

    /// The computation.
    comp: M,

    /// The initial `Disposable` action.
    disposable: D
}

impl<M, D> Event for Run<M, D>
    where M: Composite,
          D: Disposable + Clone + 'static
{
    type Item = (M::Item, DisposableBox);

    #[doc(hidden)]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Run { comp, disposable } = self;
        let disposable = disposable.into_boxed();
        comp.call_composite(disposable, p)
    }
}

/// Another version of the run function for `Composite` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Run_<M> {

    /// The computation.
    comp: M
}

impl<M> Event for Run_<M>
    where M: Composite
{
    type Item = M::Item;

    #[doc(hidden)]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Run_ { comp } = self;
        let disposable = empty_disposable().into_boxed();
        let (a, _) = comp.call_composite(disposable, p)?;
        Result::Ok(a)
    }
}

/// Computation that allows embedding `Disposable` actions.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct DisposableComposite<D> {

    /// The action.
    action: D
}

impl<D> Composite for DisposableComposite<D>
    where D: Disposable + Clone + 'static
{
    type Item = ();

    #[doc(hidden)]
    fn call_composite(self, disposable: DisposableBox, _p: &Point) -> simulation::Result<(Self::Item, DisposableBox)> {
        let DisposableComposite { action } = self;
        let disposable = disposable.merge(action).into_boxed();
        Result::Ok(((), disposable))
    }
}

/// Trace the computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Trace<M> {

    /// The computation.
    comp: M,

    /// The message to print.
    msg: String
}

impl<M> Composite for Trace<M>
    where M: Composite
{
    type Item = M::Item;

    #[doc(hidden)]
    #[inline]
    fn call_composite(self, disposable: DisposableBox, p: &Point) -> simulation::Result<(Self::Item, DisposableBox)> {
        let Trace { comp, msg } = self;
        p.trace(&msg);
        comp.call_composite(disposable, p)
    }
}
