// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use std::rc::Rc;

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
use std::cell::RefCell;

use crate::simulation::specs::*;
use crate::simulation::point::Point;
use crate::simulation::generator::Generator;
use crate::simulation::internal::event_queue::*;

#[cfg(feature="dist_mode")]
use crate::simulation::comm::context::LogicalProcessContext;

#[cfg(feature="dist_mode")]
use crate::simulation::internal::undoable_log::*;

#[cfg(feature="dist_mode")]
use crate::simulation::internal::input_message_queue::*;

#[cfg(feature="dist_mode")]
use crate::simulation::internal::transient_message_queue::*;

#[cfg(feature="dist_mode")]
use crate::simulation::internal::output_message_queue::*;

#[cfg(feature="dist_mode")]
use crate::simulation::internal::acknowledgement_message_queue::*;

/// It represents the simulation run.
#[cfg(feature="dist_mode")]
#[repr(C)]
pub struct Run<'a> {

    /// The event queue.
    #[doc(hidden)]
    pub event_queue: *mut EventQueue,

    /// The simulation specs.
    pub specs: SpecsRepr,

    /// The current simulation run index starting from zero.
    pub run_index: usize,

    /// The number of simulation runs in the Monte Carlo experiment.
    pub run_count: usize,

    /// The random number generator.
    pub generator: Generator,

    /// The logical process context.
    #[doc(hidden)]
    pub lp_context: &'a LogicalProcessContext,

    /// The log of undoable actions.
    #[doc(hidden)]
    pub undoable_log: *mut UndoableLog,

    /// The input message queue.
    #[doc(hidden)]
    pub input_messages: *mut InputMessageQueue,

    /// The transient message queue.
    #[doc(hidden)]
    pub transient_messages: *mut TransientMessageQueue,

    /// The output message queue.
    #[doc(hidden)]
    pub output_messages: *mut OutputMessageQueue,

    /// The acknowledgement message queue.
    #[doc(hidden)]
    pub acknowledgement_messages: *mut AcknowledgementMessageQueue
}

#[cfg(feature="dist_mode")]
impl<'a> Drop for Run<'a> {

    fn drop(&mut self) {
        unsafe {
            delete_extern_event_queue(self.event_queue);
            delete_extern_undoable_log(self.undoable_log);
            delete_extern_input_message_queue(self.input_messages);
            delete_extern_transient_message_queue(self.transient_messages);
            delete_extern_output_message_queue(self.output_messages);
            delete_extern_acknowledgement_message_queue(self.acknowledgement_messages);
        }
    }
}

#[cfg(feature="dist_mode")]
impl<'a> Run<'a> {

    /// Create a new run instance.
    pub fn new(specs: Specs, ctx: &'a LogicalProcessContext) -> Run<'a> {
        Self::by_index(specs, ctx, 0, 1)
    }

    /// Create a new run instance by the specified index.
    pub fn by_index(specs: Specs, ctx: &'a LogicalProcessContext, run_index: usize, run_count: usize) -> Run<'a> {
        let generator = Generator::new(&specs.generator_type);
        let specs = specs.into();
        let event_queue = unsafe {
            create_extern_event_queue(&specs)
        };
        let undoable_log = unsafe {
            create_extern_undoable_log()
        };
        let input_messages = unsafe {
            create_extern_input_message_queue()
        };
        let transient_messages = unsafe {
            create_extern_transient_message_queue()
        };
        let output_messages = unsafe {
            create_extern_output_message_queue()
        };
        let acknowledgement_messages = unsafe {
            create_extern_acknowledgement_message_queue()
        };
        Run {
            specs: specs,
            event_queue: event_queue,
            run_index: run_index,
            run_count: run_count,
            generator: generator,
            lp_context: ctx,
            undoable_log: undoable_log,
            input_messages: input_messages,
            transient_messages: transient_messages,
            output_messages: output_messages,
            acknowledgement_messages: acknowledgement_messages
        }
    }

    /// Return the point in the start simulation time.
    #[inline]
    pub fn point_in_start_time(&self) -> Point {
        let specs = &self.specs;
        let t0 = specs.start_time;
        Point {
            run: self,
            time: t0,
            priority: 0,
            iteration: 0,
            phase: 0
        }
    }

    /// Return the point in the final simulation time.
    #[inline]
    pub fn point_in_stop_time(&self) -> Point {
        let specs = &self.specs;
        let t0 = specs.start_time;
        let t2 = specs.stop_time;
        let dt = specs.dt;
        let n2 = ((t2 - t0) / dt).floor() as usize;
        Point {
            run: self,
            time: t2,
            priority: 0,
            iteration: n2,
            phase: 0
        }
    }

    /// Return the point at the specified modeling time.
    #[inline]
    pub fn point_at(&self, t: f64, priority: isize) -> Point {
        let specs = &self.specs;
        let t0 = specs.start_time;
        let dt = specs.dt;
        let n2 = ((t - t0) / dt).floor() as usize;
        Point {
            run: self,
            time: t,
            priority: priority,
            iteration: n2,
            phase: -1
        }
    }
}

/// It represents the simulation run.
#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
#[repr(C)]
pub struct Run {

    /// The event queue.
    #[doc(hidden)]
    pub event_queue: Box<EventQueue>,

    /// The simulation specs.
    pub specs: SpecsRepr,

    /// The current simulation run index starting from zero.
    pub run_index: usize,

    /// The number of simulation runs in the Monte Carlo experiment.
    pub run_count: usize,

    /// The random number generator.
    pub generator: Generator,

    /// The information about current branch or `None` for root computation.
    pub branch: Rc<Branch>,

    /// The branch identifier generator.
    pub branch_next_id: Rc<RefCell<u64>>,

    /// The branch finalizers.
    pub finalizers: RefCell<Vec<Box<dyn Fn()>>>
}

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
impl Drop for Run {

    fn drop(&mut self) {
        let fins = self.finalizers.borrow();
        for fin in (*fins).iter() {
            fin()
        }
    }
}

#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
impl Run {

    /// Create a new run instance.
    pub fn new(specs: Specs) -> Run {
        Self::by_index(specs, 0, 1)
    }

    /// Create a new run instance by the specified index.
    pub fn by_index(specs: Specs, run_index: usize, run_count: usize) -> Run {
        let generator = Generator::new(&specs.generator_type);
        let specs = specs.into();
        let event_queue = Box::new(EventQueue::new(&specs));
        let branch = Branch {
            id: 1,
            level: 0,
            parent: None
        };
        Run {
            specs: specs,
            event_queue: event_queue,
            run_index: run_index,
            run_count: run_count,
            generator: generator,
            branch: Rc::new(branch),
            branch_next_id: Rc::new(RefCell::new(1)),
            finalizers: RefCell::new(Vec::new())
        }
    }

    /// Create a new child run instance.
    pub fn new_child(&self, p: &Point) -> Run {
        let generator = self.generator.clone();
        let specs = self.specs.clone();
        let event_queue = Box::new(self.event_queue.clone_at(p));
        let branch = {
            let id = {
                let mut cell = self.branch_next_id.borrow_mut();
                *cell = 1 + (*cell);
                *cell
            };
            Branch {
                id: id,
                level: 1 + self.branch.level,
                parent: Some(self.branch.clone())
            }
        };
        Run {
            specs: specs,
            event_queue: event_queue,
            run_index: self.run_index,
            run_count: self.run_count,
            generator: generator,
            branch: Rc::new(branch),
            branch_next_id: self.branch_next_id.clone(),
            finalizers: RefCell::new(Vec::new())
        }
    }

    /// Return the point in the start simulation time.
    #[inline]
    pub fn point_in_start_time(&self) -> Point {
        let specs = &self.specs;
        let t0 = specs.start_time;
        Point {
            run: self,
            time: t0,
            priority: 0,
            iteration: 0,
            phase: 0
        }
    }

    /// Return the point in the final simulation time.
    #[inline]
    pub fn point_in_stop_time(&self) -> Point {
        let specs = &self.specs;
        let t0 = specs.start_time;
        let t2 = specs.stop_time;
        let dt = specs.dt;
        let n2 = ((t2 - t0) / dt).floor() as usize;
        Point {
            run: self,
            time: t2,
            priority: 0,
            iteration: n2,
            phase: 0
        }
    }

    /// Return the point at the specified modeling time.
    #[inline]
    pub fn point_at(&self, t: f64, priority: isize) -> Point {
        let specs = &self.specs;
        let t0 = specs.start_time;
        let dt = specs.dt;
        let n2 = ((t - t0) / dt).floor() as usize;
        Point {
            run: self,
            time: t,
            priority: priority,
            iteration: n2,
            phase: -1
        }
    }
}

/// The information about branch.
#[cfg(any(feature="branch_mode", feature="branch_wasm_mode"))]
pub struct Branch {

    /// The unique branch identifier.
    pub id: u64,

    /// The level of nested computation starting from zero.
    pub level: usize,

    /// The parent branch or `None` for root computation.
    pub parent: Option<Rc<Branch>>
}
