// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::rc::Rc;
use std::cell::UnsafeCell;
use std::ptr;

use crate::simulation;
use crate::simulation::Run;
use crate::simulation::point::Point;
use crate::simulation::event::Event;
use crate::simulation::internal::undoable_log::*;

/// A mutable cell computation synchronized with the event queue.
pub struct RefComp<T> {

    /// The underlying cell.
    cell: Rc<UnsafeCell<T>>
}

impl<T> RefComp<T> {

    /// Create a new mutable cell by the initial value.
    #[inline]
    pub fn new(val: T) -> RefComp<T> {
        RefComp { cell: Rc::new(UnsafeCell::new(val)) }
    }

    /// Read the contents of the mutable cell at the specified time point.
    #[inline]
    pub fn read_at(&self, _p: &Point) -> T
        where T: Clone
    {
        let p = self.cell.get();
        unsafe {
            (*p).clone()
        }
    }

    /// Read the contents of the mutable cell within the `Event` computation.
    #[inline]
    pub fn read(comp: Rc<Self>) -> Read<T>
        where T: Clone
    {
        Read { comp: comp }
    }

    /// Swap the contents of the mutable cell at the specified time point.
    #[inline]
    pub fn swap_at(&self, val: T, p: &Point) -> T
        where T: Clone + 'static
    {
        write_val_in_log(&self.cell, p);
        let mut t = val;
        unsafe {
            ptr::swap(&mut t, self.cell.get());
        }
        t
    }

    /// Swap the contents of the mutable cell the `Event` computation.
    #[inline]
    pub fn swap(comp: Rc<Self>, val: T) -> Swap<T>
        where T: Clone + 'static
    {
        Swap { comp: comp, val: val }
    }

    /// Write into the contents of the mutable cell at the specified time point.
    #[inline]
    pub fn write_at(&self, val: T, p: &Point)
        where T: Clone + 'static
    {
        write_val_in_log(&self.cell, p);
        let p = self.cell.get();
        unsafe {
            *p = val;
        }
    }

    /// Write into the contents of the mutable cell the `Event` computation.
    #[inline]
    pub fn write(comp: Rc<Self>, val: T) -> Write<T>
        where T: Clone + 'static
    {
        Write { comp: comp, val: val }
    }

    /// Mutate the contents of the mutable cell at the specified time point.
    #[inline]
    pub fn modify_at<F>(&self, f: F, p: &Point)
        where F: FnOnce(&T) -> T,
              T: Clone + 'static
    {
        write_val_in_log(&self.cell, p);
        let p = self.cell.get();
        unsafe {
            let t = (*p).clone();
            *p = f(&t);
        }
    }

    /// Mutate the contents of the mutable cell the `Event` computation.
    #[inline]
    pub fn modify<F>(comp: Rc<Self>, f: F) -> Modify<T, F>
        where F: FnOnce(&T) -> T,
              T: Clone + 'static
    {
        Modify { comp: comp, f: f }
    }
}

impl<T> PartialEq for RefComp<T> {

    fn eq(&self, other: &Self) -> bool {
        self.cell.get() == other.cell.get()
    }
}

impl<T> Eq for RefComp<T> {}

impl<T: Clone> Clone for RefComp<T> {

    fn clone(&self) -> Self {
        let p = self.cell.get();
        unsafe {
            let val = (*p).clone();
            RefComp { cell: Rc::new(UnsafeCell::new(val)) }
        }
    }
}

/// Write the current cell value in the undoable log.
fn write_val_in_log<T: Clone + 'static>(cell: &Rc<UnsafeCell<T>>, p: &Point) {
    let cell0   = cell.clone();
    let p0      = cell.get();
    let t0      = unsafe { (*p0).clone() };
    let action  = UndoableActionBox::new(move |_r: &Run| {
        let p = cell0.get();
        unsafe {
            *p = t0;
        }
    });
    let action  = UndoableActionRepr::into_repr(action);
    unsafe {
        extern_write_log(p.run.undoable_log, p.time, action);
    }
}

/// It reads the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Read<T> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>
}

impl<T: Clone> Event for Read<T> {

    type Item = T;

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Read { comp } = self;
        Result::Ok(comp.read_at(p))
    }
}

/// It swaps the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Swap<T> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>,

    /// A new value.
    val: T
}

impl<T> Event for Swap<T>
    where T: Clone + 'static
{
    type Item = T;

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Swap { comp, val } = self;
        Result::Ok(comp.swap_at(val, p))
    }
}

/// It writes into the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Write<T> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>,

    /// A new value.
    val: T
}

impl<T> Event for Write<T>
    where T: Clone + 'static
{
    type Item = ();

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Write { comp, val } = self;
        Result::Ok(comp.write_at(val, p))
    }
}

/// It mutates the contents of the mutable cell within the `Event` computation.
#[must_use = "computations are lazy and do nothing unless to be run"]
#[derive(Clone)]
pub struct Modify<T, F> {

    /// The corresponding reference.
    comp: Rc<RefComp<T>>,

    /// A mutator.
    f: F
}

impl<T, F> Event for Modify<T, F>
    where F: FnOnce(&T) -> T,
          T: Clone + 'static
{
    type Item = ();

    #[inline]
    fn call_event(self, p: &Point) -> simulation::Result<Self::Item> {
        let Modify { comp, f } = self;
        Result::Ok(comp.modify_at(f, p))
    }
}
